use super::discussion::Discussion;
use super::problem::Problem;
use juniper::graphql_interface;
use neo4rs::Node;
use std::str::FromStr;
use strum_macros::EnumString;

use crate::graph_context::{
    get_optional_gdb_node_string_val, get_required_gdb_node_string_val, GraphContext,
};

pub const TOPIC_MODEL_NAME: &str = "Topic";
pub const TOPIC_MODEL_ALIAS: &str = "topic";

#[derive(Copy, Clone, Eq, PartialEq, Debug, EnumString)]
pub enum TopicType {
    #[strum(serialize = "Problem")]
    Problem,
    #[strum(serialize = "Discussion")]
    Discussion,
}

#[graphql_interface(for = [Problem, Discussion], context = GraphContext)]
pub trait Topic {
    fn id(&self) -> &str;

    fn model_type(&self) -> &str;

    fn model_name(&self) -> &str;

    fn model_alias(&self) -> &str;

    fn heading(&self) -> &str;
    fn blurb(&self) -> Option<String>;
    fn description(&self) -> Option<String>;
}
pub fn topic_gdb_node_to_gql_interface(topic_node: Node) -> Option<TopicValue> {
    let id: String = topic_node.get("id").unwrap();
    let id_str = &id[..];
    let heading = get_required_gdb_node_string_val(&topic_node, "heading");
    let blurb = get_optional_gdb_node_string_val(&topic_node, "blurb");
    let description = get_optional_gdb_node_string_val(&topic_node, "description");
    let model_name: String = get_required_gdb_node_string_val(&topic_node, "model_name");
    let model_name_as_enum = TopicType::from_str(&model_name).unwrap();

    match model_name_as_enum {
        TopicType::Problem => {
            let problem = Problem::new(id_str, heading, blurb, description);
            return Some(problem.clone().into());
        }
        TopicType::Discussion => {
            let discussion = Discussion::new(id_str, heading, blurb, description);
            return Some(discussion.clone().into());
        }
    }
}
