use juniper::Context;
use klub_stores::dbs::controller::ModelParams;
use klub_stores::dbs::graph::neo4j::NodeWithRelsResult;
use klub_stores::stores::Stores;
use neo4rs::{Graph, Node};
use std::sync::Arc;

pub struct GraphContext {
    pub stores: Option<Stores>,
    pub pool: Option<Graph>,
}

impl Context for GraphContext {}

impl GraphContext {
    pub fn new() -> GraphContext {
        GraphContext {
            stores: None,
            pool: None,
        }
    }

    pub async fn find_node_by_id(&self, model_params: ModelParams, id: &str) -> Option<Node> {
        let store = &self.stores.clone().unwrap();
        match &self.pool {
            Some(p) => {
                store
                    .graph
                    .find_node_by_id(Arc::new(&p), model_params, id)
                    .await
            }
            None => {
                println!("Cannot execute query since there is not active database connection");
                None
            }
        }
    }

    pub async fn find_nodes_with_rels(
        &self,
        id: &str,
        node_one_model_params: ModelParams,
        node_two_model_params: ModelParams,
        rel_model_params: ModelParams,
    ) -> Option<Vec<NodeWithRelsResult>> {
        let store = &self.stores.clone().unwrap();
        match &self.pool {
            Some(p) => {
                store
                    .graph
                    .find_nodes_with_rels(
                        Arc::new(&p),
                        id,
                        node_one_model_params,
                        node_two_model_params,
                        rel_model_params,
                    )
                    .await
            }
            None => {
                println!("Cannot execute query since there is not active database connection");
                None
            }
        }
    }
}

pub fn get_optional_gdb_node_string_val(node: &Node, field: &str) -> Option<String> {
    let node_val = node.get(field);
    match node_val {
        Some(n) => Some(n),
        None => None,
    }
}

pub fn get_optional_gdb_node_f64_val(node: &Node, field: &str) -> Option<f64> {
    let node_val = node.get(field);
    match node_val {
        Some(n) => Some(n),
        None => None,
    }
}

pub fn get_required_gdb_node_string_val(node: &Node, field: &str) -> String {
    let node_val = node.get(field);
    match node_val {
        Some(n) => n,
        None => {
            panic!("missing required field: {:?} on node", field);
        }
    }
}
