use super::native_mobile_app::NativeMobileApp;
use super::web_desktop_app::WebDesktopApp;
use super::web_mobile_app::WebMobileApp;
use juniper::graphql_interface;
use neo4rs::Node;
use std::str::FromStr;
use strum_macros::EnumString;

use crate::graph_context::{
    get_optional_gdb_node_f64_val, get_optional_gdb_node_string_val, GraphContext,
};

pub const DEVICE_MODEL_NAME: &str = "Device";
pub const DEVICE_MODEL_ALIAS: &str = "device";

#[derive(GraphQLEnum, Copy, Clone, Eq, PartialEq, Debug, EnumString)]
pub enum DeviceType {
    #[strum(serialize = "NativeMobileApp")]
    NativeMobileApp,
    #[strum(serialize = "WebMobileApp")]
    WebMobileApp,
    #[strum(serialize = "WebDesktopApp")]
    WebDesktopApp,
}

#[graphql_interface(for = [NativeMobileApp, WebMobileApp, WebDesktopApp], context = GraphContext)]
pub trait Device {
    fn id(&self) -> &str;
    fn os(&self) -> Option<String>;
    fn locale(&self) -> Option<String>;
    fn language(&self) -> Option<String>;
    fn battery_life(&self) -> Option<f64>;
    fn browser(&self) -> Option<String>;
    fn longitude(&self) -> Option<f64>;
    fn latitude(&self) -> Option<f64>;
    fn model_type(&self) -> &str;
}

pub fn device_gdb_node_to_gql_interface(device_node: Node) -> DeviceValue {
    let id: String = device_node.get("id").unwrap();
    let id_str = &id[..];
    let os = get_optional_gdb_node_string_val(&device_node, "os");
    let carrier = get_optional_gdb_node_string_val(&device_node, "carrier");
    let locale = get_optional_gdb_node_string_val(&device_node, "locale");
    let language = get_optional_gdb_node_string_val(&device_node, "language");
    let browser = get_optional_gdb_node_string_val(&device_node, "browser");
    let battery_life = get_optional_gdb_node_f64_val(&device_node, "battery_life");
    let latitude = get_optional_gdb_node_f64_val(&device_node, "latitude");
    let longitude = get_optional_gdb_node_f64_val(&device_node, "longitude");
    let city = get_optional_gdb_node_string_val(&device_node, "city");
    let address = get_optional_gdb_node_string_val(&device_node, "address");
    let zipcode = get_optional_gdb_node_string_val(&device_node, "zipcode");
    let model_name: String = device_node.get("model_name").unwrap();
    let model_name_as_enum = DeviceType::from_str(&model_name).unwrap();

    match model_name_as_enum {
        DeviceType::NativeMobileApp => {
            let native_desktop_app = NativeMobileApp::new(
                id_str,
                os,
                carrier,
                locale,
                language,
                battery_life,
                browser,
                longitude,
                latitude,
                city,
                address,
                zipcode,
            );
            return native_desktop_app.clone().into();
        }
        DeviceType::WebMobileApp => {
            let web_mobile_app = WebMobileApp::new(
                id_str,
                os,
                carrier,
                locale,
                language,
                battery_life,
                browser,
                longitude,
                latitude,
                city,
                address,
                zipcode,
            );
            return web_mobile_app.clone().into();
        }
        DeviceType::WebDesktopApp => {
            let web_desktop_app = WebDesktopApp::new(
                id_str,
                os,
                carrier,
                locale,
                language,
                battery_life,
                browser,
                longitude,
                latitude,
                city,
                address,
                zipcode,
            );
            return web_desktop_app.clone().into();
        }
    }
}
