use super::opinion::Opinion;
use super::solution::Solution;
use juniper::graphql_interface;
use neo4rs::Node;
use std::str::FromStr;
use strum_macros::EnumString;

use crate::graph_context::{get_required_gdb_node_string_val, GraphContext};

pub const COMMENT_MODEL_NAME: &str = "Comment";
pub const COMMENT_MODEL_ALIAS: &str = "comment";

#[derive(GraphQLEnum, Copy, Clone, Eq, PartialEq, Debug, EnumString)]
pub enum CommentType {
    #[strum(serialize = "Solution")]
    Solution,
    #[strum(serialize = "Opinion")]
    Opinion,
}

#[graphql_interface(for = [Solution, Opinion], context = GraphContext)]
pub trait Comment {
    fn id(&self) -> &str;

    fn heading(&self) -> &str;

    fn text(&self) -> &str;

    fn model_type(&self) -> &str;
}

pub fn comment_gdb_node_to_gql_interface(comment_node: Node) -> Option<CommentValue> {
    let id: String = comment_node.get("id").unwrap();
    let id_str = &id[..];
    let heading = get_required_gdb_node_string_val(&comment_node, "heading");
    let text = get_required_gdb_node_string_val(&comment_node, "text");
    let model_name = get_required_gdb_node_string_val(&comment_node, "model_name");
    let model_name_as_enum = CommentType::from_str(&model_name).unwrap();

    match model_name_as_enum {
        CommentType::Solution => {
            let solution = Solution::new(id_str, heading, text);
            return Some(solution.clone().into());
        }
        CommentType::Opinion => {
            let opinion = Opinion::new(id_str, heading, text);
            return Some(opinion.clone().into());
        }
    }
}
