use std::fs;
use std::path::Path;

use crate::KllGroups;

const SAFE_CAPABILITIES: &[&str] = &[
    // PartialMap
    "layerState",
    "layerLatch",
    "layerLock",
    "layerShift",
    "layerRotate",
    "testThreadSafe",
    // USB
    "consCtrlOut",
    "noneOut",
    "sysCtrlOut",
    "usbKeyOut",
    "mouseOut",
    "mouseWheelOut",
    "flashMode",
];

pub struct Capability<'a>(&'a str, usize);
impl<'a> Capability<'a> {
    fn func_decl_str(&self) -> String {
        format!(
            "void {}_capability( TriggerMacro *trigger, uint8_t state, uint8_t stateType, uint8_t *args );",
            self.0,
        )
    }
    fn cap_str(&self, i: usize) -> String {
        let feature = match SAFE_CAPABILITIES.iter().any(|&cap| cap == self.0) {
            true => "CapabilityFeature_Safe",
            false => "CapabilityFeature_None",
        };
        format!(
            "\t/* {:2} {} */\n\t{{ {}_capability, {}, {} }},",
            i, self.0, self.0, self.1, feature,
        )
    }
}

pub struct ResultMacro<'a>(usize, usize, &'a str, usize, usize);
impl<'a> ResultMacro<'a> {
    fn guide_str(&self, i: usize) -> String {
        format!(
            "Guide_RM( {:2} ) = {{ {}, {}, KEY_{}_{}, {} }}; // (HID(USBCode,default)\"{}\"{})",
            i, self.0, self.1, self.2, self.3, self.4, self.3, self.2
        )
    }
    fn define_str(&self, i: usize) -> String {
        format!(
            "\tDefine_RM( {:2} ), // (HID(USBCode,default)\"{}\"{})",
            i, self.3, self.2
        )
    }
}

pub struct TriggerMacro<'a>(usize, &'a str, &'a str, usize, usize);
impl<'a> TriggerMacro<'a> {
    fn guide_str(&self, i: usize) -> String {
        format!(
            "Guide_TM( {:2} ) = {{ {}, TriggerType_{}, ScheduleType_{}, {:#04X}, {} }};",
            i, self.0, self.1, self.2, self.3, self.4
        )
    }
    fn define_str(&self, i: usize) -> String {
        format!("\t/* {:2} */ Define_TM( {:2}, {} ),;", i, 0, 0)
    }
}

pub struct TriggerLayer<'a>(&'a str, &'a [usize]);
impl<'a> TriggerLayer<'a> {
    fn define_str(&self, i: usize) -> String {
        if self.1.is_empty() {
            format!("Define_TL( {:2}, {:#04X} ) = {{ 0 }};", self.0, i)
        } else {
            format!(
                "Define_TL( {:2}, {:#04X} ) = {{ {}, {} }};",
                self.0,
                i,
                self.1.len(),
                self.1
                    .iter()
                    .map(|x| x.to_string())
                    .collect::<Vec<_>>()
                    .join(", ")
            )
        }
    }
}

pub struct LayerIndex<'a>(&'a str, &'a str, usize);
impl<'a> LayerIndex<'a> {
    fn define_str(&self) -> String {
        format!(
            "\tLayer_IN( {}_scanMap, \"{}\", {:#04X} ),",
            self.0, self.1, self.2
        )
    }
}

pub struct PositionEntry(f32, f32, f32, f32, f32, f32);

impl PositionEntry {
    fn define_str(&self) -> String {
        format!(
            "\tPositionEntry( {:7.2}, {:7.2}, {:7.2}, {:7.2}, {:7.2}, {:7.2} )",
            self.0, self.1, self.2, self.3, self.4, self.5
        )
    }
}

pub struct KiibohdData<'a> {
    //capabilities: &'a [Capability<'a>],
    results: &'a [ResultMacro<'a>],
    triggers: &'a [TriggerMacro<'a>],
    default_layer: &'a [TriggerLayer<'a>],
    partial_layers: &'a [TriggerLayer<'a>],
    layer_indices: &'a [LayerIndex<'a>],
    positions: &'a [PositionEntry],
    max_scan_code: usize,
    scancode_offsets: &'a [usize],
}

include!("test_data.rs");

pub fn write(file: &Path, groups: &KllGroups) {
    let config = groups.config();
    let mut capabilities = config
        .capabilities
        .values()
        .map(|cap| Capability(cap.function, cap.args.len()))
        .collect::<Vec<_>>();
    capabilities.sort_by_key(|cap| cap.0);

    let data = KIIBOHD_DATA;
    let content = format!(
        include_str!("kiibohd_keymap_template.h"),
        information = "// This file was generated by the kll compiler, DO NOT EDIT",
        capabilities_func_decl = capabilities
            .iter()
            .map(|cap| cap.func_decl_str())
            .collect::<Vec<_>>()
            .join("\n"),
        capabilities_list = capabilities
            .iter()
            .enumerate()
            .map(|(i, cap)| cap.cap_str(i))
            .collect::<Vec<_>>()
            .join("\n"),
        result_macros = data
            .results
            .iter()
            .enumerate()
            .map(|(i, result)| result.guide_str(i))
            .collect::<Vec<_>>()
            .join("\n"),
        result_macro_list = data
            .results
            .iter()
            .enumerate()
            .map(|(i, result)| result.define_str(i))
            .collect::<Vec<_>>()
            .join("\n"),
        trigger_macros = data
            .triggers
            .iter()
            .enumerate()
            .map(|(i, result)| result.guide_str(i))
            .collect::<Vec<_>>()
            .join("\n"),
        trigger_macro_list = data
            .triggers
            .iter()
            .enumerate()
            .map(|(i, result)| result.define_str(i))
            .collect::<Vec<_>>()
            .join("\n"),
        max_scan_code = data.max_scan_code,
        default_layer_trigger_list = data
            .default_layer
            .iter()
            .enumerate()
            .map(|(i, result)| result.define_str(i))
            .collect::<Vec<_>>()
            .join("\n"),
        partial_layer_trigger_lists = data
            .partial_layers
            .iter()
            .enumerate()
            .map(|(i, result)| result.define_str(i))
            .collect::<Vec<_>>()
            .join("\n"),
        scancode_interconnect_offset_list = data
            .scancode_offsets
            .iter()
            .map(|x| format!("\t{:#04X},", x))
            .collect::<Vec<_>>()
            .join("\n"),
        default_layer_scanmap = (0..data.default_layer.len())
            .map(|x| format!("default_tl_{:#04X}", x))
            .collect::<Vec<_>>()
            .join(", "),
        partial_layer_scanmaps = (0..data.partial_layers.len())
            .map(|x| format!("default_tl_{:#04X}", x))
            .collect::<Vec<_>>()
            .join(", "),
        layer_index_list = data
            .layer_indices
            .iter()
            .map(|layer| layer.define_str())
            .collect::<Vec<_>>()
            .join("\n"),
        rotation_parameters = "",
        key_positions = data
            .positions
            .iter()
            .map(|pos| pos.define_str())
            .collect::<Vec<_>>()
            .join("\n"),
        utf8_data = "",
    );
    fs::write(file, content).unwrap();
}
