use std::str::FromStr;

use serde::{Deserialize, Serialize};

use crate::error::CliError;

pub const PROJECT_CONFIG_FILE: &str = "Kite.toml";
pub const PROJECT_CONFIG_PATH: &str = "./Kite.toml";

#[derive(Default, Serialize, Deserialize)]
pub struct ProjectConfig {
    pub package: PackageConfig,
    pub build: BuildConfig,
}

impl ProjectConfig {
    pub fn read() -> Result<Self, CliError> {
        let raw = std::fs::read_to_string(PROJECT_CONFIG_PATH)?;
        Ok(toml::from_str(&raw)?)
    }

    pub fn flush(&self) -> Result<(), CliError> {
        let raw = toml::to_string(self)?;
        std::fs::write(PROJECT_CONFIG_PATH, raw)?;
        Ok(())
    }

    pub fn flush_to(&self, dir: &str) -> Result<(), CliError> {
        let path = format!("{}/{}", dir, PROJECT_CONFIG_FILE);
        let raw = toml::to_string(self)?;
        std::fs::write(path, raw)?;
        Ok(())
    }
}

#[derive(Default, Serialize, Deserialize)]
pub struct PackageConfig {
    pub id: String,
    pub name: String,
    #[serde(rename = "type", default)]
    pub kind: ProjectType,
    pub guild_id: u64,
}

#[derive(Serialize, Deserialize)]
#[serde(rename_all = "snake_case")]
pub enum ProjectType {
    #[serde(rename = "wasm")]
    WebAssembly,
    #[serde(rename = "js")]
    JavaScript,
}

impl FromStr for ProjectType {
    type Err = &'static str;

    fn from_str(s: &str) -> Result<Self, Self::Err> {
        use ProjectType::*;

        Ok(match s {
            "wasm" => WebAssembly,
            "js" => JavaScript,
            _ => return Err("Invalid project type"),
        })
    }
}

impl Default for ProjectType {
    fn default() -> Self {
        Self::WebAssembly
    }
}

#[derive(Default, Serialize, Deserialize)]
pub struct BuildConfig {
    #[serde(default)]
    pub command: Option<String>,
    pub file: String,
}
