use std::io::Error;
use confy::ConfyError;
use crate::client::ApiError;

pub type CliResult = Result<(), CliError>;

#[derive(Debug, thiserror::Error)]
pub enum CliError {
    #[error("Global config not found or invalid: {0}")]
    GlobalConfigError(ConfyError),
    #[error("Project config operation failed: {0}")]
    ProjectConfigIOError(std::io::Error),
    #[error("Project config deserialization failed: {0}")]
    ProjectConfigDeserialization(toml::de::Error),
    #[error("Project config serialization failed: {0}")]
    ProjectConfigSerialization(toml::ser::Error),
    #[error("API request failed: {0}")]
    ApiError(ApiError)
}

impl From<ConfyError> for CliError {
    fn from(e: ConfyError) -> Self {
        Self::GlobalConfigError(e)
    }
}

impl From<ApiError> for CliError {
    fn from(e: ApiError) -> Self {
        Self::ApiError(e)
    }
}

impl From<std::io::Error> for CliError {
    fn from(e: Error) -> Self {
        Self::ProjectConfigIOError(e)
    }
}

impl From<toml::de::Error> for CliError {
    fn from(e: toml::de::Error) -> Self {
        Self::ProjectConfigDeserialization(e)
    }
}

impl From<toml::ser::Error> for CliError {
    fn from(e: toml::ser::Error) -> Self {
        Self::ProjectConfigSerialization(e)
    }
}
