use std::borrow::Cow;
use std::env;
use std::io::Error;

use lazy_static::lazy_static;
use serde::de::DeserializeOwned;
use serde::{Deserialize, Serialize};

lazy_static! {
    pub static ref API_BASE_URL: String =
        env::var("API_BASE_URL").unwrap_or(String::from("https://kite.bot/api"));
}

pub struct ApiClient {
    token: String,
}

impl ApiClient {
    pub fn new(token: String) -> Self {
        Self { token }
    }

    pub fn request<D: Serialize, T: DeserializeOwned>(
        &self,
        method: &str,
        path: &str,
        data: Option<D>,
    ) -> ApiResult<T> {
        let req = ureq::request(method, &format!("{}{}", API_BASE_URL.as_str(), path))
            .set("Authorization", &self.token);

        let resp: ApiResponseWire<T> = match data {
            Some(d) => req.send_json(d),
            None => req.call(),
        }?
        .into_json()?;

        resp.into()
    }

    pub fn user_get(&self, user_id: &str) -> ApiResult<UserWire> {
        self.request::<(), _>("GET", &format!("/users/{}", user_id), None)
    }

    pub fn module_list(&self, guild_id: u64) -> ApiResult<ModuleListResponseWire> {
        self.request::<(), _>("GET", &format!("/guilds/{}/modules", guild_id), None)
    }

    pub fn module_get(&self, guild_id: u64, id: &str) -> ApiResult<ModuleWire> {
        self.request::<(), _>("GET", &format!("/guilds/{}/modules/{}", guild_id, id), None)
    }

    pub fn module_deploy(&self, guild_id: u64, req: &ModuleDeployRequestWire) -> ApiResult<ModuleWire> {
        self.request("POST", &format!("/guilds/{}/modules", guild_id), Some(req))
    }

    pub fn module_delete(&self, guild_id: u64, id: &str) -> ApiResult<ModuleDeleteResponseWire> {
        self.request::<(), _>("DELETE", &format!("/guilds/{}/modules/{}", guild_id, id), None)
    }
}

pub type ApiResult<T> = Result<T, ApiError>;

#[derive(Debug, Clone, thiserror::Error, Serialize, Deserialize)]
#[serde(rename_all = "snake_case", tag = "code")]
pub enum ApiError {
    #[error("Network error: {0}")]
    NetworkError(String),
    #[error("Request error: {0}")]
    RequestError(String),

    #[error("Not found")]
    NotFound { entity: Cow<'static, str> },
    #[error("Database operation failed")]
    DatabaseError,
    #[error("Failed to join blocking thread")]
    ThreadJoinError,
    #[error("Validation failed")]
    ValidationError {
        field: Cow<'static, str>,
        details: Cow<'static, str>,
    },
    #[error("WASM compilation has failed")]
    InvalidWasmModule { details: String },
    #[error("WASM runtime failed")]
    RuntimeError {},
    #[error("No or invalid token provided")]
    InvalidToken,
    #[error("Background fetch request failed")]
    BackgroundRequestFailed,
    #[error("Invalid snowflake provided")]
    InvalidSnowflake,
    #[error("Missing guild access")]
    MissingGuildAccess,
}

impl From<std::io::Error> for ApiError {
    fn from(e: Error) -> Self {
        Self::NetworkError(e.to_string())
    }
}

impl From<ureq::Error> for ApiError {
    fn from(e: ureq::Error) -> Self {
        match e {
            ureq::Error::Transport(e) => Self::RequestError(e.to_string()),
            ureq::Error::Status(_, resp) => {
                let e: ApiResponseWire<()> = match resp.into_json() {
                    Ok(e) => e,
                    Err(e) => return e.into()
                };
                e.error.unwrap()
            }
        }
    }
}

#[derive(Debug, Serialize, Deserialize)]
pub struct ApiResponseWire<T> {
    pub success: bool,
    pub data: Option<T>,
    pub error: Option<ApiError>,
}

impl<T> From<ApiResponseWire<T>> for ApiResult<T> {
    fn from(r: ApiResponseWire<T>) -> Self {
        if r.success {
            Ok(r.data.unwrap())
        } else {
            Err(r.error.unwrap())
        }
    }
}

#[derive(Debug, Serialize, Deserialize)]
pub struct UserWire {
    pub id: String,
    pub username: String,
    pub discriminator: String,
}

#[derive(Debug, Serialize, Deserialize)]
pub struct ModuleListResponseWire {
    pub modules: Vec<ModuleWire>,
}

#[derive(Debug, Serialize, Deserialize)]
pub struct ModuleWire {
    pub id: String,
    pub name: String,
    pub guild_id: String,
    pub enabled: bool,
}

#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct ModuleDeployRequestWire {
    pub id: String,
    pub name: String,
    pub enabled: bool,
    #[serde(flatten)]
    pub kind: ModuleDeployRequestWireType,
}

#[derive(Debug, Clone, Serialize, Deserialize)]
#[serde(tag = "type")]
pub enum ModuleDeployRequestWireType {
    #[serde(rename = "wasm")]
    WebAssembly {
        #[serde(default)]
        body: String,
    },
    #[serde(rename = "js")]
    JavaScript {
        #[serde(default)]
        body: String,
    },
}

#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct ModuleDeleteResponseWire {}
