use std::str::FromStr;

use serde::{Deserialize, Serialize};

use crate::error::CliError;

pub const PROJECT_CONFIG_FILE: &str = "Kite.toml";
pub const PROJECT_CONFIG_PATH: &str = "./Kite.toml";

#[derive(Default, Serialize, Deserialize)]
pub struct ProjectConfig {
    pub package: PackageConfig,
    pub build: BuildConfig,
}

impl ProjectConfig {
    pub fn read() -> Result<Self, CliError> {
        Ok(confy::load_path(PROJECT_CONFIG_PATH)?)
    }

    pub fn flush(&self) -> Result<(), CliError> {
        confy::store_path(PROJECT_CONFIG_PATH, self)?;
        Ok(())
    }

    pub fn flush_to(&self, dir: &str) -> Result<(), CliError> {
        let path = format!("{}/{}", dir, PROJECT_CONFIG_FILE);
        confy::store_path(path, self)?;
        Ok(())
    }
}

#[derive(Default, Serialize, Deserialize)]
pub struct PackageConfig {
    pub id: String,
    pub name: String,
    #[serde(rename = "type", default)]
    pub kind: ProjectType,
    pub guild_id: u64,
}

#[derive(Serialize, Deserialize)]
#[serde(rename_all = "snake_case")]
pub enum ProjectType {
    #[serde(rename = "wasm")]
    WebAssembly,
    #[serde(rename = "js")]
    JavaScript,
}

impl FromStr for ProjectType {
    type Err = &'static str;

    fn from_str(s: &str) -> Result<Self, Self::Err> {
        use ProjectType::*;

        Ok(match s {
            "wasm" => WebAssembly,
            "js" => JavaScript,
            _ => return Err("Invalid project type"),
        })
    }
}

impl Default for ProjectType {
    fn default() -> Self {
        Self::WebAssembly
    }
}

#[derive(Default, Serialize, Deserialize)]
pub struct BuildConfig {
    #[serde(default)]
    pub command: Option<String>,
    pub file: String,
}
