use clap::Args;
use nanoid::nanoid;

use crate::client::{ApiClient, ModuleDeployRequestWire, ModuleDeployRequestWireType};
use crate::error::CliResult;
use crate::global_config::GlobalConfig;
use crate::project_config::ProjectType;

#[derive(Args)]
pub struct DeployArgs {
    /// Path to the file that will be deployed (usually .wasm)
    pub path: String,
    /// Type of the module (wasm by default)
    #[clap(long = "type", short = 't', name = "type")]
    pub kind: Option<ProjectType>,
    /// Unique ID for the module
    #[clap(long)]
    pub id: Option<String>,
    /// Display name for the module
    #[clap(long)]
    pub name: String,
    #[clap(long)]
    pub guild_id: u64,
}

pub fn module_deploy_command(args: DeployArgs) -> CliResult {
    let file_body = std::fs::read(args.path).expect("Reading file from path");

    let kind = match args.kind.unwrap_or_default() {
        ProjectType::WebAssembly => ModuleDeployRequestWireType::WebAssembly {
            body: base64::encode(file_body),
        },
        ProjectType::JavaScript => ModuleDeployRequestWireType::JavaScript {
            body: String::from_utf8(file_body).unwrap(),
        },
    };

    let token = GlobalConfig::read_or_default()?.token.unwrap();
    let client = ApiClient::new(token);

    let module = client.module_deploy(
        args.guild_id,
        &ModuleDeployRequestWire {
            id: args.id.unwrap_or_else(|| nanoid!()),
            name: args.name,
            kind,
            enabled: true,
        },
    )?;

    println!("{:#?}", module);

    Ok(())
}
