use crate::document::{Document, DocumentState, ActionHistory};
use crate::editorgui::{Packages, ResourceType, EditorConfig, ProjectConfig};

use std::any::Any;

use keeshond::gameloop::GameControl;
use keeshond_datapack::{DataId, DataHandle, DataStore};
use keeshond_treats::spawnable::{Spawnable, SpawnableSet, Representation};

use keeshond_treats::visual::Costume;
use keeshond::renderer::Sheet;

enum SpawnableSetAction
{
    AddSpawnable(usize, Spawnable),
    RenameSpawnable(usize, String),
    SetSpawnable(usize, Spawnable),
    MoveSpawnable(usize, usize),
    RemoveSpawnable(usize),
}

pub struct SpawnableSetDocument
{
    action_history : ActionHistory,
    data_id : DataId<SpawnableSet>,
    current_spawnable : usize,
    scroll_spawn_list : bool,
    focus_spawnable_name_input : bool,
    transparent_color : [f32; 3],
}

impl SpawnableSetDocument
{
    fn add_spawnable(&mut self, index : usize, spawnables : &mut SpawnableSet, spawnable : Spawnable)
    {
        let mut target_spawnable = 0;

        if spawnables.spawnable_count() > 0
        {
            target_spawnable = index + 1;
        }

        spawnables.insert_spawnable(target_spawnable, spawnable);

        self.current_spawnable += 1;
    }

    fn move_spawnable(&mut self, spawnables : &mut SpawnableSet, old : usize, new : usize) -> bool
    {
        if new >= spawnables.spawnable_count()
        {
            return false;
        }

        if let Some(spawnable) = spawnables.spawnable(old).cloned()
        {
            let old_name = spawnables.spawnable_name(old);

            spawnables.remove_spawnable(old);
            spawnables.insert_spawnable(new, spawnable);

            if let Some(name) = old_name
            {
                spawnables.rename_spawnable(new, &name);
            }

            self.current_spawnable = new;

            return true;
        }

        false
    }

    fn spawnable_tab(&mut self, packages : &mut Packages, ui : &imgui::Ui, spawnables : &SpawnableSet, spawnable : &Spawnable)
    {
        let mut spawnable_name = spawnables.spawnable_name(self.current_spawnable).unwrap_or_default();

        if self.focus_spawnable_name_input
        {
            ui.set_keyboard_focus_here_with_offset(imgui::FocusedWidget::Offset(0));
            self.focus_spawnable_name_input = false;
        }

        if ui.input_text("Name###spawnable_name", &mut spawnable_name).enter_returns_true(true).build()
        {
            self.queue_action(Box::new(SpawnableSetAction::RenameSpawnable(self.current_spawnable, spawnable_name.to_string())));
        }

        ui.spacing();

        let mut index = spawnable.representation().index();

        if ui.combo_simple_string("Representation", &mut index, &crate::util::REPRESENTATION_NAMES)
        {
            let representation = Representation::from_index(index);

            let mut spawnable_clone = spawnable.clone();
            spawnable_clone.set_representation(representation);

            self.queue_action(Box::new(SpawnableSetAction::SetSpawnable(self.current_spawnable, spawnable_clone)));
        }

        imgui::ChildWindow::new("representation_controls").size([0.0, 90.0]).draw_background(false).border(false).build(ui, ||
        {
            ui.indent_by(10.0);

            match spawnable.representation()
            {
                Representation::None => {}
                Representation::Actor { costume } =>
                {
                    let mut update_actor = false;
                    let mut package_name = costume.package().clone();
                    let mut costume_name = costume.path().clone();

                    ui.popup("Pick Costume Menu", ||
                    {
                        let mut package_to_load: Option<String> = None;

                        for package in packages.package_iter()
                        {
                            ui.menu(package, ||
                            {
                                if !packages.package_loaded(ResourceType::Costume, package)
                                {
                                    package_to_load = Some(package.clone());
                                }

                                for costume in packages.package_contents(ResourceType::Costume, package)
                                {
                                    if imgui::Selectable::new(costume).build(ui)
                                    {
                                        package_name.clear();
                                        package_name.push_str(package);
                                        costume_name.clear();
                                        costume_name.push_str(costume);

                                        update_actor = true;
                                    }
                                }
                            });
                        }

                        if let Some(package) = package_to_load
                        {
                            packages.load_package_later(ResourceType::Costume, &package);
                        }
                    });

                    ui.spacing();
                    if ui.button("Pick Costume")
                    {
                        ui.open_popup("Pick Costume Menu");
                    }
                    ui.spacing();

                    let style_var = ui.push_style_var(imgui::StyleVar::ItemSpacing([-1.0, -1.0]));
                    let width = ui.push_item_width(ui.column_width(0) * 0.6);

                    update_actor |= ui.input_text("Costume Package", &mut package_name).enter_returns_true(true).build();
                    update_actor |= ui.input_text("Costume Name", &mut costume_name).enter_returns_true(true).build();

                    width.pop(ui);
                    style_var.pop();

                    if update_actor
                    {
                        let mut spawnable_clone = spawnable.clone();

                        spawnable_clone.set_representation(Representation::Actor
                        {
                            costume: DataHandle::with_path(&package_name, &costume_name)
                        });

                        self.queue_action(Box::new(SpawnableSetAction::SetSpawnable(self.current_spawnable, spawnable_clone)));
                    }
                }
                Representation::Tilemap => {}
            }

            ui.unindent_by(10.0);
        });

        let mut auto_layer = spawnable.auto_layer_name().clone();

        if ui.input_text("Auto-Set Layer", &mut auto_layer).enter_returns_true(true).build()
        {
            let mut spawnable_clone = spawnable.clone();

            spawnable_clone.set_auto_layer_name(auto_layer);

            self.queue_action(Box::new(SpawnableSetAction::SetSpawnable(self.current_spawnable, spawnable_clone)));
        }
    }

    fn spawnable_list_view(&mut self, ui : &imgui::Ui, spawnables : &mut SpawnableSet,
                           costume_store : &mut DataStore<Costume>, sheet_store : &mut DataStore<Sheet>,
                           packages : &mut Packages)
    {
        let main_style = ui.clone_style();

        self.current_spawnable = self.current_spawnable.min(spawnables.spawnable_count().max(1) - 1);

        if ui.button("Add")
        {
            self.queue_action(Box::new(SpawnableSetAction::AddSpawnable(self.current_spawnable, Spawnable::new())));
        }
        ui.same_line_with_spacing(0.0, 4.0);
        if ui.button("Up") && self.current_spawnable > 0
        {
            self.queue_action(Box::new(SpawnableSetAction::MoveSpawnable(self.current_spawnable, self.current_spawnable - 1)));
        }
        ui.same_line_with_spacing(0.0, 4.0);
        if ui.button("Down")
        {
            self.queue_action(Box::new(SpawnableSetAction::MoveSpawnable(self.current_spawnable, self.current_spawnable + 1)));
        }
        ui.same_line_with_spacing(0.0, 4.0);
        if ui.button("Remove")
        {
            self.queue_action(Box::new(SpawnableSetAction::RemoveSpawnable(self.current_spawnable)));
        }

        let style_spawnable_list = ui.push_style_var(imgui::StyleVar::WindowPadding([3.0, 2.0]));
        let colors_spawnable_list = ui.push_style_color(imgui::StyleColor::ChildBg, main_style.colors[imgui::StyleColor::FrameBg as usize]);

        if let Some(spawnable_list) = imgui::ChildWindow::new("spawnable_list").border(true).draw_background(true).begin(ui)
        {
            crate::util::ui_spawnable_list(spawnables, &mut self.current_spawnable, &mut self.scroll_spawn_list, costume_store, sheet_store, packages, ui);

            spawnable_list.end();
        }

        colors_spawnable_list.pop();
        style_spawnable_list.pop();
    }

    fn spawnable_image(&mut self, packages : &mut Packages, ui : &imgui::Ui, sheet_store : &mut DataStore<Sheet>, costume_store : &mut DataStore<Costume>, spawnable : &Spawnable)
    {
        let remaining = ui.content_region_avail();

        match spawnable.representation()
        {
            Representation::None => {}
            Representation::Actor { costume } =>
            {
                if let Some((mut image, rect)) = crate::util::costume_image(costume, costume_store, sheet_store, packages)
                {
                    let width_zoom = ((remaining[0] - 2.0) / rect.w.max(1.0)).floor();
                    let height_zoom = ((remaining[1] - 2.0) / rect.h.max(1.0)).floor();
                    let zoom = width_zoom.min(height_zoom).clamp(1.0, 4.0);
                    let size = [rect.w * zoom, rect.h * zoom];

                    let style_var = ui.push_style_var(imgui::StyleVar::WindowPadding([1.0, 1.0]));
                    let style_color = ui.push_style_color(imgui::StyleColor::ChildBg, [self.transparent_color[0], self.transparent_color[1], self.transparent_color[2], 1.0]);

                    imgui::ChildWindow::new("texture_view").size([size[0] + 2.0, size[1] + 2.0]).draw_background(true).border(true)
                        .scrollable(false).movable(false).build(ui, ||
                    {
                        image = image.size(size);
                        image.build(ui);
                    });

                    style_color.pop();
                    style_var.pop();
                }
            }
            Representation::Tilemap => {}
        }
    }
}

impl Document for SpawnableSetDocument
{
    fn load(game : &mut GameControl, package_name : &str, item_name : &str) -> Self where Self : Sized
    {
        let mut data_id = DataId::new(0);

        if let Ok(id) = game.res().store_mut::<SpawnableSet>().get_id_mut(package_name, item_name)
        {
            data_id = id;
        }

        SpawnableSetDocument
        {
            action_history : ActionHistory::new(),
            data_id,
            current_spawnable : 0,
            scroll_spawn_list : false,
            focus_spawnable_name_input : false,
            transparent_color : [0.5, 0.5001, 0.5002]
        }
    }

    fn do_ui(&mut self, packages : &mut Packages, _config : &mut EditorConfig,
             _project : &mut ProjectConfig, game : &mut GameControl, ui : &imgui::Ui) -> DocumentState
    {
        let mut result = DocumentState::new();

        result.active = ui.is_window_focused();

        let mut sheet_store = game.res().store_mut::<Sheet>();
        let mut costume_store = game.res().store_mut::<Costume>();
        let mut spawnable_store = game.res().store_mut::<SpawnableSet>();

        if let Some(spawnables) = spawnable_store.get_mut(self.data_id)
        {
            ui.columns(2, "##spawnable_columns", false);

            if let Some(spawnable_controls) = imgui::ChildWindow::new("spawnable_controls")
                .size([320.0, 0.0]).draw_background(true).border(false).horizontal_scrollbar(true).begin(ui)
            {
                if let Some(spawnable) = spawnables.spawnable(self.current_spawnable)
                {
                    ui.text(format!("Editing Spawnable {}", self.current_spawnable));
                    ui.spacing();
                    ui.spacing();

                    if let Some(tab_bar) = imgui::TabBar::new("spawnable_tab_bar")
                        .flags(imgui::TabBarFlags::FITTING_POLICY_SCROLL).begin(ui)
                    {
                        if let Some(tab) = imgui::TabItem::new("Spawnable").begin(ui)
                        {
                            self.spawnable_tab(packages, ui, spawnables, spawnable);

                            tab.end();
                        }

                        tab_bar.end();
                    }
                }

                ui.spacing();
                ui.spacing();

                if let Some(spawnable_list_view) = imgui::ChildWindow::new("spawnable_list_view").size([0.0, 0.0]).movable(false).begin(ui)
                {
                    self.spawnable_list_view(ui, spawnables, &mut costume_store, &mut sheet_store, packages);

                    spawnable_list_view.end();
                }

                spawnable_controls.end();
            }

            ui.set_current_column_width(ui.item_rect_size()[0] + 10.0);
            ui.next_column();

            if let Some(spawnable) = spawnables.spawnable(self.current_spawnable)
            {
                self.spawnable_image(packages, ui, &mut sheet_store, &mut costume_store, spawnable)
            }
        }

        result
    }

    fn action_history(&mut self) -> &mut ActionHistory
    {
        &mut self.action_history
    }

    fn apply_action(&mut self, game : &mut GameControl, action : &Box<dyn Any>) -> bool
    {
        if let Some(action) = action.downcast_ref::<SpawnableSetAction>()
        {
            let mut spawnable_store = game.res().store_mut::<SpawnableSet>();

            if let Some(spawnables) = spawnable_store.get_mut(self.data_id)
            {
                match action
                {
                    SpawnableSetAction::AddSpawnable(index, spawnable) =>
                    {
                        self.scroll_spawn_list = true;
                        self.focus_spawnable_name_input = true;
                        self.add_spawnable(*index, spawnables, spawnable.clone());
                        return true;
                    }
                    SpawnableSetAction::RenameSpawnable(index, name) =>
                    {
                        self.scroll_spawn_list = true;
                        return spawnables.rename_spawnable(*index, name);
                    }
                    SpawnableSetAction::SetSpawnable(index, spawnable) =>
                    {
                        self.scroll_spawn_list = true;
                        return spawnables.set_spawnable(*index, spawnable.clone());
                    }
                    SpawnableSetAction::MoveSpawnable(old_index, new_index) =>
                    {
                        self.scroll_spawn_list = true;
                        return self.move_spawnable(spawnables, *old_index, *new_index);
                    }
                    SpawnableSetAction::RemoveSpawnable(index) =>
                    {
                        self.scroll_spawn_list = true;
                        return spawnables.remove_spawnable(*index).is_some();
                    }
                }
            }
        }

        false
    }
}
