use image::RgbaImage;
use crate::editorgui::{Packages, ResourceType};

use keeshond::renderer::{SpriteSlice, Sheet};
use keeshond::util::{BakedRect, Rect};

use keeshond_datapack::{DataStore, DataHandle, DataId};
use keeshond_treats::visual::{AnimationCel, Costume, DrawableItem, MultiTileSprite, TileSprite};

use num::Float;
use toodee::TooDeeOps;

use keeshond_treats::tilemap::{Tilemap, Tileset};
use keeshond_treats::spawnable::Representation;
use log::Level;

pub const ICON_PACKAGE : &'static str = "a";
pub const ICON_SHEET : &'static str = "b";
pub const ICON_COSTUME : &'static str = "c";
pub const ICON_ACTOR : &'static str = "d";
pub const ICON_TILES : &'static str = "e";
#[allow(dead_code)] pub const ICON_SHADER : &'static str = "f";
#[allow(dead_code)] pub const ICON_SOUND : &'static str = "g";
#[allow(dead_code)] pub const ICON_MUSIC : &'static str = "h";
pub const ICON_SPAWNABLE : &'static str = "i";
pub const ICON_LEVEL : &'static str = "j";
#[allow(dead_code)] pub const ICON_PLAY : &'static str = "k";
#[allow(dead_code)] pub const ICON_STOP : &'static str = "l";
#[allow(dead_code)] pub const ICON_PAUSE : &'static str = "m";
#[allow(dead_code)] pub const ICON_SEEK_BACK : &'static str = "n";
#[allow(dead_code)] pub const ICON_SEEK_NEXT : &'static str = "o";
#[allow(dead_code)] pub const ICON_VISIBLE : &'static str = "p";
#[allow(dead_code)] pub const ICON_SETTINGS : &'static str = "q";
#[allow(dead_code)] pub const ICON_FOLDER : &'static str = "r";
pub const ICON_MARKER : &'static str = "s";
#[allow(dead_code)] pub const ICON_STRING : &'static str = "t";
#[allow(dead_code)] pub const ICON_ADD : &'static str = "u";
#[allow(dead_code)] pub const ICON_REMOVE : &'static str = "v";
#[allow(dead_code)] pub const ICON_LEFT : &'static str = "w";
#[allow(dead_code)] pub const ICON_RIGHT : &'static str = "x";
#[allow(dead_code)] pub const ICON_UP : &'static str = "y";
#[allow(dead_code)] pub const ICON_DOWN : &'static str = "z";
#[allow(dead_code)] pub const ICON_GRID : &'static str = "A";
#[allow(dead_code)] pub const ICON_SNAP : &'static str = "B";
#[allow(dead_code)] pub const ICON_PATH : &'static str = "C";
#[allow(dead_code)] pub const ICON_STAR : &'static str = "D";
pub const ICON_CURSOR : &'static str = "E";
pub const ICON_DRAW : &'static str = "F";
pub const ICON_LAYER : &'static str = "G";
pub const ICON_LOCKED : &'static str = "H";
#[allow(dead_code)] pub const ICON_UNLOCKED : &'static str = "I";

pub const ICON_FONT_SIZE : f32 = 16.0;

pub const REPRESENTATION_NAMES : [&'static str; 3] = ["None", "Actor", "Tilemap"];

const PASCAL_DISALLOWED_CHARS : [char; 33] = [' ', '!', '"', '#', '$', '%', '&', '\'', '(', ')',
    '*', '+', ',', '-', '.', '/', ':', ';', '<', '=',
    '>', '?', '@', '[', '\\', ']', '^', '_', '`', '{',
    '|', '}', '~'];

pub fn representation_icon(representation : &Representation) -> &'static str
{
    match representation
    {
        Representation::None => { crate::util::ICON_MARKER }
        Representation::Actor { .. } => { crate::util::ICON_ACTOR }
        Representation::Tilemap => { crate::util::ICON_TILES }
    }
}

pub fn window_placeholder_text(text : &str, ui : &imgui::Ui)
{
    let size = ui.calc_text_size(text);
    let avail = ui.content_region_avail();
    let mut cursor_pos = ui.cursor_pos();
    cursor_pos[0] += avail[0] / 2.0 - size[0] / 2.0;

    ui.set_cursor_pos(cursor_pos);
    ui.text_disabled(text);
}

pub fn right_aligned_label(text : &str, frame_padding : bool, ui : &imgui::Ui)
{
    let size = ui.calc_text_size(text);
    let style = ui.clone_style();

    let cursor_pos = ui.cursor_pos();
    let mut advance = 0.0;
    let avail_width = ui.current_column_width() - style.item_spacing[0] - 1.0;

    if size[0] < avail_width
    {
        advance = avail_width - size[0];
    }

    ui.set_cursor_pos([cursor_pos[0] + advance, cursor_pos[1]]);

    if frame_padding
    {
        ui.align_text_to_frame_padding();
    }

    ui.text(text);
}

pub fn underline_current_item(drop : f32, extend : f32, thickness : f32, ui : &imgui::Ui)
{
    let draw_list = ui.get_window_draw_list();
    let color = ui.style_color(imgui::StyleColor::ButtonActive);
    let size_min = ui.item_rect_min();
    let size_max = ui.item_rect_max();
    draw_list.add_line([size_min[0] - extend, size_max[1] + drop],
                       [size_max[0] + extend, size_max[1] + drop], color).thickness(thickness).build();
}

pub fn column_separator(ui : &imgui::Ui)
{
    let draw_list = ui.get_window_draw_list();
    let color = ui.style_color(imgui::StyleColor::Separator);
    let pos = ui.cursor_screen_pos();
    let width = ui.window_content_region_width();
    draw_list.add_line([pos[0], pos[1]],
                       [pos[0] + width, pos[1]], color).thickness(1.0).build();
}

pub fn error_text_color(ui : &imgui::Ui) -> [f32; 4]
{
    let mut color = ui.style_color(imgui::StyleColor::Text);

    color[0] = (color[0] + 2.0) / 3.0;
    color[1] /= 3.0;
    color[2] /= 3.0;
    color[3] = 1.0;

    color
}

pub fn text_color_for_log_level(level : log::Level, ui : &imgui::Ui) -> [f32; 4]
{
    let mut color = ui.style_color(imgui::StyleColor::Text);

    match level
    {
        Level::Error =>
        {
            color[0] = (color[0] + 2.0) / 3.0;
            color[1] /= 3.0;
            color[2] /= 3.0;
        }
        Level::Warn =>
        {
            color[0] = (color[0] + 2.0) / 3.0;
            color[1] = (color[1] + 1.0) / 3.0;
            color[2] /= 3.0;
        }
        Level::Info =>
        {
            color[0] /= 3.0;
            color[1] = (color[1] + 1.0) / 3.0;
            color[2] = (color[2] + 2.0) / 3.0;
        }
        Level::Debug =>
        {
            color[0] = (color[0] + 1.5) / 3.0;
            color[1] /= 3.0;
            color[2] = (color[2] + 2.0) / 3.0;
        }
        Level::Trace =>
        {
            color[0] /= 3.0;
            color[1] = (color[1] + 2.0) / 3.0;
            color[2] /= 3.0;
        }
    }

    color[3] = 1.0;

    color
}

pub struct BoxFitResult<T : Float>
{
    pub x : T,
    pub y : T,
    pub width : T,
    pub height : T,
    pub scale : T
}

pub fn box_fit<T : Float>(base_width : T, base_height : T, parent_width : T, parent_height : T) -> BoxFitResult<T>
{
    let mut result = BoxFitResult::<T>
    {
        x : T::from(0.0).unwrap(),
        y : T::from(0.0).unwrap(),
        width : parent_width,
        height : parent_height,
        scale : T::from(1.0).unwrap(),
    };

    let base_ratio = base_width / base_height;
    let ratio = parent_width / parent_height;

    if ratio > base_ratio
    {
        // Wider than base resolution
        result.scale = parent_height / base_height;
        result.width = (parent_height * base_ratio) as T;
        result.x = ((parent_width) - result.width) / T::from(2.0).unwrap();
    }
    else if ratio < base_ratio
    {
        // Taller than base resolution
        result.scale = parent_width / base_width;
        result.height = (parent_width / base_ratio) as T;
        result.y = ((parent_height) - result.height) / T::from(2.0).unwrap();
    }

    result
}

pub fn color_to_border_color(color : [f32; 4]) -> [f32; 4]
{
    if color[0] + color[1] + color[2] > 1.5
    {
        return [0.0, 0.0, 0.0, 1.0]
    }

    [1.0, 1.0, 1.0, 1.0]
}

pub fn slice_to_uv(sheet : &Sheet, slice : &SpriteSlice) -> BakedRect<f32>
{
    let base_width = sheet.width() as f32;
    let base_height = sheet.height() as f32;

    BakedRect::<f32>
    {
        x1 : slice.texture_x / base_width,
        y1 : slice.texture_y / base_height,
        x2 : (slice.texture_x + slice.texture_w) / base_width,
        y2 : (slice.texture_y + slice.texture_h) / base_height
    }
}

pub fn option_bool_to_double_bool(option_bool : Option<bool>) -> (bool, bool)
{
    if let Some(some_bool) = option_bool
    {
        return (true, some_bool);
    }

    (false, false)
}

pub fn double_bool_to_option_bool(is_some : bool, inner_value : bool) -> Option<bool>
{
    if !is_some
    {
        return None;
    }

    Some(inner_value)
}

pub fn get_icon_font(ui : &imgui::Ui) -> imgui::FontId
{
    ui.fonts().fonts()[ui.fonts().fonts().len() - 1]
}

pub fn icon_label(icon : &str, name : &str, ui : &imgui::Ui)
{
    let old_cursor = ui.cursor_pos();
    let height = ui.current_font().font_size;
    let icon_font = get_icon_font(ui);

    let font = ui.push_font(icon_font);
    let icon_size = ui.calc_text_size(icon);
    ui.set_cursor_pos([old_cursor[0] + ((ICON_FONT_SIZE - icon_size[0]) / 2.0), old_cursor[1]]);

    ui.text(icon);
    font.pop();

    ui.same_line();
    ui.set_cursor_pos([old_cursor[0] + 22.0, old_cursor[1] + ((icon_size[1] - height) / 2.0)]);
    ui.text(name);
}

pub fn height_with_icon(ui : &imgui::Ui) -> f32
{
    let size = ui.calc_text_size("");

    size[1].max(15.0)
}

pub fn data_path_edit(ui : &imgui::Ui, label : &str, package : &mut String, name : &mut String) -> bool
{
    let mut changed = false;

    let width = ui.push_item_width(120.0);
    changed |= ui.input_text(format!("/###{} Package", label), package).build();
    width.pop(ui);

    ui.same_line_with_spacing(0.0, 4.0);

    let width = ui.push_item_width(160.0);
    changed |= ui.input_text(format!("{} Package/Name", label), name).build();
    width.pop(ui);

    changed
}

pub fn name_to_pascal_case(name : &str) -> String
{
    let parts : Vec<String> = name.split(&PASCAL_DISALLOWED_CHARS[..]).map(|part| part.to_string()).collect();
    let mut new_name = String::new();

    for part in &parts
    {
        let mut start_of_word = true;

        for letter in part.chars()
        {
            if start_of_word
            {
                new_name.push(letter.to_ascii_uppercase());
                start_of_word = false;
            }
            else
            {
                new_name.push(letter.to_ascii_lowercase());
            }
        }
    }

    new_name
}

pub fn tile_size(tilemap : &Tilemap, tileset_store : &DataStore<Tileset>) -> (f64, f64)
{
    let mut tile_w = 16.0;
    let mut tile_h = 16.0;

    if let Some(tileset) = tilemap.tileset().get_if_resolved(tileset_store)
    {
        let (w, h) = tileset.tile_size();
        tile_w = w as f64;
        tile_h = h as f64;
    }
    (tile_w, tile_h)
}

pub fn cel_image(cel : &AnimationCel, sheet_store : &mut DataStore<Sheet>, packages : &mut Packages) -> Option<(imgui::Image, Rect<f32>)>
{
    let (package, sheet_name) = (cel.sheet.package(), cel.sheet.path());

    if package.is_empty() || sheet_name.is_empty()
    {
        return None;
    }

    if let Ok(sheet_id) = sheet_store.get_id(package, sheet_name)
    {
        if let Some(sheet) = sheet_store.get(sheet_id)
        {
            if let Some(slice_id) = sheet.slice_id(&cel.slice.slice())
            {
                if let Some(slice) = sheet.slice(slice_id)
                {
                    let (w, h) = (slice.texture_w, slice.texture_h);
                    let rect = crate::util::slice_to_uv(sheet, slice);

                    return Some((imgui::Image::new(imgui::TextureId::from(sheet_id), [w, h])
                        .uv0([rect.x1, rect.y1]).uv1([rect.x2, rect.y2]), Rect
                    {
                        x : slice.origin_x, y : slice.origin_y, w, h
                    }));
                }
            }
        }
    }
    else
    {
        packages.load_package_later(ResourceType::Sheet, package);
    }

    None
}

pub fn cel_sheet_slice(cel : &AnimationCel, sheet_store : &mut DataStore<Sheet>, packages : &mut Packages) -> Option<(DataId, usize)>
{
    let (package, sheet_name) = (cel.sheet.package(), cel.sheet.path());

    if package.is_empty() || sheet_name.is_empty()
    {
        return None;
    }

    if let Ok(sheet_id) = sheet_store.get_id(package, sheet_name)
    {
        if let Some(sheet) = sheet_store.get(sheet_id)
        {
            if let Some(slice_id) = sheet.slice_id(&cel.slice.slice())
            {
                return Some((sheet_id, slice_id));
            }
        }
    }
    else
    {
        packages.load_package_later(ResourceType::Sheet, package);
    }

    None
}

pub fn cel_image_fit(cel : &AnimationCel, width : f32, height : f32,
                            sheet_store : &mut DataStore<Sheet>, packages : &mut Packages) -> Option<(imgui::Image, BoxFitResult<f32>)>
{
    if let Some((mut image, rect)) = cel_image(cel, sheet_store, packages)
    {
        let fit = box_fit(rect.w, rect.h, width, height);

        image = image.size([fit.width, fit.height]);
        return Some((image, fit));
    }

    None
}

pub fn costume_image(costume : &DataHandle<Costume>, costume_store : &mut DataStore<Costume>,
                     sheet_store : &mut DataStore<Sheet>, packages : &mut Packages) -> Option<(imgui::Image, Rect<f32>)>
{
    if costume_store.package_loaded(costume.package())
    {
        let costume_id = costume.resolve_if_loaded(costume_store);

        if let Some(costume) = costume_store.get(costume_id)
        {
            if let Some(anim) = costume.animation(0)
            {
                if let Some(cel) = anim.cel(0)
                {
                    return cel_image(cel, sheet_store, packages);
                }
            }
        }
    }
    else
    {
        packages.load_package_later(ResourceType::Costume, costume.package());
    }

    None
}

pub fn costume_sheet_slice(costume : &DataHandle<Costume>, costume_store : &mut DataStore<Costume>,
                     sheet_store : &mut DataStore<Sheet>, packages : &mut Packages) -> Option<(DataId, usize)>
{
    if costume_store.package_loaded(costume.package())
    {
        let costume_id = costume.resolve_if_loaded(costume_store);

        if let Some(costume) = costume_store.get(costume_id)
        {
            if let Some(anim) = costume.animation(0)
            {
                if let Some(cel) = anim.cel(0)
                {
                    return cel_sheet_slice(cel, sheet_store, packages);
                }
            }
        }
    }
    else
    {
        packages.load_package_later(ResourceType::Costume, costume.package());
    }

    None
}

pub fn tileset_image_from_slice(tileset : &mut Tileset, texture_id : usize, sheet_opt : Option<&Sheet>, slice : usize, image_size : [f32; 2]) -> imgui::Image
{
    if let Some(sheet) = sheet_opt
    {
        if let Some(index) = tileset.updated_slice_id(slice, sheet)
        {
            if let Some(slice) = sheet.slice(index)
            {
                return imgui::Image::new(imgui::TextureId::from(texture_id), image_size)
                    .uv0([slice.texture_x / sheet.width() as f32,
                        slice.texture_y / sheet.height() as f32])
                    .uv1([(slice.texture_x + slice.texture_w) / sheet.width() as f32,
                        (slice.texture_y + slice.texture_h) / sheet.height() as f32]);
            }
            else
            {
                return imgui::Image::new(imgui::TextureId::from(0), image_size);
            }
        }
        else
        {
            return imgui::Image::new(imgui::TextureId::from(usize::MAX), image_size)
                .uv0([0.0, 0.0]).uv1([0.0, 0.0]);
        }
    }

    return imgui::Image::new(imgui::TextureId::from(0), image_size);
}

pub fn tile_drawable_item(tileset : &DataHandle<Tileset>, tile_index : u32, tileset_store : &mut DataStore<Tileset>,
                             sheet_store : &mut DataStore<Sheet>, packages : &mut Packages) -> DrawableItem
{
    if tileset_store.package_loaded(tileset.package())
    {
        if let Some(tileset) = tileset_store.get_mut(tileset.resolve_if_loaded(tileset_store))
        {
            if sheet_store.package_loaded(tileset.sheet().package())
            {
                if let Ok(sheet_id) = sheet_store.get_id(tileset.sheet().package(), tileset.sheet().path())
                {
                    if let Some(sheet) = sheet_store.get(sheet_id)
                    {
                        if let Some(tile) = Tilemap::drawable_positioned_tile(tileset, sheet, 0.0, 0.0, tile_index)
                        {
                            let (tile_w, tile_h) = tileset.tile_size();

                            return DrawableItem::TileSprite(TileSprite
                            {
                                texture_x : tile.texture_x,
                                texture_y : tile.texture_y,
                                texture_w : tile.texture_w,
                                texture_h : tile.texture_h,
                                tile_w : tile_w as f32,
                                tile_h : tile_h as f32,
                                sheet_id,
                                shader_id : 0,
                                r : 1.0,
                                g : 1.0,
                                b : 1.0,
                                alpha : 1.0
                            });
                        }
                    }
                }
            }
            else
            {
                packages.load_package_later(ResourceType::Sheet, tileset.sheet().package());
            }
        }
    }
    else
    {
        packages.load_package_later(ResourceType::Tileset, tileset.package());
    }

    return DrawableItem::None;
}

pub fn tilemap_drawable_item(tilemap : &Tilemap, rect : BakedRect<f64>, tileset_store : &mut DataStore<Tileset>,
                        sheet_store : &mut DataStore<Sheet>, packages : &mut Packages) -> DrawableItem
{
    if tileset_store.package_loaded(tilemap.tileset().package())
    {
        if let Some(tileset) = tileset_store.get_mut(tilemap.tileset().resolve_if_loaded(tileset_store))
        {
            if sheet_store.package_loaded(tileset.sheet().package())
            {
                if let Ok(sheet_id) = sheet_store.get_id(tileset.sheet().package(), tileset.sheet().path())
                {
                    let mut sprite = MultiTileSprite::new();
                    sprite.sheet_id = sheet_id;

                    if let Some(sheet) = sheet_store.get(sheet_id)
                    {
                        tilemap.update_sprite_layout(tileset, sheet, &mut sprite, rect);

                        return DrawableItem::MultiTileSprite(sprite);
                    }
                }
            }
            else
            {
                packages.load_package_later(ResourceType::Sheet, tileset.sheet().package());
            }
        }
    }
    else
    {
        packages.load_package_later(ResourceType::Tileset, tilemap.tileset().package());
    }

    return DrawableItem::None;
}

#[derive(Debug)]
struct FloodFillInfo
{
    min_x : usize,
    max_x : usize,
    y : usize,
    extend_left : bool,
    extend_right : bool
}

// Modified from http://will.thimbleby.net/scanline-flood-fill/
pub fn flood_fill<A, T, S>(target : &mut toodee::TooDee<A>, x : usize, y : usize, diagonal : bool, mut test_pixel : T, mut set_pixel : S)
    where T : FnMut(&mut toodee::TooDee<A>, usize, usize) -> bool, S : FnMut(&mut toodee::TooDee<A>, usize, usize)
{
    let width = target.num_cols();
    let height = target.num_rows();
    let mut ranges = Vec::new();

    if !test_pixel(target, x, y)
    {
        return;
    }

    ranges.push(FloodFillInfo
    {
        min_x : x,
        max_x : x,
        y,
        extend_left : true,
        extend_right : true
    });

    while !ranges.is_empty()
    {
        let mut range = ranges.pop().unwrap();

        range.min_x = usize::min(usize::max(0, range.min_x), width - 1);
        range.max_x = usize::max(usize::min(usize::max(0, range.max_x), width - 1), range.min_x);

        for range_x in range.min_x..range.max_x + 1
        {
            set_pixel(target, range_x, range.y);
        }

        let mut extended_min_x = range.min_x;

        if range.extend_left
        {
            while extended_min_x > 0 && test_pixel(target, extended_min_x - 1, range.y)
            {
                extended_min_x -= 1;
                set_pixel(target, extended_min_x, range.y);
            }
        }

        let mut extended_max_x = range.max_x;

        if range.extend_right
        {
            while extended_max_x < width - 1 && test_pixel(target, extended_max_x + 1, range.y)
            {
                extended_max_x += 1;
                set_pixel(target, extended_max_x, range.y);
            }
        }

        if diagonal
        {
            if extended_min_x > 0 { extended_min_x -= 1; }
            if extended_max_x < width - 1 { extended_max_x += 1; }
        }

        let mut add_range = |new_y|
        {
            let mut range_active = false;
            let mut range_count : usize = 0;
            let mut range_start : usize = 0;

            for x in extended_min_x..extended_max_x + 1
            {
                if test_pixel(target, x, new_y)
                {
                    if !range_active
                    {
                        range_start = x;
                        range_active = true;
                    }
                }
                else
                {
                    if range_active
                    {
                        ranges.push(FloodFillInfo
                        {
                            min_x : range_start,
                            max_x : x - 1,
                            y : new_y,
                            extend_left : range_count == 0,
                            extend_right : false
                        });

                        range_active = false;
                        range_count += 1;
                    }
                }
            }

            if range_active
            {
                ranges.push(FloodFillInfo
                {
                    min_x : range_start,
                    max_x : extended_max_x,
                    y : new_y,
                    extend_left : range_count == 0,
                    extend_right : false
                });

                range_count += 1;
            }

            if range_count > 0
            {
                if let Some(range) = ranges.last_mut()
                {
                    range.extend_right = true;
                }
            }
        };

        if range.y > 0
        {
            add_range(range.y - 1);
        }
        if range.y < height - 1
        {
            add_range(range.y + 1);
        }
    }
}

struct TransparentPixel
{
    x : u32, y : u32, set : bool
}

pub fn defringe_image(texture : &mut RgbaImage, wrap : bool)
{
    let mut trans_pixels = Vec::new();
    let mut processed_trans = vec![false; (texture.width() * texture.height()) as usize];
    let mut processed_trans_buffer = vec![false; (texture.width() * texture.height()) as usize];

    for y in 0..texture.height()
    {
        for x in 0..texture.width()
        {
            let pixel = texture.get_pixel_mut(x, y);

            if pixel[3] == 0
            {
                trans_pixels.push(TransparentPixel
                {
                    x,
                    y,
                    set: false
                })
            }
        }
    }

    let mut colors = Vec::new();

    while trans_pixels.len() > 0
    {
        let mut processed = false;

        for entry in &mut trans_pixels
        {
            colors.clear();
            let (x, y) = (entry.x, entry.y);

            for (offset_x, offset_y) in &[(0, -1), (-1, 0), (1, 0), (0, 1)]
            {
                let check_x = x as i32 + offset_x;
                let check_y = y as i32 + offset_y;

                if wrap || (check_x >= 0 && check_x < texture.width() as i32
                    && check_y >= 0 && check_y < texture.height() as i32)
                {
                    let check_x_wrapped = check_x as u32 % texture.width();
                    let check_y_wrapped = check_y as u32 % texture.height();
                    let other_pixel = texture.get_pixel(check_x_wrapped, check_y_wrapped);

                    if other_pixel[3] > 0 || processed_trans[(check_y_wrapped * texture.width() + check_x_wrapped) as usize]
                    {
                        colors.push((other_pixel[0], other_pixel[1], other_pixel[2]));
                    }
                }
            }

            if colors.len() > 0
            {
                let mut red = 0.0;
                let mut green = 0.0;
                let mut blue = 0.0;

                for (other_r, other_g, other_b) in &colors
                {
                    red += *other_r as f32;
                    green += *other_g as f32;
                    blue += *other_b as f32;
                }

                red /= colors.len() as f32;
                green /= colors.len() as f32;
                blue /= colors.len() as f32;

                let pixel = texture.get_pixel_mut(x, y);

                pixel[0] = red as u8;
                pixel[1] = green as u8;
                pixel[2] = blue as u8;

                processed = true;
                entry.set = true;
                processed_trans_buffer[(y * texture.width() + x) as usize] = true;
            }
        }

        trans_pixels.retain(|entry| entry.set == false);
        processed_trans.copy_from_slice(&processed_trans_buffer[..]);

        // Don't get stuck if there are no opaque pixels
        if !processed
        {
            break;
        }
    }
}
