/// (ADVANCED) Vulkan rendering backend

use std::cell::RefCell;
use std::ffi::CString;
use std::rc::Rc;
use std::sync::Arc;

use crate::renderer::{DrawControl, DrawControlResult, DrawTransform, DrawSpriteInfo, DrawSpriteRawInfo, DrawBackgroundColorInfo, RendererNewInfo, RendererError, Sheet, Shader, ViewportMode};
use crate::util::BakedRect;
use crate::thirdparty::sendable::Sendable;

use keeshond_datapack::{DataId, DataStore, PreparedStore, PreparedStoreError, DataPreparer};

use vulkano::instance::{Instance, RawInstanceExtensions, PhysicalDevice, PhysicalDeviceType};
use vulkano::VulkanObject;
use vulkano::command_buffer::{AutoCommandBufferBuilder, DynamicState, SubpassContents};
use vulkano::device::{Device, DeviceExtensions, Features, QueuesIter, Queue};
use vulkano::swapchain::{Surface, Swapchain, CompositeAlpha, SurfaceTransform, PresentMode, FullscreenExclusive, ColorSpace, AcquireError, SwapchainAcquireFuture, SwapchainCreationError};
use vulkano::format::Format;
use vulkano::image::{ImageUsage, SwapchainImage};
use vulkano::sync::{GpuFuture, FlushError};
use vulkano::framebuffer::{RenderPassAbstract, FramebufferAbstract, Framebuffer, Subpass};
use vulkano::pipeline::viewport::{Viewport, Scissor};
use vulkano::pipeline::{GraphicsPipeline, GraphicsPipelineAbstract};
use vulkano::pipeline::vertex::OneVertexOneInstanceDefinition;
use vulkano::buffer::{CpuAccessibleBuffer, BufferUsage, CpuBufferPool, ImmutableBuffer};

const UNBORKED_IDENTITY_TRANSFORM : [f32; 9] = [2.0, 0.0, 0.0, 0.0, 2.0, 0.0, -1.0, -1.0, 1.0];
const INSTANCES_MAX : usize = 8192;

mod vs
{
    vulkano_shaders::shader!
    {
        ty : "vertex",
        src : "#version 450

layout(location = 0) in vec2 a_position;
layout(location = 1) in vec2 a_transform_col1;
layout(location = 2) in vec2 a_transform_col2;
layout(location = 3) in vec2 a_transform_col3;
layout(location = 4) in vec2 a_transform_offset;
layout(location = 5) in vec4 a_tex_coords;
layout(location = 6) in vec3 a_color;
layout(location = 7) in float a_alpha;

layout(location = 0) out vec2 v_uv;
layout(location = 1) out vec3 v_color;
layout(location = 2) out float v_alpha;

void main()
{
    mat3 instance_transform = mat3(vec3(a_transform_col1, 0.0), vec3(a_transform_col2, 0.0), vec3(a_transform_col3, 1.0));
    vec2 position = (instance_transform * vec3(a_position + a_transform_offset, 1.0)).xy;

    v_uv = vec2(a_position.x * a_tex_coords[2] + a_tex_coords[0], a_position.y * a_tex_coords[3] + a_tex_coords[1]);
    v_alpha = a_alpha;
    v_color = a_color;

    gl_Position = vec4(position, 0.0, 1.0);
}
"
    }
}

mod fs
{
    vulkano_shaders::shader!
    {
        ty : "fragment",
        src : "#version 450

layout(location = 0) in vec2 v_uv;
layout(location = 1) in vec3 v_color;
layout(location = 2) in float v_alpha;

layout(location = 0) out vec4 TargetScreen;

void main()
{
    vec4 color = vec4(v_color.r, v_color.g, v_color.b, v_alpha);
    TargetScreen = color;
}
"
    }
}

#[derive(Default, Debug, Clone)]
struct Vertex
{
    a_position : [f32; 2],
}

impl_vertex!(Vertex, a_position);

#[derive(Default, Debug, Clone)]
struct VertexInstance
{
    a_transform_col1 : [f32; 2],
    a_transform_col2 : [f32; 2],
    a_transform_col3 : [f32; 2],
    a_transform_offset : [f32; 2],
    a_tex_coords : [f32; 4],
    a_color : [f32; 3],
    a_alpha : f32
}

impl_vertex!(VertexInstance, a_transform_col1, a_transform_col2, a_transform_col3,
    a_transform_offset, a_tex_coords, a_color, a_alpha);

const VERTEX_QUAD : [Vertex; 4] =
[
    Vertex { a_position: [ 0.0, 0.0 ] },
    Vertex { a_position: [ 1.0, 0.0 ] },
    Vertex { a_position: [ 1.0, 1.0 ] },
    Vertex { a_position: [ 0.0, 1.0 ] }
];
const INDEX_QUAD : [u16; 6] = [ 0, 1, 2, 2, 3, 0 ];

type VertexDef = OneVertexOneInstanceDefinition::<Vertex, VertexInstance>;

#[derive(Debug, Copy, Clone, Eq, PartialEq)]
enum DrawOperation
{
    None,
    DrawSprite { sheet : DataId, shader : DataId }
}

fn vulkan_device_priority(device : &PhysicalDevice) -> u32
{
    match device.ty()
    {
        PhysicalDeviceType::DiscreteGpu => 0,
        PhysicalDeviceType::IntegratedGpu => 1,
        PhysicalDeviceType::VirtualGpu => 2,
        PhysicalDeviceType::Cpu => 3,
        PhysicalDeviceType::Other => 4
    }
}

fn get_vulkan_device(mut devices: Vec<PhysicalDevice>,
                     surface : Arc<Surface<Sdl2WindowContext>>) -> Result<(Arc<Device>, QueuesIter), RendererError>
{
    devices.sort_by(|a, b|
    {
        let priority_a = vulkan_device_priority(a);
        let priority_b = vulkan_device_priority(b);

        priority_a.cmp(&priority_b)
    });

    for device in devices.drain(..)
    {
        info!("Trying Vulkan device {}", device.index());

        match device.queue_families().find(|&family| family.supports_graphics() && surface.is_supported(family).unwrap_or(false))
        {
            Some(queue_family) =>
            {
                match Device::new(device, &Features::none(),
                                  &DeviceExtensions { khr_swapchain : true, .. DeviceExtensions::none() },
                                  [(queue_family, 0.5)].iter().cloned())
                {
                    Ok((device, queues)) =>
                    {
                        return Ok((device, queues));
                    },
                    Err(error) => error!("Device initialization failed: {}", error)
                }
            },
            None => info!("Not a graphical device")
        }
    }

    return Err(RendererError::RendererInitFailed(format!("No Vulkan hardware available")));
}

fn create_framebuffers(swap_images : &[Arc<SwapchainImage<Sdl2WindowContext>>],
                       render_pass : Arc<dyn RenderPassAbstract + Send + Sync>,
                       dynamic_state : &mut DynamicState) -> Result<Vec<Arc<dyn FramebufferAbstract + Send + Sync>>, RendererError>
{
    let mut framebuffers : Vec<Arc<dyn FramebufferAbstract + Send + Sync>> = Vec::new();
    let size = swap_images[0].dimensions();
    let viewport = Viewport
    {
        origin: [0.0, 0.0],
        dimensions: [size[0] as f32, size[1] as f32],
        depth_range: 0.0..1.0
    };

    dynamic_state.viewports = Some(vec!(viewport));

    for swap_image in swap_images
    {
        match Framebuffer::start(render_pass.clone()).add(swap_image.clone())
        {
            Ok(builder) =>
            {
                match builder.build()
                {
                    Ok(framebuffer) => framebuffers.push(Arc::new(framebuffer)),
                    Err(error) => return Err(RendererError::RendererInitFailed(format!("Could not build framebuffer: {}", error)))
                }
            },
            Err(error) => return Err(RendererError::RendererInitFailed(format!("Could not add framebuffer: {}", error)))
        }
    }

    Ok(framebuffers)
}

type Sdl2WindowContext = Sendable<Rc<sdl2::video::WindowContext>>;

pub struct VulkanDrawControl
{
    window : Rc<RefCell<sdl2::video::Window>>,
    instance : Arc<Instance>,
    device : Arc<Device>,
    queue : Arc<Queue>,
    surface : Arc<Surface<Sdl2WindowContext>>,
    swapchain : Arc<Swapchain<Sdl2WindowContext>>,
    swap_images : Vec<Arc<SwapchainImage<Sdl2WindowContext>>>,
    swap_index : usize,
    swapchain_dirty : bool,
    framebuffers : Vec<Arc<dyn FramebufferAbstract + Send + Sync>>,
    dynamic_state : DynamicState,
    acquire_future : Option<SwapchainAcquireFuture<Sdl2WindowContext>>,
    buffer_builder : Option<AutoCommandBufferBuilder>,
    last_frame : Box<dyn GpuFuture>,
    render_pass : Arc<dyn RenderPassAbstract + Send + Sync>,
    pipeline : Arc<dyn GraphicsPipelineAbstract + Send + Sync>,
    vertex_buffer : Arc<ImmutableBuffer<[Vertex]>>,
    index_buffer : Arc<ImmutableBuffer<[u16]>>,
    instance_buffer_src : Vec<VertexInstance>,
    instance_buffer : CpuBufferPool<VertexInstance>,
    draw_op : DrawOperation,
    can_present : bool,
    base_width : f32,
    base_height : f32,
    scale_offset_x : f32,
    scale_offset_y : f32,
    scale_width : f32,
    scale_height : f32,
    scissor : Scissor,
    transform : DrawTransform,
    viewport_mode : ViewportMode,
}

impl VulkanDrawControl
{
    fn check_draw_op(&mut self, op : DrawOperation)
    {
        if self.draw_op != op || self.instance_buffer_src.len() >= INSTANCES_MAX
        {
            self.draw_buffers();

            self.draw_op = op.clone();
        }
    }

    fn draw_buffers(&mut self)
    {
        if self.instance_buffer_src.is_empty()
        {
            return;
        }

        if let Some(builder) = &mut self.buffer_builder
        {
            match self.draw_op
            {
                DrawOperation::None => (),
                DrawOperation::DrawSprite{ sheet, shader } =>
                {
                    let sub_buffer = self.instance_buffer.chunk(self.instance_buffer_src.clone()).expect("Failed to allocate buffer chunk");
                    let vertex_slice = self.vertex_buffer.clone();

                    builder.draw_indexed(self.pipeline.clone(), &self.dynamic_state,
                                         vec![vertex_slice, Arc::new(sub_buffer)],
                                         self.index_buffer.clone(), (), ()).expect("Failed to draw buffer");
                }
            }
        }

        self.instance_buffer_src.clear();
        self.draw_op = DrawOperation::None;
    }
}

impl DrawControl for VulkanDrawControl
{
    fn new(renderer_new_info : &mut RendererNewInfo) -> DrawControlResult where Self : Sized
    {
        warn!("The Vulkan renderer is UNFINISHED!");

        let video_subsystem = &renderer_new_info.video_subsystem;
        let mut window_builder = video_subsystem.window("", renderer_new_info.width, renderer_new_info.height);

        window_builder.hidden().position_centered().resizable().allow_highdpi().vulkan();

        let window_raw = try_or_else!(window_builder.build(),
            |error| Err(RendererError::RendererInitFailed(format!("Could not create game window: {}", error))));
        let instance_extensions = try_or_else!(window_raw.vulkan_instance_extensions(),
            |error| Err(RendererError::RendererInitFailed(format!("Could not read Vulkan instance extensions: {}", error))));
        let instance_extensions_raw = RawInstanceExtensions::new(instance_extensions.iter().map(
            |&extension| CString::new(&extension[0..extension.find("\0").unwrap_or(extension.len())]).unwrap()
        ));

        let mut extension_list = String::new();

        for extension in instance_extensions
        {
            extension_list.push(' ');
            extension_list.push_str(&extension);
        }

        info!("Vulkan instance extensions:{}", extension_list);

        let instance = try_or_else!(Instance::new(None, instance_extensions_raw, None),
            |error| Err(RendererError::RendererInitFailed(format!("Could not create Vulkan instance: {}", error))));

        let surface_raw = try_or_else!(window_raw.vulkan_create_surface(instance.internal_object()),
            |error| Err(RendererError::RendererInitFailed(format!("Could not create Vulkan surface: {}", error))));

        let surface = unsafe
        {
            Arc::new(Surface::from_raw_surface(instance.clone(), surface_raw, Sendable::new(window_raw.context())))
        };

        let devices : Vec<PhysicalDevice> = PhysicalDevice::enumerate(&instance).collect();

        for device in devices.iter()
        {
            info!("Vulkan device {}: {}", device.index(), device.name());
            info!("    UUID: {}", hex::encode(device.uuid()));
            info!("    {:?}, Driver version 0x{:x?}, Vulkan version {}", device.ty(), device.driver_version(), device.api_version().to_string());
        }

        let (device, mut queue_iter) = get_vulkan_device(devices, surface.clone())?;

        let queue = try_opt_or_else!(queue_iter.next(),
            || Err(RendererError::RendererInitFailed(format!("No graphical queue available"))));

        info!("Using Vulkan device {}", device.physical_device().index());

        let caps = try_or_else!(surface.capabilities(device.physical_device()),
            |error| Err(RendererError::RendererInitFailed(format!("Could not get Vulkan surface capabilities: {}", error))));

        let (swapchain, swap_images) = try_or_else!(Swapchain::new(device.clone(), surface.clone(), caps.min_image_count,
            Format::B8G8R8A8Srgb, [renderer_new_info.width, renderer_new_info.height], 1, ImageUsage::color_attachment(), &queue,
            SurfaceTransform::Identity, CompositeAlpha::Opaque, PresentMode::Fifo, FullscreenExclusive::Default,
            true, ColorSpace::SrgbNonLinear),
            |error| Err(RendererError::RendererInitFailed(format!("Could not create Vulkan swapchain: {}", error))));

        let window = Rc::new(RefCell::new(window_raw));

        let last_frame = vulkano::sync::now(device.clone()).boxed();

        let render_pass = Arc::new(try_or_else!(vulkano::single_pass_renderpass!(device.clone(),
            attachments :
            {
                color :
                {
                    load : Clear,
                    store : Store,
                    format : swapchain.format(),
                    samples : 1,
                }
            },
            pass:
            {
                color : [color],
                depth_stencil : {}
            }),
            |error| Err(RendererError::RendererInitFailed(format!("Could not create Vulkan swapchain: {}", error)))));

        let mut dynamic_state = DynamicState::default();
        let framebuffers = create_framebuffers(&swap_images, render_pass.clone(), &mut dynamic_state)?;

        let vs = try_or_else!(vs::Shader::load(device.clone()),
            |error| Err(RendererError::RendererInitFailed(format!("Could not load vertex shader: {}", error))));
        let fs = try_or_else!(fs::Shader::load(device.clone()),
            |error| Err(RendererError::RendererInitFailed(format!("Could not load fragment shader: {}", error))));

        let (vertex_buffer, vertex_future) = try_or_else!(ImmutableBuffer::from_iter(VERTEX_QUAD.iter().cloned(), BufferUsage::vertex_buffer(), queue.clone()),
            |error| Err(RendererError::RendererInitFailed(format!("Could not create vertex buffer: {}", error))));
        let (index_buffer, index_future) = try_or_else!(ImmutableBuffer::from_iter(INDEX_QUAD.iter().cloned(), BufferUsage::index_buffer(), queue.clone()),
            |error| Err(RendererError::RendererInitFailed(format!("Could not create index buffer: {}", error))));

        try_or_else!(vertex_future.flush(),
            |error| Err(RendererError::RendererInitFailed(format!("Could not flush buffer: {}", error))));
        try_or_else!(index_future.flush(),
            |error| Err(RendererError::RendererInitFailed(format!("Could not flush buffer: {}", error))));

        let instance_buffer = CpuBufferPool::vertex_buffer(device.clone());

        let pipeline = Arc::new(
            try_or_else!(GraphicsPipeline::start()
                .vertex_input(VertexDef::new())
                .vertex_shader(vs.main_entry_point(), ())
                .triangle_list()
                .viewports_scissors_dynamic(1)
                .fragment_shader(fs.main_entry_point(), ())
                .blend_alpha_blending()
                .render_pass(try_opt_or_else!(Subpass::from(render_pass.clone(), 0),
                    || Err(RendererError::RendererInitFailed(format!("Could not init render pass")))))
                .build(device.clone()),
            |error| Err(RendererError::RendererInitFailed(format!("Could not create pipeline: {}", error))))
        );

        Ok((Box::new(VulkanDrawControl
        {
            window : window.clone(),
            instance,
            device,
            queue,
            surface,
            swapchain,
            swap_images,
            swap_index : 0,
            swapchain_dirty : false,
            framebuffers,
            dynamic_state,
            acquire_future : None,
            buffer_builder : None,
            last_frame,
            render_pass,
            pipeline,
            vertex_buffer,
            index_buffer,
            instance_buffer,
            instance_buffer_src : Vec::new(),
            draw_op : DrawOperation::None,
            can_present : false,
            base_width : renderer_new_info.width as f32,
            base_height : renderer_new_info.height as f32,
            scale_offset_x : 0.0,
            scale_offset_y : 0.0,
            scale_width : 0.0,
            scale_height : 0.0,
            scissor : Scissor::irrelevant(),
            transform : DrawTransform::identity(),
            viewport_mode : renderer_new_info.viewport_mode,
        }), window))
    }

    fn base_rect(&self) -> BakedRect<f32>
    {
        BakedRect
        {
            x1: 0.0,
            y1: 0.0,
            x2: self.base_width,
            y2: self.base_height
        }
    }

    fn screen_transform(&self) -> &DrawTransform
    {
        &self.transform
    }
    
    #[allow(unused_variables)]
    fn sync_sheet_store(&mut self, sheet_store : &mut DataStore<Sheet>) -> Result<(), PreparedStoreError>
    {
        Ok(())
    }
    
    #[allow(unused_variables)]
    fn sync_shader_store(&mut self, shader_store : &mut DataStore<Shader>) -> Result<(), PreparedStoreError>
    {
        Ok(())
    }
    
    #[allow(unused_variables)]
    fn draw_sprite(&mut self, draw_info : &DrawSpriteInfo)
    {
        let mut transform = draw_info.transform.clone();
        let mut transform_offset_x = -0.5;
        let mut transform_offset_y = -0.5;
        let mut texture_x = 0.0;
        let mut texture_y = 0.0;
        let mut texture_w = 1.0;
        let mut texture_h = 1.0;
        let mut sheet_id = 0;

        transform.translate(draw_info.x, draw_info.y);
        if draw_info.angle != 0.0
        {
            transform.rotate(draw_info.angle);
        }

        let mut got_slice = false;

        // FIXME
        /*if let Some(gpu_sheet_info) = self.gpu_sheet_store.get(draw_info.sheet_id)
        {
            if draw_info.slice < gpu_sheet_info.slices.len()
            {
                let slice = &gpu_sheet_info.slices[draw_info.slice];

                transform.scale(draw_info.scale_x * slice.scale_x, draw_info.scale_y * slice.scale_y);

                sheet_id = draw_info.sheet_id;
                transform_offset_x = slice.offset_x;
                transform_offset_y = slice.offset_y;
                texture_x = slice.texture_x;
                texture_y = slice.texture_y;
                texture_w = slice.texture_w;
                texture_h = slice.texture_h;

                got_slice = true;
            }
        }*/

        if !got_slice
        {
            transform.scale(draw_info.scale_x * 32.0, draw_info.scale_y * 32.0);
        }

        self.draw_sprite_raw(&DrawSpriteRawInfo
        {
            sheet_id,
            shader_id: draw_info.shader_id,
            transform: transform,
            transform_offset_x,
            transform_offset_y,
            texture_x,
            texture_y,
            texture_w,
            texture_h,
            r: draw_info.r,
            g: draw_info.g,
            b: draw_info.b,
            alpha: draw_info.alpha
        });
    }

    #[allow(unused_variables)]
    fn draw_sprite_raw(&mut self, draw_info: &DrawSpriteRawInfo)
    {
        self.check_draw_op(DrawOperation::DrawSprite { sheet : draw_info.sheet_id, shader : draw_info.shader_id });

        let transform = &draw_info.transform;
        let instance = VertexInstance
        {
            a_transform_col1 : [ transform.mat.x.x, transform.mat.x.y ],
            a_transform_col2 : [ transform.mat.y.x, transform.mat.y.y ],
            a_transform_col3 : [ transform.mat.z.x, transform.mat.z.y ],
            a_transform_offset : [ draw_info.transform_offset_x, draw_info.transform_offset_y ],
            a_tex_coords : [ draw_info.texture_x, draw_info.texture_y, draw_info.texture_w, draw_info.texture_h ],
            a_color : [ draw_info.r, draw_info.g, draw_info.b ],
            a_alpha : draw_info.alpha
        };

        self.instance_buffer_src.push(instance);
    }

    #[allow(unused_variables)]
    fn draw_background_color(&mut self, draw_info : &DrawBackgroundColorInfo)
    {
        warn!("Unimplemented");
    }

    #[cfg(feature = "imgui_base")]
    #[allow(unused_variables)]
    fn draw_imgui(&mut self, ui : imgui::Ui)
    {
        warn!("Unimplemented");
    }

    fn recalculate_viewport(&mut self, base_width : f32, base_height : f32)
    {
        self.base_width = base_width;
        self.base_height = base_height;

        let (width, height) = self.window.borrow().size();
        let info = crate::renderer::recalculate_viewport_common(base_width, base_height, width, height, self.viewport_mode);

        self.scale_width = info.scale_width;
        self.scale_height = info.scale_height;
        self.scale_offset_x = info.scale_offset_x;
        self.scale_offset_y = info.scale_offset_y;

        self.scissor = match info.scissor
        {
            Some(scissor) =>
            {
                Scissor
                {
                    origin: [scissor.x as i32, scissor.y as i32],
                    dimensions: [scissor.w, scissor.h]
                }
            },
            None => Scissor::irrelevant()
        }
    }

    fn start_drawing(&mut self)
    {
        self.last_frame.cleanup_finished();

        if self.swapchain_dirty
        {
            let (width, height) = self.window.borrow().vulkan_drawable_size();

            match self.swapchain.recreate_with_dimensions([width, height])
            {
                Ok((swapchain, swap_images)) =>
                {
                    self.swapchain = swapchain;
                    self.swap_images = swap_images;
                    self.swapchain_dirty = false;

                    self.framebuffers = create_framebuffers(&self.swap_images, self.render_pass.clone(), &mut self.dynamic_state)
                        .expect("Could not recreate framebuffer");
                },
                Err(SwapchainCreationError::UnsupportedDimensions) => return,
                Err(error) => panic!("Could not create Vulkan swapchain: {}", error),
            }
        }

        self.dynamic_state.scissors = Some(vec!(self.scissor));

        match vulkano::swapchain::acquire_next_image(self.swapchain.clone(), None)
        {
            Ok((swap_index, suboptimal, acquire_future)) =>
            {
                if suboptimal
                {
                    self.swapchain_dirty = true;
                }

                self.swap_index = swap_index;
                self.acquire_future = Some(acquire_future);
            },
            Err(AcquireError::OutOfDate) =>
            {
                self.swapchain_dirty = true;
                return;
            },
            Err(error) => panic!("Failed to acquire swapchain image: {}", error)
        }

        let mut buffer_builder = AutoCommandBufferBuilder::primary_one_time_submit(self.device.clone(), self.queue.family())
            .expect("Could not create command buffer");

        buffer_builder.begin_render_pass(self.framebuffers[self.swap_index].clone(), SubpassContents::Inline, vec![[0.0, 0.0, 0.0, 1.0].into()])
            .expect("Could not clear framebuffer");

        self.buffer_builder = Some(buffer_builder);

        let mut view_transform = DrawTransform::from_array(&UNBORKED_IDENTITY_TRANSFORM);

        match self.viewport_mode
        {
            ViewportMode::Independent | ViewportMode::OneToOne =>
            {
                view_transform.scale(1.0 / self.scale_width, 1.0 / self.scale_height);
                view_transform.translate(self.scale_offset_x, self.scale_offset_y);
            },
            ViewportMode::Pixel =>
            {
                unimplemented!();
            }
        }

        self.transform = view_transform.clone();
    }

    fn end_drawing(&mut self)
    {
        self.draw_buffers();

        if self.swapchain_dirty
        {
            return;
        }

        if let Some(builder) = &mut self.buffer_builder
        {
            builder.end_render_pass().expect("Could not finish render pass");
        }
        else
        {
            panic!("Command buffer builder not present!");
        }

        if self.acquire_future.is_none()
        {
            panic!("Acquire future not present!");
        }

        let mut acquire_future = None;
        let mut last_frame = vulkano::sync::now(self.device.clone()).boxed();
        let mut buffer_builder = None;

        std::mem::swap(&mut self.acquire_future, &mut acquire_future);
        std::mem::swap(&mut self.last_frame, &mut last_frame);
        std::mem::swap(&mut self.buffer_builder, &mut buffer_builder);

        self.last_frame = last_frame.join(acquire_future.unwrap())
            .then_execute(self.queue.clone(), buffer_builder.unwrap().build().expect("Command buffer build failed"))
            .expect("Queue execution failed").boxed();

        self.can_present = true;
    }

    fn present(&mut self)
    {
        if !self.can_present || self.swapchain_dirty
        {
            return;
        }

        let mut last_frame = vulkano::sync::now(self.device.clone()).boxed();

        std::mem::swap(&mut self.last_frame, &mut last_frame);

        match last_frame.then_swapchain_present(self.queue.clone(), self.swapchain.clone(), self.swap_index)
            .then_signal_fence_and_flush()
        {
            Ok(future) =>
            {
                self.last_frame = future.boxed();
            }
            Err(FlushError::OutOfDate) =>
            {
                self.swapchain_dirty = true;
                self.last_frame = vulkano::sync::now(self.device.clone()).boxed();
            }
            Err(error) =>
            {
                warn!("Swap flush failed: {}", error);
                self.last_frame = vulkano::sync::now(self.device.clone()).boxed();
            }
        }

        self.can_present = false;
    }

    #[cfg(feature = "imgui_base")]
    #[allow(unused_variables)]
    fn init_imgui_renderer(&mut self, imgui : &mut imgui::Context) -> Result<(), RendererError>
    {
        Ok(())
    }

    #[cfg(feature = "imgui_base")]
    #[allow(unused_variables)]
    fn update_imgui_renderer(&mut self, imgui : &mut imgui::Context)
    {

    }
}