
use core::iter::Iterator;
use std::collections::{HashMap, BTreeMap};

use serde::{Deserialize, Deserializer, Serialize, Serializer};
use std::ops::{Add, Sub, AddAssign, SubAssign};
use std::fmt::Debug;

#[derive(Debug, Clone, Copy, Default, Eq, PartialEq, Serialize, Deserialize)]
pub struct Rect<T : Copy + Add<Output = T> + AddAssign + SubAssign + PartialOrd>
{
    pub x : T,
    pub y : T,
    pub w : T,
    pub h : T
}

impl<T : Debug + Copy + Add<Output = T> + AddAssign + SubAssign + PartialOrd> Rect<T>
{
    #[inline(always)]
    pub fn x2(&self) -> T
    {
        self.x + self.w
    }

    #[inline(always)]
    pub fn y2(&self) -> T
    {
        self.y + self.h
    }

    pub fn translate(&mut self, x : T, y : T)
    {
        self.x += x;
        self.y += y;
    }

    pub fn extrude(&mut self, amount : T)
    {
        self.x -= amount;
        self.y -= amount;
        self.w += amount + amount;
        self.h += amount + amount;
    }

    pub fn contains(&self, x : T, y : T) -> bool
    {
        x >= self.x && x < self.x + self.w && y >= self.y && y < self.y + self.h
    }
}

#[derive(Debug, Clone, Copy, Default, Eq, PartialEq, Serialize, Deserialize)]
pub struct BakedRect<T : Copy + Sub<Output = T> + AddAssign + SubAssign + PartialOrd>
{
    pub x1 : T,
    pub y1 : T,
    pub x2 : T,
    pub y2 : T
}

impl<T : Debug + Copy + Sub<Output = T> + AddAssign + SubAssign + PartialOrd> BakedRect<T>
{
    #[inline(always)]
    pub fn w(&self) -> T
    {
        self.x2 - self.x1
    }

    #[inline(always)]
    pub fn h(&self) -> T
    {
        self.y2 - self.y1
    }

    pub fn translate(&mut self, x : T, y : T)
    {
        self.x1 += x;
        self.y1 += y;
        self.x2 += x;
        self.y2 += y;
    }

    pub fn extrude(&mut self, amount : T)
    {
        self.x1 -= amount;
        self.y1 -= amount;
        self.x2 += amount;
        self.y2 += amount;
    }

    pub fn combine(&mut self, other : BakedRect<T>)
    {
        if other.x1 < self.x1 { self.x1 = other.x1; }
        if other.y1 < self.y1 { self.y1 = other.y1; }
        if other.x2 > self.x2 { self.x2 = other.x2; }
        if other.y2 > self.y2 { self.y2 = other.y2; }
    }

    pub fn make_positive(&mut self)
    {
        if self.x2 < self.x1
        {
            std::mem::swap(&mut self.x1, &mut self.x2);
        }
        if self.y2 < self.y1
        {
            std::mem::swap(&mut self.y1, &mut self.y2);
        }
    }

    pub fn contains(&self, x : T, y : T) -> bool
    {
        x >= self.x1 && x < self.x2 && y >= self.y1 && y < self.y2
    }

    pub fn overlaps(&self, other : &BakedRect<T>) -> bool
    {
        let result = self.x1 < other.x2 && self.x2 >= other.x1 && self.y1 < other.y2 && self.y2 >= other.y1;

        result
    }
}

fn sort_map<S, T : Serialize + Ord, U : Serialize>(map : &HashMap<T, U>, serializer : S) -> Result<S::Ok, S::Error> where S : Serializer
{
    let sorted : BTreeMap<_, _> = map.iter().collect();

    sorted.serialize(serializer)
}

#[derive(Clone, Default, Serialize, Debug)]
pub struct NamedItemStore<T>
{
    #[serde(rename = "i")]
    #[serde(alias = "items")]
    items : Vec<T>,
    #[serde(serialize_with = "sort_map")]
    #[serde(rename = "n")]
    #[serde(alias = "names")]
    names : HashMap<String, usize>,
    #[serde(skip_serializing)]
    ids : HashMap<usize, String>
}

impl<T> NamedItemStore<T>
{
    pub fn new() -> NamedItemStore<T>
    {
        NamedItemStore::<T>
        {
            items: Vec::new(),
            names: HashMap::new(),
            ids: HashMap::new()
        }
    }

    pub fn get(&self, id : usize) -> Option<&T>
    {
        self.items.get(id)
    }

    pub fn get_mut(&mut self, id : usize) -> Option<&mut T>
    {
        self.items.get_mut(id)
    }

    pub fn set(&mut self, id : usize, item : T) -> bool
    {
        if let Some(old_item) = self.items.get_mut(id)
        {
            *old_item = item;

            return true;
        }

        false
    }

    pub fn get_name(&self, id : usize) -> Option<String>
    {
        self.ids.get(&id).cloned()
    }

    pub fn get_id(&self, name : &str) -> Option<usize>
    {
        self.names.get(name).cloned()
    }

    pub fn insert_item(&mut self, pos : usize, item : T) -> bool
    {
        if pos > self.items.len()
        {
            return false;
        }

        for pos in (pos..self.items.len()).rev()
        {
            if let Some(moved_name) = self.ids.remove(&pos)
            {
                self.names.insert(moved_name.clone(), pos + 1);
                self.ids.insert(pos + 1, moved_name);
            }
        }

        self.items.insert(pos, item);

        true
    }

    pub fn remove_item(&mut self, id : usize) -> Option<T>
    {
        if id >= self.items.len()
        {
            return None;
        }

        self.rename_item(id, "");

        for pos in (id + 1)..self.items.len()
        {
            if let Some(moved_name) = self.ids.remove(&pos)
            {
                self.names.insert(moved_name.clone(), pos - 1);
                self.ids.insert(pos - 1, moved_name);
            }
        }

        Some(self.items.remove(id))
    }

    pub fn rename_item(&mut self, id : usize, new_name : &str) -> bool
    {
        if !new_name.is_empty() && self.names.contains_key(new_name)
        {
            return false;
        }

        if let Some(old_name) = self.ids.get(&id).cloned()
        {
            let mut bad = false;

            if self.ids.remove(&id).is_none() { bad = true; }
            if self.names.remove(&old_name).is_none() { bad = true; }

            if bad
            {
                warn!("Inconsistent item name storage: {} {}", id, old_name);
                return false;
            }
        }

        if !new_name.is_empty()
        {
            self.ids.insert(id, String::from(new_name));
            self.names.insert(String::from(new_name), id);
        }

        true
    }

    pub fn len(&self) -> usize
    {
        self.items.len()
    }
}

impl<'a, T> IntoIterator for &'a NamedItemStore<T>
{
    type Item = &'a T;
    type IntoIter = std::slice::Iter<'a, T>;

    fn into_iter(self) -> Self::IntoIter
    {
        self.items.iter()
    }
}

#[derive(Deserialize, Debug)]
struct NamedItemStoreRaw<T>
{
    #[serde(rename = "i")]
    #[serde(alias = "items")]
    items : Vec<T>,
    #[serde(rename = "n")]
    #[serde(alias = "names")]
    names : HashMap<String, usize>
}

impl<'de, T : Deserialize<'de>> Deserialize<'de> for NamedItemStore<T>
{
    fn deserialize<D>(deserializer: D) -> Result<Self, <D as Deserializer<'de>>::Error>
        where D : Deserializer<'de>
    {
        let store_raw = NamedItemStoreRaw::<T>::deserialize(deserializer)?;

        let mut ids = HashMap::new();

        for (name, id) in &store_raw.names
        {
            ids.insert(*id, name.clone());
        }

        Ok(NamedItemStore::<T>
        {
            items : store_raw.items,
            names : store_raw.names,
            ids
        })
    }
}
