//! (ADVANCED) OpenGL 3 rendering backend

use std::io::Cursor;
use std::rc::Rc;
use std::cell::RefCell;

use glium::{Frame, Surface, Program, ProgramCreationError, IndexBuffer, VertexBuffer};
use glium::texture::{RawImage2d, Texture2d, UncompressedFloatFormat};
use glium::uniforms::{Sampler, MagnifySamplerFilter, MinifySamplerFilter, SamplerWrapFunction};
use glium::program::ProgramCreationInput;
use glium::backend::Facade;
use crate::thirdparty::glium_sdl2::{DisplayBuild, SDL2Facade};

use keeshond_datapack::{DataId, DataStore, PreparedStore, PreparedStoreError, DataPreparer};

use crate::renderer::{DrawControl, DrawControlResult, DrawTransform, DrawSpriteRawInfo, DrawBackgroundColorInfo, RendererNewInfo, RendererError, Sheet, Shader, ViewportMode, DrawSpriteInfo, Color};

#[cfg(feature = "imgui_opengl")] use imgui_glium_renderer;
#[cfg(feature = "imgui_opengl")] use glium::uniforms::SamplerBehavior;
use crate::util::BakedRect;

const INSTANCES_MAX : usize = 8192;
const INSTANCE_BUFFER_MAX: usize = 16;

const UNBORKED_IDENTITY_TRANSFORM : [f32; 9] = [2.0, 0.0, 0.0, 0.0, -2.0, 0.0, -1.0, 1.0, 1.0];
const UNBORKED_UPSCALE_TRANSFORM : [f32; 9] = [2.0, 0.0, 0.0, 0.0, 2.0, 0.0, -1.0, -1.0, 1.0];

#[derive(Copy, Clone, PartialEq)]
enum RendererOp
{
    #[allow(dead_code)]
    BackgroundColor,
    #[allow(dead_code)]
    Sprite
}

impl Default for RendererOp
{
    fn default() -> Self
    {
        RendererOp::BackgroundColor
    }
}

#[derive(Copy, Clone)]
struct Vertex
{
    a_position : [f32; 2],
}

implement_vertex!(Vertex, a_position);

#[derive(Copy, Clone, Default)]
struct Instance
{
    a_transform_col1 : [f32; 2],
    a_transform_col2 : [f32; 2],
    a_transform_col3 : [f32; 2],
    a_transform_offset : [f32; 2],
    a_tex_coords : [f32; 4],
    a_color : [f32; 3],
    a_alpha : f32
}

implement_vertex!(Instance, a_transform_col1, a_transform_col2, a_transform_col3,
    a_transform_offset, a_tex_coords, a_color, a_alpha);

const VERTEX_QUAD : [Vertex; 4] =
[
    Vertex { a_position: [ 0.0, 0.0 ] },
    Vertex { a_position: [ 1.0, 0.0 ] },
    Vertex { a_position: [ 1.0, 1.0 ] },
    Vertex { a_position: [ 0.0, 1.0 ] }
];
const INDEX_QUAD : [u16; 6] = [ 0, 1, 2, 2, 3, 0 ];

struct GpuSpriteSlice
{
    pub texture_x : f32,
    pub texture_y : f32,
    pub texture_w : f32,
    pub texture_h : f32,
    pub scale_x : f32,
    pub scale_y : f32,
    pub offset_x : f32,
    pub offset_y : f32,
}

struct GpuSheetHandle
{
    texture : Rc<Texture2d>,
    mag_filter : MagnifySamplerFilter,
    min_filter : MinifySamplerFilter,
    slices : Vec<GpuSpriteSlice>
}

struct GpuSheetPreparer
{
    display : Rc<RefCell<SDL2Facade>>,
    #[allow(dead_code)]
    default_texture : Rc<Texture2d>,
    texture_filtering : bool,
    #[cfg(feature = "imgui_opengl")]
    imgui_renderer : Rc<RefCell<Option<imgui_glium_renderer::Renderer>>>
}

impl GpuSheetPreparer
{
    fn texture_raw_from_data(data : &Vec<u8>, palette : &Option<Vec<Color>>) -> Vec<u8>
    {
        if let Some(palette) = palette
        {
            let mut expanded = Vec::with_capacity(data.len() * 4);

            for i in 0..data.len()
            {
                let color = data[i] as usize;

                if color < palette.len()
                {
                    expanded.push(palette[color].r);
                    expanded.push(palette[color].g);
                    expanded.push(palette[color].b);
                    expanded.push(palette[color].a);
                }
                else
                {
                    for _ in 0..4
                    {
                        expanded.push(0);
                    }
                }
            }

            return expanded;
        }

        data.clone()
    }

    fn make_sheet_handle(&mut self, data: &mut Sheet) -> GpuSheetHandle
    {
        let texture;

        {
            let display_ref: &SDL2Facade = &self.display.borrow();
            let image_raw = GpuSheetPreparer::texture_raw_from_data(&data.image_raw, &data.palette);

            let image = RawImage2d::from_raw_rgba(image_raw, (data.width, data.height));
            texture = Rc::new(Texture2d::new(display_ref, image).expect("Texture upload failed"));
        }

        self.make_sheet_handle_from_texture(data, texture)
    }

    fn make_sheet_handle_from_texture(&mut self, data: &mut Sheet, texture : Rc<Texture2d>) -> GpuSheetHandle
    {
        let (mag_filter, min_filter) = self.sample_filters(data);

        let mut slices = Vec::new();

        for slice in &data.metadata.slices
        {
            let texture_w = slice.texture_w / data.width as f32;
            let texture_h = slice.texture_h / data.height as f32;

            slices.push(GpuSpriteSlice
            {
                texture_x: slice.texture_x / data.width as f32,
                texture_y: slice.texture_y / data.height as f32,
                texture_w,
                texture_h,
                scale_x: slice.texture_w,
                scale_y: slice.texture_h,
                offset_x: -slice.origin_x / slice.texture_w,
                offset_y: -slice.origin_y / slice.texture_h,
            })
        }

        GpuSheetHandle
        {
            texture,
            mag_filter,
            min_filter,
            slices
        }
    }

    #[cfg(feature = "imgui_opengl")]
    fn update_imgui_texture(&mut self, id : DataId, handle : &mut GpuSheetHandle)
    {
        let mut imgui_borrow = self.imgui_renderer.borrow_mut();

        if let Some(imgui_renderer) = imgui_borrow.as_mut()
        {
            imgui_renderer.textures().replace(imgui::TextureId::from(id), imgui_glium_renderer::Texture
            {
                texture : handle.texture.clone(),
                sampler : SamplerBehavior
                {
                    wrap_function: (SamplerWrapFunction::Repeat, SamplerWrapFunction::Repeat, SamplerWrapFunction::Repeat),
                    minify_filter: handle.min_filter,
                    magnify_filter: handle.mag_filter,
                    .. Default::default()
                }
            });
        }
    }

    fn sample_filters(&mut self, data : &mut Sheet) -> (MagnifySamplerFilter, MinifySamplerFilter)
    {
        let filtered;

        if let Some(filter) = data.metadata.texture_filtering
        {
            filtered = filter;
        }
        else
        {
            filtered = self.texture_filtering;
        }

        let mag_filter = match filtered
        {
            true => MagnifySamplerFilter::Linear,
            false => MagnifySamplerFilter::Nearest
        };
        let min_filter = match filtered
        {
            true => MinifySamplerFilter::Linear,
            false => MinifySamplerFilter::Nearest
        };

        (mag_filter, min_filter)
    }
}

impl DataPreparer<Sheet, GpuSheetHandle> for GpuSheetPreparer
{
    #[allow(unused_variables)]
    fn prepare(&mut self, data : &mut Sheet, id : DataId) -> GpuSheetHandle
    {
        #[allow(unused_mut)]
        let mut handle = self.make_sheet_handle(data);

        data.texture_dirty = false;

        #[cfg(feature = "imgui_opengl")]
        self.update_imgui_texture(id, &mut handle);

        handle
    }

    #[allow(unused_variables)]
    fn unprepare(&mut self, _prepared : &mut GpuSheetHandle, id : DataId)
    {
        #[cfg(feature = "imgui_opengl")]
        {
            let mut imgui_borrow = self.imgui_renderer.borrow_mut();

            if let Some(imgui_renderer) = imgui_borrow.as_mut()
            {
                imgui_renderer.textures().replace(imgui::TextureId::from(id), imgui_glium_renderer::Texture
                {
                    texture : self.default_texture.clone(),
                    sampler : SamplerBehavior
                    {
                        wrap_function: (SamplerWrapFunction::Repeat, SamplerWrapFunction::Repeat, SamplerWrapFunction::Repeat),
                        minify_filter: MinifySamplerFilter::Linear,
                        magnify_filter: MagnifySamplerFilter::Linear,
                        .. Default::default()
                    }
                });
            }
        }
    }

    #[allow(unused_variables)]
    fn reprepare(&mut self, data : &mut Sheet, prepared : &mut GpuSheetHandle, id : DataId)
    {
        if data.texture_dirty
        {
            *prepared = self.make_sheet_handle(data);
        }
        else
        {
            *prepared = self.make_sheet_handle_from_texture(data, prepared.texture.clone());
        }

        #[cfg(feature = "imgui_opengl")]
        self.update_imgui_texture(id, prepared);

        data.texture_dirty = false;
    }
}


struct GpuShaderHandle
{
    program : Program
}

struct GpuShaderPreparer
{
    display : Rc<RefCell<SDL2Facade>>
}

impl DataPreparer<Shader, GpuShaderHandle> for GpuShaderPreparer
{
    fn prepare(&mut self, data : &mut Shader, _id : DataId) -> GpuShaderHandle
    {
        let program = make_program(&self.display.borrow(),
            include_str!("shader/vert_standard_140.glsl"),
            &data.program_source).expect("Shader compilation failed");
        
        GpuShaderHandle
        {
            program
        }
    }
    fn unprepare(&mut self, _prepared : &mut GpuShaderHandle, _id : DataId)
    {
        
    }
}

#[derive(Copy, Clone, Default)]
struct InstanceBufferParam
{
    op : RendererOp,
    sheet : DataId,
    shader : DataId
}

fn make_program(display : &SDL2Facade, vertex_shader : &str, fragment_shader : &str) -> Result<Program, ProgramCreationError>
{
    let source = ProgramCreationInput::SourceCode
    {
        vertex_shader,
        fragment_shader,
        geometry_shader : None,
        tessellation_control_shader : None,
        tessellation_evaluation_shader : None,
        transform_feedback_varyings : None,
        uses_point_size : false,
        outputs_srgb : true
    };
    
    glium::Program::new(display, source)
}

fn upscale_instance_from_transform(transform : &DrawTransform) -> Instance
{
    Instance
    {
        a_transform_col1 : [ transform.mat.x.x, transform.mat.x.y ],
        a_transform_col2 : [ transform.mat.y.x, transform.mat.y.y ],
        a_transform_col3 : [ transform.mat.z.x, transform.mat.z.y ],
        a_transform_offset : [ 0.0, 0.0 ],
        a_tex_coords : [0.0, 0.0, 1.0, 1.0],
        a_color : [1.0, 1.0, 1.0],
        a_alpha : 1.0
    }
}

pub struct GlDrawControl
{
    display : Rc<RefCell<SDL2Facade>>,
    window : Rc<RefCell<sdl2::video::Window>>,
    base_width : f32,
    base_height : f32,
    scale_width : f32,
    scale_height : f32,
    need_recalc_viewport : bool,
    viewport_offset : (f32, f32),
    viewport_mode : ViewportMode,
    quad_buffer : VertexBuffer<Vertex>,
    index_buffer : IndexBuffer<u16>,
    instance_buffer_src : Box<[Instance ; INSTANCES_MAX]>,
    instance_buffer_param : Box<[InstanceBufferParam ; INSTANCES_MAX]>,
    instance_buffer : Vec<VertexBuffer<Instance>>,
    instance_buffer_flip : usize,
    default_texture : Rc<Texture2d>,
    shader_list : Vec<Program>,
    scissor : Option<glium::Rect>,
    frame : Option<Frame>,
    pixel_scale : u32,
    last_pixel_scale : u32,
    pixel_texture : Option<Texture2d>,
    pixel_upscale_texture : Option<Texture2d>,
    current_index : usize,
    last_view_transform : DrawTransform,
    gpu_sheet_store : PreparedStore<Sheet, GpuSheetHandle>,
    gpu_shader_store : PreparedStore<Shader, GpuShaderHandle>,
    #[cfg(feature = "imgui_opengl")]
    imgui_renderer : Rc<RefCell<Option<imgui_glium_renderer::Renderer>>>
}

impl GlDrawControl
{
    fn draw_buffers(&mut self)
    {
        if self.current_index <= 0
        {
            return;
        }
        
        let last = self.current_index;

        self.instance_buffer[self.instance_buffer_flip].invalidate();
        self.instance_buffer[self.instance_buffer_flip].write(&self.instance_buffer_src[..]);
        
        let mut last_drawn : usize = 0;
        let mut current_op = RendererOp::BackgroundColor;
        let mut current_sheet : DataId = 0;
        let mut current_shader : DataId = 0;
        
        // I can't belive Rust is making me do this
        let buffer_params = self.instance_buffer_param.clone();
        
        let mut do_draw = |first : usize, last : usize, sheet : DataId, shader : DataId, op : RendererOp|
        {
            if first >= last
            {
                return;
            }
            
            if let Some(frame) = &mut self.frame
            {
                let mut texture_ref = self.default_texture.as_ref();
                let mut shader_ref = &self.shader_list[op as usize];
                let mut mag_filter = MagnifySamplerFilter::Linear;
                let mut min_filter = MinifySamplerFilter::Linear;
                
                if let Some(gpu_sheet_info) = self.gpu_sheet_store.get(sheet)
                {
                    texture_ref = gpu_sheet_info.texture.as_ref();
                    mag_filter = gpu_sheet_info.mag_filter;
                    min_filter = gpu_sheet_info.min_filter;
                }
                if let Some(gpu_shader_info) = self.gpu_shader_store.get(shader)
                {
                    shader_ref = &gpu_shader_info.program;
                }
                
                let uniform = uniform!
                {
                    t_texture : Sampler::new(texture_ref)
                        .magnify_filter(mag_filter).minify_filter(min_filter)
                        .wrap_function(SamplerWrapFunction::Repeat)
                };
                
                match self.viewport_mode
                {
                    ViewportMode::Independent | ViewportMode::OneToOne =>
                    {
                        let params = glium::DrawParameters
                        {
                            blend : glium::draw_parameters::Blend::alpha_blending(),
                            scissor : self.scissor,
                            .. Default::default()
                        };
                        
                        frame.draw((&self.quad_buffer,
                            self.instance_buffer[self.instance_buffer_flip].slice(first..last)
                                .expect("Failed to prepare instance buffer").per_instance()
                                .expect("Failed to prepare instance buffer")),
                            &self.index_buffer, shader_ref,
                            &uniform, &params).expect("Draw operation failed");
                    },
                    ViewportMode::Pixel =>
                    {
                        let params = glium::DrawParameters
                        {
                            blend : glium::draw_parameters::Blend::alpha_blending(),
                            scissor : self.scissor,
                            multisampling : false,
                            dithering : false,
                            .. Default::default()
                        };
                        
                        self.pixel_texture.as_ref().unwrap().as_surface().draw((&self.quad_buffer,
                            self.instance_buffer[self.instance_buffer_flip].slice(first..last)
                                .expect("Failed to prepare instance buffer").per_instance()
                                .expect("Failed to prepare instance buffer")),
                            &self.index_buffer, shader_ref,
                            &uniform, &params).expect("Draw operation failed");
                    }
                }
                
                
            }
        };
        
        for i in 0..last
        {
            let param = &buffer_params[i];
            
            if current_op == param.op && current_sheet == param.sheet
                && current_shader == param.shader && i < last - 1
            {
                continue;
            }
            
            do_draw(last_drawn, i, current_sheet, current_shader, current_op);
            
            current_op = param.op;
            current_sheet = param.sheet;
            current_shader = param.shader;
            last_drawn = i;
        }
        
        do_draw(last_drawn, last, current_sheet, current_shader, current_op);
        
        self.current_index = 0;

        self.instance_buffer_flip = (self.instance_buffer_flip + 2) % self.instance_buffer.len();
    }

    fn flush_drawing(&mut self)
    {
        self.draw_buffers();

        self.display.borrow().get_context().flush();
    }
    
    fn update_pixel_upscale(&mut self)
    {
        if self.pixel_scale == self.last_pixel_scale
        {
            return;
        }
        
        self.last_pixel_scale = self.pixel_scale;
        let display_ref : &SDL2Facade = &self.display.borrow();
        let pixel_surface = self.pixel_texture.as_mut().unwrap().as_surface();
        let (width, height) = pixel_surface.get_dimensions();
        
        self.pixel_upscale_texture = Some(try_or_else!(Texture2d::empty_with_format(
            display_ref, UncompressedFloatFormat::U8U8U8U8, glium::texture::MipmapsOption::NoMipmap,
            width * self.pixel_scale, height * self.pixel_scale),
            |error| panic!("Could not create upscale framebuffer: {}", error)));
    }
}

impl DrawControl for GlDrawControl
{
    fn new(renderer_new_info : &mut RendererNewInfo) -> DrawControlResult where Self : Sized
    {
        let video_subsystem = &renderer_new_info.video_subsystem;
        let mut window_builder = video_subsystem.window("", renderer_new_info.width * renderer_new_info.default_zoom,
        renderer_new_info.height * renderer_new_info.default_zoom);
        
        window_builder.hidden().position_centered().resizable().allow_highdpi();
        
        video_subsystem.gl_attr().set_context_profile(sdl2::video::GLProfile::Core);
        video_subsystem.gl_attr().set_context_version(3, 2);
        
        let display_raw = try_or_else!(window_builder.build_glium(),
            |error| Err(RendererError::RendererInitFailed(format!("Could not create game window: {}", error))));
        
        let mut pixel_texture = None;
        
        match renderer_new_info.viewport_mode
        {
            ViewportMode::Independent | ViewportMode::OneToOne => (),
            ViewportMode::Pixel =>
            {
                pixel_texture = Some(try_or_else!(Texture2d::empty_with_format(
                    &display_raw, UncompressedFloatFormat::U8U8U8U8, glium::texture::MipmapsOption::NoMipmap, renderer_new_info.width, renderer_new_info.height),
                    |error| Err(RendererError::RendererInitFailed(format!("Could not create framebuffer: {}", error)))));
            }
        };
        
        let quad_buffer = try_or_else!(glium::VertexBuffer::new(&display_raw, &VERTEX_QUAD),
            |error| Err(RendererError::RendererInitFailed(format!("Could not create vertex buffer: {}", error))));
        let index_buffer = try_or_else!(glium::IndexBuffer::new(&display_raw,
            glium::index::PrimitiveType::TrianglesList, &INDEX_QUAD),
            |error| Err(RendererError::RendererInitFailed(format!("Could not create index buffer: {}", error))));

        let mut instance_buffer = Vec::new();

        for _ in 0..INSTANCE_BUFFER_MAX
        {
            instance_buffer.push(try_or_else!(glium::VertexBuffer::<Instance>::empty_dynamic(&display_raw, INSTANCES_MAX),
                |error| return Err(RendererError::RendererInitFailed(format!("Could not create instance buffer: {}", error)))));
        }
        
        let context = display_raw.get_context();
        info!("Renderer: {} {}", context.get_opengl_vendor_string(), context.get_opengl_renderer_string());
        info!("OpenGL version: {}", context.get_opengl_version_string());
        if let Some(free_vram) = context.get_free_video_memory()
        {
            info!("VRAM: {} MiB free", free_vram / 1024 / 1024);
        }
        
        let shader_background = try_or_else!(make_program(&display_raw,
            include_str!("shader/vert_standard_140.glsl"),
            include_str!("shader/frag_solidcolor_140.glsl")),
            |error| Err(RendererError::RendererInitFailed(format!("Could not load shader: {}", error))));
        let shader_sprite = try_or_else!(make_program(&display_raw,
            include_str!("shader/vert_standard_140.glsl"),
            include_str!("shader/frag_sprite_140.glsl")),
            |error| Err(RendererError::RendererInitFailed(format!("Could not load shader: {}", error))));
        
        let shader_list = vec!(shader_background, shader_sprite);

        let (default_texture, default_width, default_height) = crate::renderer::png_to_raw(
            Cursor::new(include_bytes!("data/default.png").to_vec()))?;
        
        let image = RawImage2d::from_raw_rgba(default_texture, (default_width, default_height));
        let default_texture = Rc::new(try_or_else!(Texture2d::new(&display_raw, image),
            |error| Err(RendererError::RendererInitFailed(format!("Could not upload texture: {}", error)))));
        
        let window = display_raw.window_clone();
        
        let display = Rc::new(RefCell::new(display_raw));
        #[cfg(feature = "imgui_opengl")]
        let imgui_renderer = Rc::new(RefCell::new(None));
        let sheet_data_preparer = Box::new(GpuSheetPreparer
        {
            display : display.clone(),
            default_texture: default_texture.clone(),
            texture_filtering : renderer_new_info.texture_filtering,
            #[cfg(feature = "imgui_opengl")]
            imgui_renderer: imgui_renderer.clone()
        });
        let shader_data_preparer = Box::new(GpuShaderPreparer
        {
            display : display.clone()
        });
        
        let gpu_sheet_store = try_or_else!(PreparedStore::new(&mut renderer_new_info.resources.store_mut(),
            sheet_data_preparer),
            |error| Err(RendererError::LoadResourceFailed(format!("Failed to create GPU sheet handle store: {}", error))));
        
        let gpu_shader_store = try_or_else!(PreparedStore::new(&mut renderer_new_info.resources.store_mut(),
            shader_data_preparer),
            |error| Err(RendererError::LoadResourceFailed(format!("Failed to create GPU shader handle store: {}", error))));
        
        Ok((Box::new(GlDrawControl
        {
            display,
            window : window.clone(),
            viewport_offset : (0.0, 0.0),
            viewport_mode : renderer_new_info.viewport_mode,
            base_width : renderer_new_info.width as f32,
            base_height : renderer_new_info.height as f32,
            scale_width : 0.0,
            scale_height : 0.0,
            need_recalc_viewport : false,
            quad_buffer,
            index_buffer,
            instance_buffer_src : Box::new([Default::default() ; INSTANCES_MAX]),
            instance_buffer_param : Box::new([Default::default() ; INSTANCES_MAX]),
            instance_buffer,
            instance_buffer_flip : 0,
            default_texture,
            shader_list,
            scissor : None,
            frame : None,
            pixel_scale : 1,
            last_pixel_scale : 0,
            pixel_texture,
            pixel_upscale_texture : None,
            current_index : 0,
            last_view_transform : DrawTransform::identity(),
            gpu_sheet_store,
            gpu_shader_store,
            #[cfg(feature = "imgui_opengl")]
            imgui_renderer
        }), window))
    }

    fn base_rect(&self) -> BakedRect<f32>
    {
        BakedRect
        {
            x1: 0.0,
            y1: 0.0,
            x2: self.base_width,
            y2: self.base_height
        }
    }

    fn screen_transform(&self) -> &DrawTransform
    {
        &self.last_view_transform
    }
    
    fn sync_sheet_store(&mut self, sheet_store : &mut DataStore<Sheet>) -> Result<(), PreparedStoreError>
    {
        self.gpu_sheet_store.sync(sheet_store)
    }
    
    fn sync_shader_store(&mut self, shader_store : &mut DataStore<Shader>) -> Result<(), PreparedStoreError>
    {
        self.gpu_shader_store.sync(shader_store)
    }

    fn draw_sprite(&mut self, draw_info: &DrawSpriteInfo)
    {
        let mut transform = draw_info.transform.clone();
        let mut transform_offset_x = -0.5;
        let mut transform_offset_y = -0.5;
        let mut texture_x = 0.0;
        let mut texture_y = 0.0;
        let mut texture_w = 1.0;
        let mut texture_h = 1.0;
        let mut sheet_id = 0;

        transform.translate(draw_info.x, draw_info.y);
        if draw_info.angle != 0.0
        {
            transform.rotate(draw_info.angle);
        }

        let mut got_slice = false;

        if let Some(gpu_sheet_info) = self.gpu_sheet_store.get(draw_info.sheet_id)
        {
            if draw_info.slice < gpu_sheet_info.slices.len()
            {
                let slice = &gpu_sheet_info.slices[draw_info.slice];

                transform.scale(draw_info.scale_x * slice.scale_x, draw_info.scale_y * slice.scale_y);

                sheet_id = draw_info.sheet_id;
                transform_offset_x = slice.offset_x;
                transform_offset_y = slice.offset_y;
                texture_x = slice.texture_x;
                texture_y = slice.texture_y;
                texture_w = slice.texture_w;
                texture_h = slice.texture_h;

                got_slice = true;
            }
        }

        if !got_slice
        {
            transform.scale(draw_info.scale_x * 32.0, draw_info.scale_y * 32.0);
        }

        self.draw_sprite_raw(&DrawSpriteRawInfo
        {
            sheet_id,
            shader_id: draw_info.shader_id,
            transform: transform,
            transform_offset_x,
            transform_offset_y,
            texture_x,
            texture_y,
            texture_w,
            texture_h,
            r: draw_info.r,
            g: draw_info.g,
            b: draw_info.b,
            alpha: draw_info.alpha
        });
    }
    
    fn draw_sprite_raw(&mut self, draw_info : &DrawSpriteRawInfo)
    {
        if self.current_index >= INSTANCES_MAX
        {
            self.flush_drawing();
        }
        
        let transform = &draw_info.transform;
        let instance = Instance
        {
            a_transform_col1 : [ transform.mat.x.x, transform.mat.x.y ],
            a_transform_col2 : [ transform.mat.y.x, transform.mat.y.y ],
            a_transform_col3 : [ transform.mat.z.x, transform.mat.z.y ],
            a_transform_offset : [ draw_info.transform_offset_x, draw_info.transform_offset_y ],
            a_tex_coords : [ draw_info.texture_x, draw_info.texture_y, draw_info.texture_w, draw_info.texture_h ],
            a_color : [ draw_info.r, draw_info.g, draw_info.b ],
            a_alpha : draw_info.alpha
        };
        self.instance_buffer_src[self.current_index] = instance;
        self.instance_buffer_param[self.current_index] = InstanceBufferParam
        {
            op : RendererOp::Sprite,
            sheet : draw_info.sheet_id,
            shader : draw_info.shader_id
        };
        
        self.current_index += 1;
    }
    
    fn draw_background_color(&mut self, draw_info : &DrawBackgroundColorInfo )
    {
        if self.current_index >= INSTANCES_MAX
        {
            self.flush_drawing();
        }
        
        let instance = Instance
        {
            a_transform_col1 : [ 2.0, 0.0 ],
            a_transform_col2 : [ 0.0, -2.0 ],
            a_transform_col3 : [ -1.0, 1.0 ],
            a_transform_offset : [ 0.0, 0.0 ],
            a_tex_coords : [ 0.0, 0.0, 1.0, 1.0 ],
            a_color : [ draw_info.r, draw_info.g, draw_info.b ],
            a_alpha : draw_info.alpha
        };
        self.instance_buffer_src[self.current_index] = instance;
        self.instance_buffer_param[self.current_index] = InstanceBufferParam
        {
            op : RendererOp::BackgroundColor,
            sheet : 0,
            shader : draw_info.shader_id
        };
        
        self.current_index += 1;
    }
    
    #[cfg(feature = "imgui_base")]
    fn draw_imgui(&mut self, ui : imgui::Ui)
    {
        self.flush_drawing();
        
        if let Some(frame) = &mut self.frame
        {
            let mut imgui_borrow = self.imgui_renderer.borrow_mut();

            if let Some(imgui_renderer) = imgui_borrow.as_mut()
            {
                let draw_data = ui.render();
                imgui_renderer.render(frame, &draw_data).expect("imgui render failed");
            }
        }
    }
    
    fn recalculate_viewport(&mut self, base_width : f32, base_height : f32)
    {
        self.base_width = base_width;
        self.base_height = base_height;

        if self.frame.is_some()
        {
            self.need_recalc_viewport = true;
            return;
        }

        let (width, height) = self.window.borrow().size();
        let info = crate::renderer::recalculate_viewport_common(base_width, base_height, width, height, self.viewport_mode);

        self.scale_width = info.scale_width;
        self.scale_height = info.scale_height;
        self.viewport_offset = (info.scale_offset_x, info.scale_offset_y);
        self.pixel_scale = info.pixel_scale;
        self.scissor = match info.scissor
        {
            Some(scissor) =>
            {
                Some(glium::Rect
                {
                    left : scissor.x,
                    bottom : scissor.y,
                    width : scissor.w,
                    height : scissor.h
                })
            },
            None => None
        }
    }
    
    fn start_drawing(&mut self)
    {
        if self.frame.is_some()
        {
            panic!("Cannot call draw() twice without a present()");
        }

        if self.need_recalc_viewport
        {
            self.recalculate_viewport(self.base_width, self.base_height);

            self.need_recalc_viewport = false;
        }
        
        self.frame = Some(self.display.borrow().draw());
        
        self.frame.as_mut().unwrap().clear_color_and_depth((0.0, 0.0, 0.0, 1.0), 1.0);
        
        let mut view_transform = DrawTransform::from_array(&UNBORKED_IDENTITY_TRANSFORM);
        
        match self.viewport_mode
        {
            ViewportMode::Independent | ViewportMode::OneToOne =>
            {
                let (x, y) = self.viewport_offset;
                
                view_transform.scale(1.0 / self.scale_width, 1.0 / self.scale_height);
                view_transform.translate(x, y);
            },
            ViewportMode::Pixel =>
            {
                let mut pixel_surface = self.pixel_texture.as_mut().unwrap().as_surface();
                let (width, height) = pixel_surface.get_dimensions();
                
                pixel_surface.clear_color_and_depth((0.0, 0.0, 0.0, 1.0), 1.0);
                
                view_transform.scale(1.0 / width as f32, 1.0 / height as f32);
            }
        }
        
        self.last_view_transform = view_transform.clone();
        self.instance_buffer_flip = (self.instance_buffer_flip + 1) % self.instance_buffer.len();
    }

    fn end_drawing(&mut self)
    {
        self.flush_drawing();
    }
    
    fn present(&mut self)
    {
        if self.frame.is_some()
        {
            match self.viewport_mode
            {
                ViewportMode::Independent | ViewportMode::OneToOne => (),
                ViewportMode::Pixel =>
                {
                    // Low-res pixel to high-res pixel
                    
                    let mut transform = DrawTransform::from_array(&UNBORKED_UPSCALE_TRANSFORM);
                    
                    let params = glium::DrawParameters::default();
                    let instance = upscale_instance_from_transform(&transform);
                    
                    self.instance_buffer[self.instance_buffer_flip].invalidate();
                    self.instance_buffer[self.instance_buffer_flip].map_write().set(0, instance);
                    
                    self.update_pixel_upscale();
                    
                    let uniform = uniform!
                    {
                        t_texture : Sampler::new(self.pixel_texture.as_ref().unwrap())
                            .magnify_filter(MagnifySamplerFilter::Nearest)
                            .minify_filter(MinifySamplerFilter::Nearest)
                            .wrap_function(SamplerWrapFunction::Clamp)
                    };
                    
                    self.pixel_upscale_texture.as_mut().unwrap().as_surface().draw((&self.quad_buffer,
                        self.instance_buffer[self.instance_buffer_flip].slice(0..1)
                            .expect("Failed to prepare instance buffer").per_instance()
                            .expect("Failed to prepare instance buffer")),
                        &self.index_buffer, &self.shader_list[RendererOp::Sprite as usize],
                        &uniform, &params).expect("Draw operation failed");
                    
                    // High-res pixel to screen
                    
                    let uniform = uniform!
                    {
                        t_texture : Sampler::new(self.pixel_upscale_texture.as_ref().unwrap())
                            .magnify_filter(MagnifySamplerFilter::Linear)
                            .minify_filter(MinifySamplerFilter::Linear)
                            .wrap_function(SamplerWrapFunction::Clamp)
                    };
                    
                    let (x, y) = self.viewport_offset;
                    transform.scale(1.0 / self.scale_width, 1.0 / self.scale_height);
                    transform.translate(x, y);
                    let instance = upscale_instance_from_transform(&transform);
                    
                    self.instance_buffer[self.instance_buffer_flip].map_write().set(1, instance);
                    
                    self.frame.as_mut().unwrap().draw((&self.quad_buffer,
                        self.instance_buffer[self.instance_buffer_flip].slice(1..2)
                            .expect("Failed to prepare instance buffer").per_instance()
                            .expect("Failed to prepare instance buffer")),
                        &self.index_buffer, &self.shader_list[RendererOp::Sprite as usize],
                        &uniform, &params).expect("Draw operation failed");
                }
            }
            
            let mut frame = None;        
            std::mem::swap(&mut frame, &mut self.frame);
            
            frame.unwrap().finish().expect("Draw finish failed");
        }
    }
    
    #[cfg(feature = "imgui_base")]
    fn init_imgui_renderer(&mut self, imgui : &mut imgui::Context) -> Result<(), RendererError>
    {
        let mut imgui_borrow = self.imgui_renderer.borrow_mut();

        if imgui_borrow.is_some()
        {
            return Err(RendererError::ImguiRendererFailed("Already initialized".to_string()));
        }
        
        let mut imgui_renderer = try_or_else!(imgui_glium_renderer::Renderer::init(
            imgui, &*self.display.borrow()),
            |error| Err(RendererError::ImguiRendererFailed(format!("Could not initialize imgui renderer: {}", error))));

        imgui_renderer.textures().replace(imgui::TextureId::from(0), imgui_glium_renderer::Texture
        {
            texture : self.default_texture.clone(),
            sampler : SamplerBehavior
            {
                wrap_function: (SamplerWrapFunction::Repeat, SamplerWrapFunction::Repeat, SamplerWrapFunction::Repeat),
                minify_filter: MinifySamplerFilter::Linear,
                magnify_filter: MagnifySamplerFilter::Linear,
                .. Default::default()
            }
        });
        
        *imgui_borrow = Some(imgui_renderer);
        
        Ok(())
    }

    #[cfg(feature = "imgui_base")]
    fn update_imgui_renderer(&mut self, imgui : &mut imgui::Context)
    {
        let mut imgui_borrow = self.imgui_renderer.borrow_mut();

        if let Some(imgui_renderer) = imgui_borrow.as_mut()
        {
            if !imgui.fonts().is_built()
            {
                imgui_renderer.reload_font_texture(imgui).expect("Font texture update failed");
            }
        }
    }
}
