//! Input module, for handling control bindings

use std::collections::{HashMap, HashSet};

use sdl2::controller::{Button, Axis};

const ANGLE_UP_RIGHT_RIGHT : f64 = -std::f64::consts::FRAC_PI_8;
const ANGLE_UP_RIGHT : f64 = -std::f64::consts::FRAC_PI_4;
const ANGLE_UP_UP_RIGHT : f64 = -std::f64::consts::FRAC_PI_4 - std::f64::consts::FRAC_PI_8;
const ANGLE_UP_UP_LEFT : f64 = -std::f64::consts::FRAC_PI_2 - std::f64::consts::FRAC_PI_8;
const ANGLE_UP_LEFT : f64 = -std::f64::consts::FRAC_PI_2 - std::f64::consts::FRAC_PI_4;
const ANGLE_UP_LEFT_LEFT : f64 = -std::f64::consts::FRAC_PI_2 - std::f64::consts::FRAC_PI_4 - std::f64::consts::FRAC_PI_8;
const ANGLE_DOWN_LEFT_LEFT : f64 = std::f64::consts::FRAC_PI_2 + std::f64::consts::FRAC_PI_4 + std::f64::consts::FRAC_PI_8;
const ANGLE_DOWN_LEFT : f64 = std::f64::consts::FRAC_PI_2 + std::f64::consts::FRAC_PI_4;
const ANGLE_DOWN_DOWN_LEFT : f64 = std::f64::consts::FRAC_PI_2 + std::f64::consts::FRAC_PI_8;
const ANGLE_DOWN_DOWN_RIGHT : f64 = std::f64::consts::FRAC_PI_4 + std::f64::consts::FRAC_PI_8;
const ANGLE_DOWN_RIGHT : f64 = std::f64::consts::FRAC_PI_4;
const ANGLE_DOWN_RIGHT_RIGHT : f64 = std::f64::consts::FRAC_PI_8;

const DIAGONAL_DEAD_ZONE : f64 = std::f64::consts::FRAC_PI_8 / 2.0;
const DEFAULT_STICK_DEADZONE : f64 = 0.2;
const DEFAULT_TRIGGER_DEADZONE : f64 = 0.15;
const AXIS_DEADZONE_MAX : f64 = 0.999;
const AXIS_TO_KEY_DEADZONE : f64 = 0.5;

struct InputAction
{
    time_in_state : u64,
    held : bool,
    old_held : bool,
    axis : f64
}

impl InputAction
{
    pub fn new() -> InputAction
    {
        InputAction { time_in_state : 0, held : false, old_held : false, axis : 0.0 }
    }
    
    pub fn update_state(&mut self, is_held : bool, axis : f64)
    {
        self.old_held = self.held;
        
        if self.held == is_held
        {
            self.time_in_state += 1;
        }
        else
        {
            self.time_in_state = 1;
            self.held = is_held
        }

        self.axis = axis;
    }
    
    pub fn time_in_state(&self) -> u64
    {
        self.time_in_state
    }
    
    pub fn is_held(&self) -> bool
    {
        self.held
    }
    
    pub fn down_once(&self) -> bool
    {
        self.held && !self.old_held
    }
    
    pub fn up_once(&self) -> bool
    {
        !self.held && self.old_held
    }

    pub fn axis(&self) -> f64
    {
        self.axis
    }
}

#[derive(Debug, Clone)]
struct AxisData
{
    raw : f64,
    value : f64
}

#[derive(Debug, Clone)]
enum CompositeAxisData
{
    AxisTrigger,
    AxisStick
    {
        neg_x : String,
        pos_x : String,
        neg_y : String,
        pos_y : String
    }
}

#[derive(Debug, Clone)]
struct DeadzoneData
{
    composite_axis : CompositeAxisData,
    deadzone : f64
}

#[derive(Debug, Clone, Copy, PartialEq, Eq)]
pub enum InputDirection4
{
    Left,
    Right,
    Up,
    Down
}

#[derive(Debug, Clone, Copy, PartialEq, Eq)]
pub enum InputDirection8
{
    Left,
    UpLeft,
    Up,
    UpRight,
    Right,
    DownRight,
    Down,
    DownLeft
}

fn apply_deadzone(deadzone_opt : Option<&f64>, value : f64) -> f64
{
    if let Some(deadzone) = deadzone_opt
    {
        let deadzone = deadzone.clamp(0.0, AXIS_DEADZONE_MAX);

        return ((value - deadzone) * (1.0 / (1.0 - deadzone))).clamp(0.0, 1.0);
    }

    value
}

pub struct Input
{
    keys_held : HashSet<String>,
    axis_values : HashMap<String, AxisData>,
    actions_to_keys : HashMap<String, Vec<String>>,
    action_map : HashMap<String, InputAction>,
    gamepad_to_key : HashMap<Button, String>,
    gamepad_to_axis_pos : HashMap<Axis, String>,
    gamepad_to_axis_neg : HashMap<Axis, String>,
    composite_axes : HashMap<String, CompositeAxisData>,
    gamepad_deadzones : HashMap<String, f64>,
    mouse_util : Option<sdl2::mouse::MouseUtil>
}

impl Input
{
    pub(crate) fn new(mouse_util : Option<sdl2::mouse::MouseUtil>) -> Input
    {
        let mut gamepad_to_key = HashMap::new();

        gamepad_to_key.insert(Button::A, String::from("gamepad a"));
        gamepad_to_key.insert(Button::B, String::from("gamepad b"));
        gamepad_to_key.insert(Button::X, String::from("gamepad x"));
        gamepad_to_key.insert(Button::Y, String::from("gamepad y"));
        gamepad_to_key.insert(Button::Back, String::from("gamepad back"));
        gamepad_to_key.insert(Button::Start, String::from("gamepad start"));
        gamepad_to_key.insert(Button::LeftStick, String::from("gamepad lstick"));
        gamepad_to_key.insert(Button::RightStick, String::from("gamepad rstick"));
        gamepad_to_key.insert(Button::LeftShoulder, String::from("gamepad l"));
        gamepad_to_key.insert(Button::RightShoulder, String::from("gamepad r"));
        gamepad_to_key.insert(Button::DPadLeft, String::from("gamepad left"));
        gamepad_to_key.insert(Button::DPadRight, String::from("gamepad right"));
        gamepad_to_key.insert(Button::DPadUp, String::from("gamepad up"));
        gamepad_to_key.insert(Button::DPadDown, String::from("gamepad down"));
        gamepad_to_key.insert(Button::Guide, String::from("gamepad guide"));

        let mut gamepad_to_axis_pos = HashMap::new();
        let mut gamepad_to_axis_neg = HashMap::new();

        gamepad_to_axis_neg.insert(Axis::LeftX, String::from("gamepad lstick left"));
        gamepad_to_axis_pos.insert(Axis::LeftX, String::from("gamepad lstick right"));
        gamepad_to_axis_neg.insert(Axis::LeftY, String::from("gamepad lstick up"));
        gamepad_to_axis_pos.insert(Axis::LeftY, String::from("gamepad lstick down"));
        gamepad_to_axis_neg.insert(Axis::RightX, String::from("gamepad rstick left"));
        gamepad_to_axis_pos.insert(Axis::RightX, String::from("gamepad rstick right"));
        gamepad_to_axis_neg.insert(Axis::RightY, String::from("gamepad rstick up"));
        gamepad_to_axis_pos.insert(Axis::RightY, String::from("gamepad rstick down"));

        // L and R triggers only go from 0.0 to 1.0, so they have no negative axis.
        gamepad_to_axis_pos.insert(Axis::TriggerLeft, String::from("gamepad ltrigger"));
        gamepad_to_axis_pos.insert(Axis::TriggerRight, String::from("gamepad rtrigger"));

        let mut composite_axes = HashMap::new();

        composite_axes.insert(String::from("gamepad lstick"), CompositeAxisData::AxisStick
        {
            neg_x : String::from("gamepad lstick left"),
            pos_x : String::from("gamepad lstick right"),
            neg_y : String::from("gamepad lstick up"),
            pos_y : String::from("gamepad lstick down")
        });
        composite_axes.insert(String::from("gamepad rstick"), CompositeAxisData::AxisStick
        {
            neg_x : String::from("gamepad rstick left"),
            pos_x : String::from("gamepad rstick right"),
            neg_y : String::from("gamepad rstick up"),
            pos_y : String::from("gamepad rstick down")
        });
        composite_axes.insert(String::from("gamepad ltrigger"), CompositeAxisData::AxisTrigger);
        composite_axes.insert(String::from("gamepad rtrigger"), CompositeAxisData::AxisTrigger);

        let mut gamepad_deadzones = HashMap::new();

        gamepad_deadzones.insert(String::from("gamepad lstick"), DEFAULT_STICK_DEADZONE);
        gamepad_deadzones.insert(String::from("gamepad rstick"), DEFAULT_STICK_DEADZONE);
        gamepad_deadzones.insert(String::from("gamepad ltrigger"), DEFAULT_TRIGGER_DEADZONE);
        gamepad_deadzones.insert(String::from("gamepad rtrigger"), DEFAULT_TRIGGER_DEADZONE);

        Input
        {
            keys_held : HashSet::new(),
            axis_values: HashMap::new(),
            actions_to_keys : HashMap::new(),
            action_map : HashMap::new(),
            gamepad_to_key,
            gamepad_to_axis_pos,
            gamepad_to_axis_neg,
            composite_axes,
            gamepad_deadzones,
            mouse_util
        }
    }
    
    pub fn add_action(&mut self, name : &str)
    {
        self.action_map.insert(name.to_string(), InputAction::new());
        self.actions_to_keys.insert(name.to_string(), Vec::new());
    }
    
    pub fn add_bind(&mut self, name : &str, key : &str)
    {
        if !self.action_map.contains_key(name)
        {
            self.add_action(name);
        }

        let lower_key = key.to_lowercase();

        if let Some(key_list) = self.actions_to_keys.get_mut(name)
        {
            key_list.push(lower_key);
        }
    }
    
    pub fn clear_binds(&mut self, name : &str)
    {
        if let Some(key_list) = self.actions_to_keys.get_mut(name)
        {
            key_list.clear();
        }

        if let Some(action) = self.action_map.get_mut(name)
        {
            action.axis = 0.0;
        }
    }

    pub fn deadzone(&self, composite_axis : &str) -> Option<f64>
    {
        self.gamepad_deadzones.get(composite_axis).cloned()
    }

    pub fn set_deadzone(&mut self, composite_axis : &str, deadzone : f64) -> bool
    {
        let clamped = deadzone.clamp(0.0, AXIS_DEADZONE_MAX);

        if let Some(deadzone) = self.gamepad_deadzones.get_mut(composite_axis)
        {
            *deadzone = clamped;
            return true;
        }

        false
    }
    
    pub fn down_once(&self, name : &str) -> bool
    {
        if let Some(action) = self.action_map.get(name)
        {
            return action.down_once();
        }

        false
    }
    
    pub fn up_once(&self, name : &str) -> bool
    {
        if let Some(action) = self.action_map.get(name)
        {
            return action.up_once();
        }

        false
    }
    
    pub fn held(&self, name : &str) -> bool
    {
        if let Some(action) = self.action_map.get(name)
        {
            return action.is_held();
        }

        false
    }
    
    pub fn time_held(&self, name : &str) -> u64
    {
        if let Some(action) = self.action_map.get(name)
        {
            if action.is_held()
            {
                return action.time_in_state();
            }
        }
        
        0
    }
    
    pub fn time_released(&self, name : &str) -> u64
    {
        if let Some(action) = self.action_map.get(name)
        {
            if !action.is_held()
            {
                return action.time_in_state();
            }
        }
        
        0
    }

    pub fn axis(&self, name : &str) -> f64
    {
        if let Some(action) = self.action_map.get(name)
        {
            return action.axis().clamp(0.0, 1.0);
        }

        0.0
    }

    pub fn paired_axis(&self, name_neg : &str, name_pos : &str) -> f64
    {
        let neg_value = self.axis(name_neg);
        let pos_value = self.axis(name_pos);

        pos_value - neg_value
    }

    pub fn angle_and_distance(&self, name_left : &str, name_right : &str, name_up : &str, name_down : &str) -> (f64, f64)
    {
        let x = self.paired_axis(name_left, name_right);
        let y = self.paired_axis(name_up, name_down);

        let angle = y.atan2(x);
        let distance = (x.powi(2) + y.powi(2)).sqrt().min(1.0);

        (angle, distance)
    }

    pub fn four_way_direction(&self, name_left : &str, name_right : &str, name_up : &str, name_down : &str) -> Option<InputDirection4>
    {
        let (angle, distance) = self.angle_and_distance(name_left, name_right, name_up, name_down);

        if distance < AXIS_TO_KEY_DEADZONE
        {
            return None;
        }

        if angle < ANGLE_UP_LEFT - DIAGONAL_DEAD_ZONE || angle > ANGLE_DOWN_LEFT + DIAGONAL_DEAD_ZONE
        {
            return Some(InputDirection4::Left)
        }
        else if angle > ANGLE_UP_RIGHT + DIAGONAL_DEAD_ZONE && angle < ANGLE_DOWN_RIGHT - DIAGONAL_DEAD_ZONE
        {
            return Some(InputDirection4::Right)
        }
        else if angle > ANGLE_UP_LEFT + DIAGONAL_DEAD_ZONE && angle < ANGLE_UP_RIGHT - DIAGONAL_DEAD_ZONE
        {
            return Some(InputDirection4::Up)
        }
        else if angle > ANGLE_DOWN_RIGHT + DIAGONAL_DEAD_ZONE && angle < ANGLE_DOWN_LEFT - DIAGONAL_DEAD_ZONE
        {
            return Some(InputDirection4::Down)
        }

        None
    }

    pub fn eight_way_direction(&self, name_left : &str, name_right : &str, name_up : &str, name_down : &str) -> Option<InputDirection8>
    {
        let (angle, distance) = self.angle_and_distance(name_left, name_right, name_up, name_down);

        if distance < AXIS_TO_KEY_DEADZONE
        {
            return None;
        }

        if angle < ANGLE_UP_LEFT_LEFT || angle >= ANGLE_DOWN_LEFT_LEFT
        {
            return Some(InputDirection8::Left)
        }
        else if angle >= ANGLE_UP_LEFT_LEFT && angle < ANGLE_UP_UP_LEFT
        {
            return Some(InputDirection8::UpLeft)
        }
        else if angle >= ANGLE_UP_UP_LEFT && angle < ANGLE_UP_UP_RIGHT
        {
            return Some(InputDirection8::Up)
        }
        else if angle >= ANGLE_UP_UP_RIGHT && angle < ANGLE_UP_RIGHT_RIGHT
        {
            return Some(InputDirection8::UpRight)
        }
        else if angle >= ANGLE_UP_RIGHT_RIGHT && angle < ANGLE_DOWN_RIGHT_RIGHT
        {
            return Some(InputDirection8::Right)
        }
        else if angle >= ANGLE_DOWN_RIGHT_RIGHT && angle < ANGLE_DOWN_DOWN_RIGHT
        {
            return Some(InputDirection8::DownRight)
        }
        else if angle >= ANGLE_DOWN_DOWN_RIGHT && angle < ANGLE_DOWN_DOWN_LEFT
        {
            return Some(InputDirection8::Down)
        }
        else if angle >= ANGLE_DOWN_DOWN_LEFT && angle < ANGLE_DOWN_LEFT_LEFT
        {
            return Some(InputDirection8::DownLeft)
        }

        return None;
    }

    pub fn cursor_visible(&self) -> bool
    {
        if let Some(mouse_util) = &self.mouse_util
        {
            return mouse_util.is_cursor_showing();
        }

        true
    }

    pub fn set_cursor_visible(&mut self, visible : bool)
    {
        if let Some(mouse_util) = &self.mouse_util
        {
            mouse_util.show_cursor(visible);
        }
    }
    
    pub(crate) fn update_key(&mut self, key : &str, pressed : bool, update_axis : bool)
    {
        let lower_key = key.to_lowercase();

        if pressed
        {
            self.keys_held.insert(lower_key.clone());

            if update_axis
            {
                self.axis_values.insert(lower_key, AxisData { value: 1.0, raw : 1.0 });
            }
        }
        else
        {
            self.keys_held.remove(&lower_key);

            if update_axis
            {
                self.axis_values.insert(lower_key, AxisData { value: 0.0, raw : 0.0 });
            }
        }
    }

    pub(crate) fn update_axis(&mut self, axis : &str, value : f64)
    {
        let lower_axis = axis.to_lowercase();

        self.axis_values.insert(lower_axis, AxisData { value, raw : value } );
    }

    pub(crate) fn update_gamepad_key(&mut self, key : Button, pressed : bool)
    {
        let gamepad_key_string = self.gamepad_to_key[&key].clone();

        self.update_key(&gamepad_key_string, pressed, true)
    }

    pub(crate) fn update_gamepad_axis(&mut self, axis : Axis, value : i16)
    {
        if let Some(gamepad_axis_string_neg) = self.gamepad_to_axis_neg.get(&axis).cloned()
        {
            let split_value = (value as f64 / -32768.0).clamp(0.0, 1.0);

            self.update_axis(&gamepad_axis_string_neg, split_value);
            self.update_key(&gamepad_axis_string_neg, split_value >= AXIS_TO_KEY_DEADZONE, false);
        }
        if let Some(gamepad_axis_string_pos) = self.gamepad_to_axis_pos.get(&axis).cloned()
        {
            let split_value = (value as f64 / 32767.0).clamp(0.0, 1.0);

            self.update_axis(&gamepad_axis_string_pos, split_value);
            self.update_key(&gamepad_axis_string_pos, split_value >= AXIS_TO_KEY_DEADZONE, false);
        }
    }

    fn adjust_axes_for_deadzone(&mut self)
    {
        fn try_get_axis_raw(axis_map : &HashMap<String, AxisData>, axis_name : &str) -> f64
        {
            if let Some(axis) = axis_map.get(axis_name)
            {
                return axis.raw;
            }

            0.0
        }

        for (composite_axis, axes) in self.composite_axes.iter()
        {
            match axes
            {
                CompositeAxisData::AxisTrigger =>
                {
                    if let Some(axis) = self.axis_values.get_mut(composite_axis)
                    {
                        axis.value = apply_deadzone(self.gamepad_deadzones.get(composite_axis), axis.raw);
                    }
                }
                CompositeAxisData::AxisStick { neg_x, pos_x, neg_y, pos_y } =>
                {
                    let x = try_get_axis_raw(&self.axis_values, pos_x) - try_get_axis_raw(&self.axis_values, neg_x);
                    let y = try_get_axis_raw(&self.axis_values, pos_y) - try_get_axis_raw(&self.axis_values, neg_y);

                    let distance = (x.powi(2) + y.powi(2)).sqrt().min(1.0);
                    let adjusted = apply_deadzone(self.gamepad_deadzones.get(composite_axis), distance);

                    for axis_name in &[neg_x, pos_x, neg_y, pos_y]
                    {
                        if let Some(axis) = self.axis_values.get_mut(*axis_name)
                        {
                            axis.value = axis.raw * adjusted;
                        }
                    }
                }
            }
        }
    }

    pub(crate) fn update_actions(&mut self)
    {
        self.adjust_axes_for_deadzone();

        for (name, action) in self.action_map.iter_mut()
        {
            let mut is_held = false;
            let mut axis_cumulative = 0.0;

            for key in &self.actions_to_keys[name]
            {
                if self.keys_held.contains(key)
                {
                    is_held = true;
                }

                if let Some(data) = self.axis_values.get_mut(key)
                {
                    axis_cumulative += data.value;
                }
            }

            axis_cumulative = axis_cumulative.clamp(-1.0, 1.0);

            action.update_state(is_held, axis_cumulative);
        }
    }
}
