//! High-performance, pure-Rust implementation of the Keccak-_p_ permutation.
//!
//! Optimized for generic 64-bit processors.

#![no_std]

mod macros;

/// The Keccak-_p_\[1600,12\] permutation.
///
/// ```rust
/// use keccak_p::keccak_p1600_12;
///
/// let mut lanes = [0u64; 25];
/// keccak_p1600_12(&mut lanes);
///
/// assert_eq!(lanes, [
///     0x8E5E5438B9A78617, 0xD9CD6A50F259D01E, 0x87B8E7C652A91F35, 0x1093E067CDE4E0C5, 0xB033AB90F2D95A45,
///     0xE0A72F72A8DD1A45, 0xC53780AA14672F9C, 0x3EDD47F50051071D, 0xB3A31D310C178ACC, 0x79B586A59257AAA0,
///     0xBC4A7C3DB3B1F99B, 0x68874063E68A6793, 0x5C6C03332E0E2566, 0x9CAA1202B9F030DA, 0x5F3B9A782BCF7A9F,
///     0xE536C1E061AE7923, 0x6DE9B618B73C87EC, 0x2ABED1F170918AC2, 0x6AABBD53DAED24B7, 0xBFC1416A2C2EE15A,
///     0xC6CFE036B90952AF, 0x45503617DC7060D7, 0x625611B2C29F7AE4, 0xD43671DB2C30647A, 0xCFFD0D76222CA01C,
/// ]);
/// ```
#[inline(always)]
#[allow(unused_assignments)]
pub fn keccak_p1600_12(lanes: &mut [u64; 25]) {
    iter_rounds!(
        lanes,
        (0x000000008000808b, 0x800000000000008b),
        (0x8000000000008089, 0x8000000000008003),
        (0x8000000000008002, 0x8000000000000080),
        (0x000000000000800a, 0x800000008000000a),
        (0x8000000080008081, 0x8000000000008080),
        (0x0000000080000001, 0x8000000080008008)
    );
}

/// The Keccak-_p_\[1600,14\] permutation.
///
/// ```rust
/// use keccak_p::keccak_p1600_14;
///
/// let mut lanes = [0u64; 25];
/// keccak_p1600_14(&mut lanes);
///
/// print!("[");
/// for v in lanes {
///     print!("0x{:08X}, ", v);
/// }
/// println!("]");
///
/// assert_eq!(lanes, [
///     0x93055C6025AE39F4, 0xFFBA77C16772F3A5, 0x51ED61D5555A51EA, 0x5358D0835CE5F0CC, 0xAE01AC77DF72FB3E,
///     0xE0767F85AC129A50, 0xCE0B02509CD0F064, 0xF7CFCE4BF6F57FCA, 0xAC81B1F1908316E1, 0x4DEBF0A389590553,
///     0x86092CE8FA183B03, 0x4E591644A4D9C2AD, 0x2AB0A369261CA0DD, 0xD3198C101AA84596, 0x6101474A582C10CE,
///     0x861648623AE90D39, 0xE465E43209057AD6, 0x3BB6AA81191AE532, 0xE9545E5542A6B7E2, 0x4231AE72A53C78BC,
///     0xA78626648D818094, 0x415594F60CFCCD6E, 0xE0B098A311C22888, 0x6F17D285E131AEE8, 0x43F8D0E199901150,
/// ]);
/// ```
#[inline(always)]
#[allow(unused_assignments)]
pub fn keccak_p1600_14(lanes: &mut [u64; 25]) {
    iter_rounds!(
        lanes,
        (0x0000000080008009, 0x000000008000000a),
        (0x000000008000808b, 0x800000000000008b),
        (0x8000000000008089, 0x8000000000008003),
        (0x8000000000008002, 0x8000000000000080),
        (0x000000000000800a, 0x800000008000000a),
        (0x8000000080008081, 0x8000000000008080),
        (0x0000000080000001, 0x8000000080008008)
    );
}

/// The Keccak-_f_\[1600\] permutation.
///
/// ```rust
/// use keccak_p::keccak_f1600;
///
/// let mut lanes = [0u64; 25];
/// keccak_f1600(&mut lanes);
///
/// assert_eq!(lanes, [
///     0xF1258F7940E1DDE7, 0x84D5CCF933C0478A, 0xD598261EA65AA9EE, 0xBD1547306F80494D,
///     0x8B284E056253D057, 0xFF97A42D7F8E6FD4, 0x90FEE5A0A44647C4, 0x8C5BDA0CD6192E76,
///     0xAD30A6F71B19059C, 0x30935AB7D08FFC64, 0xEB5AA93F2317D635, 0xA9A6E6260D712103,
///     0x81A57C16DBCF555F, 0x43B831CD0347C826, 0x01F22F1A11A5569F, 0x05E5635A21D9AE61,
///     0x64BEFEF28CC970F2, 0x613670957BC46611, 0xB87C5A554FD00ECB, 0x8C3EE88A1CCF32C8,
///     0x940C7922AE3A2614, 0x1841F924A2C509E4, 0x16F53526E70465C2, 0x75F644E97F30A13B,
///     0xEAF1FF7B5CECA249,
/// ]);
/// ```
#[inline(always)]
#[allow(unused_assignments)]
pub fn keccak_f1600(lanes: &mut [u64; 25]) {
    iter_rounds!(
        lanes,
        (0x0000000000000001, 0x0000000000008082),
        (0x800000000000808a, 0x8000000080008000),
        (0x000000000000808b, 0x0000000080000001),
        (0x8000000080008081, 0x8000000000008009),
        (0x000000000000008a, 0x0000000000000088),
        (0x0000000080008009, 0x000000008000000a),
        (0x000000008000808b, 0x800000000000008b),
        (0x8000000000008089, 0x8000000000008003),
        (0x8000000000008002, 0x8000000000000080),
        (0x000000000000800a, 0x800000008000000a),
        (0x8000000080008081, 0x8000000000008080),
        (0x0000000080000001, 0x8000000080008008)
    );
}
