# kdl-schema

[KDL Schema](https://github.com/kdl-org/kdl/blob/1.0.0/SCHEMA-SPEC.md) types and parsing.

## examples

### building a schema

```rust
use kdl_schema::*;
let schema = Schema {
    document: Document {
        info: Info {
            title: vec![TextValue {
                text: "Sample Schema".to_string(),
                lang: Some("en".to_string()),
            }],
            ..Info::default()
        },
        nodes: vec![Node {
            name: Some("name".to_string()),
            values: vec![Value {
                validations: vec![Validation::Type("string".to_string())],
                ..Value::default()
            }],
            ..Node::default()
        }]
    }
};
println!("{:?}", schema);
```

### parsing a schema KDL

```rust
#[cfg(feature = "parse-knuffel")] {
    let schema_kdl = r#"
document {
    info {
        title "Sample Schema" lang="en"
        description "An example schema" lang="en"
        author "boringcactus"
    }
    node "name" {
        value {
            type "string"
        }
    }
    node "age" {
        value {
            type "number"
        }
    }
}
"#;
    let _matching_document = r#"
name "Joe"
age 69
"#;
    let schema = kdl_schema::Schema::parse(schema_kdl).unwrap();
    assert_eq!(schema.document.info.title[0].text, "Sample Schema");
}
```

### referencing the schema schema

```rust
assert_eq!(kdl_schema::SCHEMA_SCHEMA.document.info.title[0].text, "KDL Schema");
```

## cargo features

- `parse-knuffel` - expose `Schema::parse`, powered by [the `knuffel` crate](https://crates.io/crates/knuffel)

## conditions blocking version 1.0.0

- consistently good documentation
- merge ref target over ref source if ref source has anything else
- good API for parsing from a file
- types actually match the schema (currently I'm omitting several things because the schema schema doesn't use them)
- ergonomic builder API to define a schema in Rust in a non-ugly way
- can generate KDL from schema object in Rust
- can choose kdl or knuffel as parser
- make sure `required false` in a `prop` works properly
- validate the schema at parse time (ensure that e.g. refs are resolvable, IDs are unique)
- make sure enums work with int values
- allow rich types for language tags, modification dates, etc

## license

[Blue Oak Model License 1.0.0](https://blueoakcouncil.org/license/1.0.0).
