extern crate kanorg;
use kanorg::KanOrgBoard;

#[cfg(test)]
mod integration_tests {
    extern crate anyhow;

    use super::*;
    use anyhow::Result;
    use std::io::prelude::*;
    use std::path::PathBuf;

    const KO_BASE_DIR: &'static str = ".kanorg.d";
    const KO_CFG_FILE: &'static str = "config";
    const KO_ACTIVE_TASKS_DIR: &'static str = "active.d";
    const KO_ARCHIVED_TASKS_DIR: &'static str = "archive.d";

    fn setup_workspace(copy_example_config: bool) -> Result<tempfile::TempDir> {
        let temp_dir = tempfile::tempdir()?;

        if copy_example_config {
            fs_extra::copy_items(
                &[PathBuf::from(env!("PWD"))
                    .join("examples")
                    .join(KO_BASE_DIR)],
                temp_dir.path(),
                &fs_extra::dir::CopyOptions::new(),
            )?;
        }

        Ok(temp_dir)
    }

    #[test]
    fn create_add_move_show() -> Result<()> {
        let tmpdir = setup_workspace(false)?;
        let temp_dir = tmpdir.path().to_path_buf();

        let base_dir = temp_dir.join(KO_BASE_DIR);
        let config_file = base_dir.join(KO_CFG_FILE);
        let active_tasks_dir = base_dir.join(KO_ACTIVE_TASKS_DIR);
        let archive_tasks_dir = base_dir.join(KO_ARCHIVED_TASKS_DIR);

        // --------------------------- //
        // Configuration tree creation //
        // ----------------------------//

        // First we wont have the configuration directory
        assert!(!base_dir.is_dir());

        KanOrgBoard::create(&temp_dir.to_str().unwrap())?;

        // Ensure everything is created
        assert!(config_file.is_file());
        assert!(active_tasks_dir.is_dir());
        assert!(archive_tasks_dir.is_dir());

        // ------------------ //
        // Board modification //
        // -------------------//

        // The new task file still does not exist
        assert!(!active_tasks_dir.join("1").is_file());

        let mut k = KanOrgBoard::new(&temp_dir).expect("There was an error creatint the board.");

        assert!(k.add("Hello, I'm a new task", Some("todo")).is_ok());

        // The file is created without problems
        assert!(active_tasks_dir.join("1").is_file());

        // Move the tawk to other workflow
        assert!(k.relocate("1", Some("doing")).is_ok());

        // ---------------- //
        // Board displaying //
        // -----------------//

        let expected_output = "\
        DOING\n   \
        1 Hello, I'm a new task\n\
        ";
        let mut stdout_capture = std::io::Cursor::new(Vec::new());

        assert!(k.show(Some("doing"), &mut stdout_capture).is_ok());

        stdout_capture.seek(std::io::SeekFrom::Start(0)).unwrap();
        // Ensure correct output
        assert_eq!(
            expected_output,
            String::from_utf8(stdout_capture.into_inner())?,
        );

        Ok(())
    }
}
