extern crate anyhow;
extern crate clap;

use anyhow::Result;
use clap::{App, Arg, SubCommand};
use kanorg::KanOrgBoard;
use std::{env, process};

/// Returns the argument matches from the command line
pub fn get_matches<'a>() -> clap::ArgMatches<'a> {
    App::new(env!("CARGO_PKG_NAME"))
        .version(env!("CARGO_PKG_VERSION"))
        .author(env!("CARGO_PKG_AUTHORS"))
        .about(env!("CARGO_PKG_DESCRIPTION"))
        .subcommands(vec![
            SubCommand::with_name("create")
                .about("Create a default config in the current directory")
                .arg(
                    Arg::with_name("target_dir")
                        .index(1)
                        .help("Target directory to create the new config"),
                ),
            SubCommand::with_name("show")
                .about("Show the KanOrg display")
                .arg(
                    Arg::with_name("workflow")
                        .index(1)
                        .help("Specific workflow to display"),
                ),
            SubCommand::with_name("add")
                .about("Add a new task to the workflow")
                .args(&[
                    Arg::with_name("title")
                        .index(1)
                        .required(true)
                        .help("Define the new task title"),
                    Arg::with_name("workflow")
                        .index(2)
                        .help("Specify the initial task workflow"),
                ]),
            SubCommand::with_name("move")
                .about("Move a task to another workflow")
                .args(&[
                    Arg::with_name("task").index(1).help("Task identifier"),
                    Arg::with_name("workflow")
                        .index(2)
                        .help("New workflow to which move the task to"),
                ]),
            SubCommand::with_name("edit")
                .about("Edit the title or description from a task")
                .arg(Arg::with_name("task").help("Task identifier")),
            SubCommand::with_name("delete")
                .about("Delete a task from any workflow")
                .arg(Arg::with_name("task").help("Task identifier")),
        ])
        .get_matches()
}

/// Collect errors form the subcommands and print them. Then exit the program.
fn print_error_and_exit(result: Result<()>) {
    if let Err(err) = result {
        println!("{}", err);
        process::exit(1);
    };
}

fn main() -> Result<()> {
    let matches = get_matches();

    match matches.subcommand() {
        ("create", Some(subcommand)) => {
            print_error_and_exit(KanOrgBoard::create(
                subcommand.value_of("target_dir").unwrap_or("."),
            ));
        }
        _ => {
            let mut board = KanOrgBoard::new(&env::current_dir()?)?;

            match matches.subcommand() {
                ("show", Some(subcommand)) => {
                    print_error_and_exit(
                        board.show(subcommand.value_of("workflow"), &mut std::io::stdout()),
                    );
                }
                ("add", Some(subcommand)) => {
                    print_error_and_exit(board.add(
                        subcommand.value_of("title").unwrap(),
                        subcommand.value_of("workflow"),
                    ));
                }
                ("relocate", Some(subcommand)) => {
                    print_error_and_exit(
                        board
                            .relocate(subcommand.value_of("task"), subcommand.value_of("workflow")),
                    );
                }
                ("edit", Some(subcommand)) => {
                    print_error_and_exit(board.edit(subcommand.value_of("task")));
                }
                ("delete", Some(subcommand)) => {
                    print_error_and_exit(board.delete(subcommand.value_of("task")));
                }
                _ => println!("{}", matches.usage()),
            }
        }
    }

    Ok(())
}
