//! KanOrg allows you easily organize all your tasks from the terminal.
//!
//! This commandline application will help you save everything important and will let you take the
//! control of your work (and not the other way arround).
//!
//! For reference about the main config file checkout the [`Workflows`] struct. Furthermore, if you
//! want to know more about the CLI commands, take a look to the [`KanOrgBoard`] struct.
extern crate anyhow;
extern crate home;
extern crate serde;
extern crate terminal_size;
extern crate toml;

use anyhow::{anyhow, Result};
use serde::{Deserialize, Serialize};
use std::{
    collections::VecDeque,
    fs::{create_dir_all, rename, OpenOptions},
    io::{prelude::*, BufReader, Write},
    path::PathBuf,
};
use terminal_size::Width;

#[cfg(not(test))]
use terminal_size::terminal_size;

#[cfg(test)]
pub fn terminal_size() -> Option<(Width, ())> {
    Some((Width(80), ()))
}

/// Base KanOrg config directory.
const KO_BASE_DIR: &'static str = ".kanorg.d";
/// Configuration file name.
const KO_CFG_FILE: &'static str = "config";
/// Active tasks directory.
const KO_ACTIVE_TASKS_DIR: &'static str = "active.d";
/// Archived tasks directory.
const KO_ARCHIVED_TASKS_DIR: &'static str = "archive.d";

/// Base config file translation.
///
/// In order to have a correct config file, you will need to provide the five fields: `backlog`,
/// `todo`, `doing`, `done` and `last_task`.
///
/// When serialising/deserialising to/from a config file with the [`toml`] package by default, the
/// following format will be found:
///
/// ```plain,no_run
/// backlog = [12, 11, 10, 9]
/// todo = [8]
/// doing = [7, 6]
/// done = [5, 4, 3, 2, 1]
/// last_task = 12
/// ```
///
/// The four workflows plus the last task must be found in order the deserialisation to work
/// properly.
#[derive(Deserialize, Serialize, Debug)]
struct Workflows {
    /// All the tasks that are in the bench, for future pick up.
    pub backlog: VecDeque<u16>,
    /// Current Sprint tasks.
    pub todo: VecDeque<u16>,
    /// The tasks that are currently under development.
    pub doing: VecDeque<u16>,
    /// Completed tasks. This will contain the last 10 completed. The other rest are in the archive.
    pub done: VecDeque<u16>,
    /// Last task ID.
    pub last_task: u16,
}

impl Workflows {
    /// Check if a `task` is contained in any of the workflows.
    ///
    /// # Arguments:
    ///
    /// * `task` - name of the task to search for.
    pub fn task_exists(&self, task: &u16) -> bool {
        self.backlog.contains(task)
            || self.todo.contains(task)
            || self.doing.contains(task)
            || self.done.contains(task)
    }
    /// Obtain the max number of tasks that are in any of the `todo`, `doing` and `done` workflows.
    pub fn main_max_len(&self) -> usize {
        match vec![self.todo.len(), self.doing.len(), self.done.len()]
            .iter()
            .max()
        {
            None => 0,
            Some(max_len) => max_len.clone(),
        }
    }
    /// Add a task to the end of a workflow
    ///
    /// If the selected workflow is `done`, as it has a limit of *5* tasks, the oldest one will be
    /// removed from the workflow.
    ///
    /// # Arguments:
    ///
    /// * `workflow` - name of the column we want to add a task to.
    ///
    /// # Errors:
    ///
    /// If the workflow name does not exist, an error is returned.
    pub fn push_new_task(&mut self, workflow: &str) -> Result<Option<u16>> {
        self.last_task += 1;
        match workflow {
            "backlog" => self.backlog.push_front(self.last_task),
            "todo" => self.todo.push_front(self.last_task),
            "doing" => self.doing.push_front(self.last_task),
            "done" => {
                self.done.push_front(self.last_task);
                if self.done.len() > 5 {
                    return Ok(self.done.pop_back());
                }
            }
            _ => {
                return Err(anyhow!(
                    "No workflow named `{}`. It has to be one of backlog, todo, doing or done.",
                    workflow
                ))
            }
        }
        Ok(None)
    }
    /// Retrieve a workflow
    ///
    /// This is a simple workflow retriever by name.
    ///
    /// # Arguments:
    ///
    /// * `workflow` - name of the column we want to retrieve.
    ///
    /// # Errors:
    ///
    /// If the workflow name does not exist, an error is returned.
    pub fn get_workflow(&self, workflow: &str) -> Result<&VecDeque<u16>> {
        match workflow {
            "backlog" => Ok(&self.backlog),
            "todo" => Ok(&self.todo),
            "doing" => Ok(&self.doing),
            "done" => Ok(&self.done),
            _ => {
                return Err(anyhow!(
                    "No workflow named `{}`. It has to be one of backlog, todo, doing or done.",
                    workflow
                ))
            }
        }
    }
}

/// Kanban organization workflows.
///
/// This utility will allow you create tasks and move them from one workspace to another. Edit and
/// delete them.
pub struct KanOrgBoard {
    /// Workflows found in the configuration file.
    workflows: Workflows,
    /// Task files directory path.
    base_dir: PathBuf,
}

impl KanOrgBoard {
    /// Search for a config file and return it if succesfully found.
    ///
    /// The base file name is `config` and is located inside the base directory `.kanorg.d/`. It
    /// will be searched in all the `hints` directories provided.
    ///
    /// # Arguments:
    ///
    /// * `hints` - paths to search the config file in
    ///
    /// # Errors:
    ///
    /// If the config file was not found, an error will be raised.
    fn search_config_file(hints: &[&PathBuf]) -> Result<PathBuf> {
        let relative_cfg_file = format!("{}/{}", KO_BASE_DIR, KO_CFG_FILE);
        for search_dir in hints {
            let config_file = search_dir.join(&relative_cfg_file);

            if config_file.is_file() {
                return Ok(config_file);
            }
        }

        Err(anyhow!(
            "No configuration file found in any of the following paths: {:#?}\nPlease \
            consider creating a new configuration with the `create` subcommand.",
            hints
        ))
    }
    /// Saves the current [`KanOrgBoard`] workflows into the configuration file.
    ///
    /// This will save all the workspaces as arrays of tasks with the TOML format.
    fn save_config(&self) -> Result<()> {
        let config_file = self.base_dir.join(KO_CFG_FILE);
        // println!("DEBUG: saving config to {:?} file", &config_file);

        OpenOptions::new()
            .write(true)
            .truncate(true)
            .open(&config_file)?
            .write_all(toml::to_string(&self.workflows)?.as_bytes())?;

        // println!("DEBUG: saving config to {:?} file... done", &config_file);
        Ok(())
    }
    /// Check if a task is part of the workflow.
    ///
    /// This method will check if the task appears in any of the workflows.
    ///
    /// # Arguments:
    ///
    /// * `task` - the task to check if it is active.
    fn task_exists(&self, task: &u16) -> bool {
        self.workflows.task_exists(task)
    }
    /// Read the first line of a file.
    ///
    /// This method will serve as an utility for the [`KanOrgBoard::show()`] method, opening each
    /// task to read the title (first line). Note that this function will strip the leading `# ` and
    /// the trailing whitespaces from the line.
    ///
    /// # Arguments:
    ///
    /// * `file_path` - path to the file to perform a read.
    ///
    /// # Examples:
    ///
    /// Having the file `/some/file`:
    ///
    /// ```plain,no_run
    /// # This is the title of the 1st task
    ///
    /// this is a description
    /// ```
    ///
    /// We can run with the following:
    ///
    /// ```plain,no_run
    /// let file = PathBuf::from("/some/file");
    /// assert_eq!(
    ///     KanOrgBoard::read_first_line_cleaned(file)?,
    ///     "This is the title of the 1st task".to_owned(),
    /// );
    /// ```
    fn read_first_line_cleaned(file_path: PathBuf) -> Result<String> {
        let mut first_line: String = String::new();
        BufReader::new(OpenOptions::new().read(true).open(file_path)?)
            .read_line(&mut first_line)?;
        Ok(first_line.strip_prefix("# ").unwrap().trim().to_owned())
    }
    /// Format the task title given the task ID.
    ///
    /// Making use of the [`KanOrgBoard::read_first_line_cleaned`] function, the task's title will
    /// be pretty printed along with the task's ID.
    ///
    /// # Arguments:
    ///
    /// * `task` - ID of the task to format its title.
    ///
    /// * `max_size` - maximun size of title at which the string is trimmed.
    ///
    /// # Examples:
    ///
    /// Having the file named `1` located under `$PWD/.kanorg.d/active.d/`:
    ///
    /// ```plain,no_run
    /// # This is the title of the 1st task
    ///
    /// this is a description
    /// ```
    ///
    /// We can run with the following:
    ///
    /// ```plain,no_run
    /// let kanban = KanOrgBoard::new()?;
    /// assert_eq!(
    ///     kanban.format_task_title(Some(1), 40 as usize)?,
    ///     "This is the title of the 1st task".to_owned()
    /// );
    /// assert_eq!(
    ///     kanban.format_task_title(Some(1), 10 as usize)?,
    ///     "This is th".to_owned()
    /// );
    /// assert_eq!(
    ///     kanban.format_task_title(Some(12312313), 40 as usize)?,
    ///     String::new(),
    /// );
    /// ```
    fn format_task_title(&self, task: Option<&u16>, max_size: usize) -> Result<String> {
        match task {
            Some(task_id) => Ok(format!(
                "{1:>4} {2:<.0$}",
                max_size - 6,
                task_id,
                Self::read_first_line_cleaned(
                    self.base_dir
                        .join(KO_ACTIVE_TASKS_DIR)
                        .join(task_id.to_string())
                )?
            )),
            None => Ok(String::new()),
        }
    }
    /// Creates a new instance of [`KanOrgBoard`] loading a configuration file.
    ///
    /// The config file is searched in the current directory or the default user
    /// folder. Checkout [`KanOrgBoard::search_config_file()`] to know more about
    /// how it is searched. Once the config file is located, it is read and the
    /// instance attributes populated. The module [`toml`] will be used to
    /// deserialize the file contents. Check the [`Workflows`] struct to know
    /// more about the config file format.
    pub fn new(current_working_dir: &PathBuf) -> Result<Self> {
        let config_file =
            Self::search_config_file(&[current_working_dir, &home::home_dir().unwrap()])?;
        // println!("DEBUG: loading config from {:?} file", &config_file);
        let base_dir = config_file.parent().unwrap().to_path_buf();

        create_dir_all(base_dir.join(KO_ACTIVE_TASKS_DIR))?;
        create_dir_all(base_dir.join(KO_ARCHIVED_TASKS_DIR))?;

        let mut contents: String = String::new();
        OpenOptions::new()
            .read(true)
            .open(&config_file)?
            .read_to_string(&mut contents)?;
        let workflows: Workflows = match toml::from_str(&contents) {
            Ok(file_contents) => file_contents,
            Err(err) => {
                return Err(anyhow!(
                    "Could not properly parse the config file {:?}:\n\t{}",
                    config_file,
                    err
                ))
            }
        };

        Ok(Self {
            workflows: workflows,
            base_dir: base_dir,
        })
    }
    /// Create the base configuration in the chosen directory.
    ///
    /// This function will create the following structure:
    ///
    /// ```plain,no_run
    /// .kanorg.d/
    /// |-- active.d
    /// |-- archive.d
    /// `-- config
    /// ```
    ///
    /// Being:
    ///
    /// * `.kanorg.d` - the base directory where the configuration and tasks are contained.
    ///
    /// * `.kanorg.d/config` - configuration file. The file will have the contents:
    ///
    ///     ```plain,no_run
    ///     backlog = [<task1>, <task2>, ..., <taskn>]
    ///     todo = [<task1>, <task2>, ..., <taskn>]
    ///     doing = [<task1>, <task2>, ..., <taskn>]
    ///     done = [<task1>, <task2>, ..., <taskn>]
    ///     last_task = <last task ID>
    ///     ```
    ///
    /// * `.kanorg.d/active.d` - directory in where the active task files (the ones found in
    ///   `backlog`, `todo`, `doing` or `done`) are located.
    ///
    /// * `.kanorg.d/archive.d` - in this other directory, the popped task files from the `done`
    ///   workflow will be found.
    ///
    /// # Arguments:
    ///
    /// * `target_dir` - the target thir you want to create the base configuration. If no target
    ///   path is specified, the current directory will be used.
    pub fn create(target_dir: &str) -> Result<()> {
        // println!("DEBUG: creating config in {:?} directory", &target_dir);
        let target_base_dir = PathBuf::from(target_dir).canonicalize()?.join(KO_BASE_DIR);

        create_dir_all(target_base_dir.join(KO_ACTIVE_TASKS_DIR))?;
        create_dir_all(target_base_dir.join(KO_ARCHIVED_TASKS_DIR))?;

        let config_file = target_base_dir.join(KO_CFG_FILE);
        if !config_file.is_file() {
            OpenOptions::new()
                .write(true)
                .read(true)
                .create(true)
                .open(&config_file)?
                .write_all(
                    "backlog = []\ntodo = []\ndoing = []\ndone = []\nlast_task = 0\n".as_bytes(),
                )?;
        }

        Ok(())
    }
    /// Provide an organized view of the current tasks.
    ///
    /// This method will print the three main workflows (todo, doing and done)
    /// in a table style output. Also after finishing the main tasks, the
    /// backlog will be printed alone with a maximun of 5 tasks.
    ///
    /// This is an example output you can get:
    ///
    /// ```plain,no_run
    /// |          TODO           |          DOING          |          DONE           |
    /// |-------------------------|-------------------------|-------------------------|
    /// |  15 Task fifteen title  |  13 Task thirteen title |  29 Esta es una mísera  |
    /// |  12 Task twelve title   |  10 Task ten title      |  11 Task eleven title   |
    /// |  14 Task fourteen title |                         |   9 Task nine title     |
    /// |                         |                         |   6 Task six title      |
    /// |                         |                         |   8 Task eight title    |
    ///
    /// BACKLOG
    ///   26 este es el nuevo titulo
    ///   25 Task twenty-five title
    ///   24 Task twnety-four title
    ///   23 Task twenty-three title
    ///   22 Task twenty-two title
    /// WARNING: The backlog has been trimmed. Run `ko show backlog` to see all the backlog tasks.
    /// ```
    ///
    /// The program will addapt to the terminal size and will print the columns
    /// in concordance.
    ///
    /// # Arguments:
    ///
    /// * `workflow` - name of the workflow to show. If this parameter is
    ///   provided, the output will be similar to the previous backlog list:
    ///
    ///     ```plain,no_run
    ///     TODO
    ///       15 Task fifteen title
    ///       12 Task twelve title
    ///       14 Task fourteen title
    ///     ```
    ///
    /// # Example usage:
    ///
    /// To show all the available workflows:
    ///
    /// ```shell,no_run
    /// ko show
    /// ```
    ///
    /// To show only one specific workflow:
    ///
    /// ```shell,no_run
    /// ko show todo
    /// ```
    pub fn show<W: Write>(&self, workflow: Option<&str>, writer: &mut W) -> Result<()> {
        match terminal_size() {
            None => return Err(anyhow!("Could not guess the terminal size")),
            Some((Width(width), _)) => {
                if let Some(single_workflow) = workflow {
                    let selected_workflow = self.workflows.get_workflow(single_workflow)?;
                    let selected_workflow_name = single_workflow.to_owned().to_uppercase();
                    let mut selected_workflow_iter = selected_workflow.iter();
                    writeln!(writer, "{}", selected_workflow_name)?;
                    for _ in 0..selected_workflow.len() {
                        writeln!(
                            writer,
                            "{}",
                            self.format_task_title(selected_workflow_iter.next(), width as usize)?
                        )?;
                    }
                } else {
                    // We need one separator character per workflow plus the final one
                    let column_gap: usize = (width as usize - (3 + 1)) / 3;

                    let mut backlog_iter = self.workflows.backlog.iter();
                    let mut todo_iter = self.workflows.todo.iter();
                    let mut doing_iter = self.workflows.doing.iter();
                    let mut done_iter = self.workflows.done.iter();

                    writeln!(
                        writer,
                        "|{:^3$}|{:^3$}|{:^3$}|",
                        "TODO", "DOING", "DONE", column_gap
                    )?;
                    writeln!(writer, "|{0}|{0}|{0}|", str::repeat("-", column_gap))?;
                    for _ in 0..self.workflows.main_max_len() {
                        writeln!(
                            writer,
                            "|{:3$}|{:3$}|{:3$}|",
                            self.format_task_title(todo_iter.next(), column_gap)?,
                            self.format_task_title(doing_iter.next(), column_gap)?,
                            self.format_task_title(done_iter.next(), column_gap)?,
                            column_gap,
                        )?;
                    }
                    writeln!(writer, "\nBACKLOG")?;
                    let backlog_len = self.workflows.backlog.len();
                    let backlog_max_len = if backlog_len <= 5 { backlog_len } else { 5 };
                    for _ in 0..backlog_max_len {
                        writeln!(
                            writer,
                            "{}",
                            self.format_task_title(backlog_iter.next(), width as usize)?
                        )?;
                    }
                    if backlog_len > 5 {
                        writeln!(
                            writer,
                            "WARNING: The backlog has been trimmed. Run `ko show backlog` to see \
                            all the backlog tasks.",
                        )?;
                    }
                }
            }
        }

        Ok(())
    }
    /// Add a new task to the choosen workflow.
    ///
    /// This method will add a new task to the desired workflow. Moreover, a file will be created
    /// with the title as the first line.
    ///
    /// Note that when adding a new task to the `done` workflow, if there are already 5 tasks, the
    /// oldest one will be phisically moved to the `archive` and removed from the mentioned
    /// workflow.
    ///
    /// # Arguments:
    ///
    /// * `title` - summary of the new task. It will be placed in the first line of the task file.
    ///
    /// * `workflow` - initial column to put the new task. If no workflow is specified, it will be
    ///   placed in `backlog` (default workflow).
    ///
    /// # Example usage:
    ///
    /// To add a new task into the backlog introducing:
    ///
    /// ```shell,no_run
    /// ko add "new task fancy title"
    /// ```
    ///
    /// To add a new task to the workflow `todo`:
    ///
    /// ```shell,no_run
    /// ko add "new task fancy title" todo
    /// ```
    pub fn add(&mut self, title: &str, workflow: Option<&str>) -> Result<()> {
        let task_title: String = title.to_owned();

        let selected_workflow_name = match workflow {
            Some(workflow) => workflow,
            None => "backlog",
        };

        let active_tasks_dir = self.base_dir.join(KO_ACTIVE_TASKS_DIR);
        let archive_tasks_dir = self.base_dir.join(KO_ARCHIVED_TASKS_DIR);

        if let Some(bench_task) = self.workflows.push_new_task(selected_workflow_name)? {
            // There is an overflow in the `done` workflow. We will move the
            // popped task to the `archive`
            rename(
                active_tasks_dir.join(bench_task.to_string()),
                archive_tasks_dir.join(bench_task.to_string()),
            )?;
        }

        let last_task_path = active_tasks_dir.join(self.workflows.last_task.to_string());
        match OpenOptions::new()
            .create(true)
            .write(true)
            .open(&last_task_path)
        {
            Ok(mut writing_file) => {
                if let Err(err) = writing_file.write_all(format!("# {}\n", task_title).as_bytes()) {
                    return Err(anyhow!(
                        "There was a problem writing the file {:?}: {}",
                        last_task_path,
                        err
                    ));
                }
            }
            Err(err) => {
                return Err(anyhow!(
                    "There was a problem opening the file {:?}: {}",
                    last_task_path,
                    err
                ))
            }
        }

        self.save_config()?;
        Ok(())
    }
    pub fn relocate(&mut self, task: Option<&str>, workflow: Option<&str>) -> Result<()> {
        self.save_config()?;
        Ok(())
    }
    pub fn edit(&self, task: Option<&str>) -> Result<()> {
        Ok(())
    }
    pub fn delete(&mut self, task: Option<&str>) -> Result<()> {
        self.save_config()?;
        Ok(())
    }
}

#[cfg(test)]
mod unit_tests {
    use super::*;

    mod workflows {
        use super::*;

        #[test]
        fn task_exists() {
            let w: Workflows = Workflows {
                backlog: vec![12, 11, 10, 9].into_iter().collect(),
                todo: vec![8].into_iter().collect(),
                doing: vec![7, 6].into_iter().collect(),
                done: vec![5, 4, 3, 2, 1].into_iter().collect(),
                last_task: 12,
            };
            assert!(w.task_exists(&7u16));
            assert!(!w.task_exists(&200u16));
        }

        #[test]
        fn main_max_len() {
            let w = Workflows {
                backlog: vec![12, 11, 10, 9].into_iter().collect(),
                todo: vec![8].into_iter().collect(),
                doing: vec![7, 6].into_iter().collect(),
                done: vec![5, 4, 3, 2, 1].into_iter().collect(),
                last_task: 12,
            };
            assert_eq!(w.main_max_len(), 5);

            let w = Workflows {
                backlog: VecDeque::new(),
                todo: VecDeque::new(),
                doing: VecDeque::new(),
                done: VecDeque::new(),
                last_task: 0,
            };
            assert_eq!(w.main_max_len(), 0);
        }

        #[test]
        fn push_new_task() -> Result<()> {
            let mut w = Workflows {
                backlog: vec![12, 11, 10, 9].into_iter().collect(),
                todo: vec![8].into_iter().collect(),
                doing: vec![7, 6].into_iter().collect(),
                done: vec![5, 4, 3, 2, 1].into_iter().collect(),
                last_task: 12,
            };
            assert_eq!(
                format!("{:?}", w),
                "Workflows { \
                    backlog: [12, 11, 10, 9], \
                    todo: [8], \
                    doing: [7, 6], \
                    done: [5, 4, 3, 2, 1], \
                    last_task: 12 \
                }"
                .to_owned()
            );
            w.push_new_task("done")?;
            assert_eq!(
                format!("{:?}", w),
                "Workflows { \
                    backlog: [12, 11, 10, 9], \
                    todo: [8], \
                    doing: [7, 6], \
                    done: [13, 5, 4, 3, 2], \
                    last_task: 13 \
                }"
                .to_owned()
            );
            Ok(())
        }

        #[test]
        fn get_workflow() -> Result<()> {
            let w: Workflows = Workflows {
                backlog: vec![12, 11, 10, 9].into_iter().collect(),
                todo: vec![8].into_iter().collect(),
                doing: vec![7, 6].into_iter().collect(),
                done: vec![5, 4, 3, 2, 1].into_iter().collect(),
                last_task: 12,
            };

            let workflow = w.get_workflow("doing");
            assert_eq!(workflow?.as_slices().0, &[7, 6][..]);

            let workflow = w.get_workflow("invented");
            assert!(workflow.is_err());
            Ok(())
        }
    }

    mod kanorgboard {
        extern crate fs_extra;
        extern crate tempfile;

        use super::*;

        fn setup_workspace(copy_example_config: bool) -> Result<tempfile::TempDir> {
            let temp_dir = tempfile::tempdir()?;

            if copy_example_config {
                fs_extra::copy_items(
                    &[PathBuf::from(env!("PWD"))
                        .join("examples")
                        .join(KO_BASE_DIR)],
                    temp_dir.path(),
                    &fs_extra::dir::CopyOptions::new(),
                )?;
            }

            Ok(temp_dir)
        }

        #[test]
        fn search_config_file() -> Result<()> {
            let tmpkeep = setup_workspace(true)?;
            let temp_dir = tmpkeep.path();
            let found_file = KanOrgBoard::search_config_file(&[&temp_dir.to_path_buf()])?;

            assert_eq!(
                found_file,
                temp_dir.join(format!("{}/{}", KO_BASE_DIR, KO_CFG_FILE))
            );

            Ok(())
        }

        #[test]
        fn save_config() -> Result<()> {
            let tmpkeep = setup_workspace(false)?;
            let temp_dir_base = tmpkeep.path().join(KO_BASE_DIR);
            create_dir_all(&temp_dir_base)?;
            let temp_config_file = temp_dir_base.join(KO_CFG_FILE);
            OpenOptions::new()
                .create(true)
                .write(true)
                .truncate(true)
                .open(&temp_config_file)?;

            KanOrgBoard {
                workflows: Workflows {
                    backlog: vec![5].into_iter().collect(),
                    todo: vec![4, 3].into_iter().collect(),
                    doing: vec![2].into_iter().collect(),
                    done: vec![1].into_iter().collect(),
                    last_task: 5,
                },
                base_dir: temp_dir_base,
            }
            .save_config()?;

            let mut file_contents = String::new();

            OpenOptions::new()
                .read(true)
                .open(&temp_config_file)?
                .read_to_string(&mut file_contents)?;

            assert_eq!(
                file_contents,
                "backlog = [5]\ntodo = [4, 3]\ndoing = [2]\ndone = [1]\nlast_task = 5\n"
            );

            Ok(())
        }

        #[test]
        fn task_exists() -> Result<()> {
            let tmpkeep = setup_workspace(false)?;
            let temp_dir = tmpkeep.path();

            let ko = KanOrgBoard {
                workflows: Workflows {
                    backlog: vec![5].into_iter().collect(),
                    todo: vec![4, 3].into_iter().collect(),
                    doing: vec![2].into_iter().collect(),
                    done: vec![1].into_iter().collect(),
                    last_task: 5,
                },
                base_dir: temp_dir.to_path_buf(),
            };

            assert!(ko.task_exists(&5u16));

            assert!(!ko.task_exists(&12345u16));

            Ok(())
        }

        #[test]
        fn read_first_line_cleaned() -> Result<()> {
            let tmpkeep = setup_workspace(false)?;
            let temp_file_path = tmpkeep.path().join("some_file");

            OpenOptions::new()
                .create(true)
                .write(true)
                .open(&temp_file_path)?
                .write_all("# This is a test file\n\nthe description\n".as_bytes())?;

            assert_eq!(
                KanOrgBoard::read_first_line_cleaned(temp_file_path)?,
                "This is a test file".to_owned()
            );

            Ok(())
        }

        #[test]
        fn format_task_title() -> Result<()> {
            let tmpkeep = setup_workspace(true)?;
            let k = KanOrgBoard::new(&tmpkeep.path().to_path_buf())?;

            assert_eq!(
                k.format_task_title(Some(&8u16), 50 as usize)?,
                "   8 This is the task 8 sample title".to_owned(),
            );

            Ok(())
        }

        #[test]
        fn create() -> Result<()> {
            let tmpkeep = setup_workspace(false)?;
            let temp_dir = tmpkeep.path().to_path_buf();
            let temp_base_dir = temp_dir.join(KO_BASE_DIR);

            assert!(!&temp_base_dir.exists());

            KanOrgBoard::create(&temp_dir.to_str().unwrap())?;

            assert!(temp_base_dir.join(KO_CFG_FILE).is_file());
            assert!(temp_base_dir.join(KO_ACTIVE_TASKS_DIR).is_dir());
            assert!(temp_base_dir.join(KO_ARCHIVED_TASKS_DIR).is_dir());

            Ok(())
        }

        #[test]
        fn show() -> Result<()> {
            let tmpkeep = setup_workspace(true)?;
            let temp_dir = tmpkeep.path().to_path_buf();

            let k = KanOrgBoard::new(&temp_dir)?;
            let expected_output = "\
            |          TODO           |          DOING          |          DONE           |\n\
            |-------------------------|-------------------------|-------------------------|\n\
            |  18 This is the task 18 |  14 This is the task 14 |  12 This is the task 12 |\n\
            |  17 This is the task 17 |  13 This is the task 13 |  11 This is the task 11 |\n\
            |  16 This is the task 16 |                         |  10 This is the task 10 |\n\
            |  15 This is the task 15 |                         |   9 This is the task 9  |\n\
            |                         |                         |   8 This is the task 8  |\n\
            \n\
            BACKLOG\n  \
              25 This is the task 25 sample title\n  \
              24 This is the task 24 sample title\n  \
              23 This is the task 23 sample title\n  \
              22 This is the task 22 sample title\n  \
              21 This is the task 21 sample title\n\
            WARNING: The backlog has been trimmed. Run `ko show backlog` to see all the \
            backlog tasks.\n";
            let mut stdout_capture = std::io::Cursor::new(Vec::new());

            k.show(None, &mut stdout_capture)?;

            stdout_capture.seek(std::io::SeekFrom::Start(0)).unwrap();

            assert_eq!(
                expected_output,
                String::from_utf8(stdout_capture.into_inner())?,
            );

            Ok(())
        }

        #[test]
        fn show_single() -> Result<()> {
            let tmpkeep = setup_workspace(true)?;
            let temp_dir = tmpkeep.path().to_path_buf();

            let k = KanOrgBoard::new(&temp_dir)?;
            let expected_output = "\
            DONE\n  \
            12 This is the task 12 sample title\n  \
            11 This is the task 11 sample title\n  \
            10 This is the task 10 sample title\n   \
            9 This is the task 9 sample title\n   \
            8 This is the task 8 sample title\n\
            ";
            let mut stdout_capture = std::io::Cursor::new(Vec::new());

            k.show(Some("done"), &mut stdout_capture)?;

            stdout_capture.seek(std::io::SeekFrom::Start(0)).unwrap();

            assert_eq!(
                expected_output,
                String::from_utf8(stdout_capture.into_inner())?,
            );

            Ok(())
        }

        #[test]
        fn add() -> Result<()> {
            let tmpkeep = setup_workspace(true)?;
            let temp_dir = tmpkeep.path().to_path_buf();

            let mut k = KanOrgBoard::new(&temp_dir)?;
            k.add("some task title", Some("doing"))?;

            let last_task = k.workflows.last_task;

            assert!(temp_dir
                .join(format!(
                    "{}/{}/{}",
                    KO_BASE_DIR, KO_ACTIVE_TASKS_DIR, last_task,
                ))
                .is_file());

            Ok(())
        }

        #[test]
        fn add_overflow() -> Result<()> {
            let tmpkeep = setup_workspace(true)?;
            let temp_dir = tmpkeep.path().to_path_buf();

            let mut k = KanOrgBoard::new(&temp_dir)?;

            let oldest_done_task = k.workflows.done[4];

            k.add("some task title", Some("done"))?;

            assert!(temp_dir
                .join(format!(
                    "{}/{}/{}",
                    KO_BASE_DIR, KO_ARCHIVED_TASKS_DIR, oldest_done_task,
                ))
                .is_file());

            Ok(())
        }
    }
}
