use itertools::multizip;
use kalast::{Object3D, MINIMUM_PRECISION};
use std::path::Path;

#[test]
fn parse_simple_facet() {
    // Single triangulate facet centered 2 meter side
    let obj = Object3D::new(Path::new("rsc/obj/facet.obj"));

    // Checking it's only one facet
    assert_eq!(obj.number_faces(), 1);

    // Checking vertices.
    let vertices = obj.vertices();
    let expected_vertices_components = tool::List::from_column_slice(&[
        -0.666667, 0.666667, 0.0, -0.666667, -1.333333, 0.0, 1.333333, 0.666667, 0.,
    ]);

    for (component, expected_component) in
        multizip((vertices.iter(), expected_vertices_components.iter()))
    {
        assert!(relative_eq!(
            component,
            expected_component,
            epsilon = MINIMUM_PRECISION
        ));
    }

    // Checking face.
    let face = obj.face_iter_all().next().unwrap();
    let expected_face_components =
        tool::List::from_column_slice(&[0., 0., 0., 0., 0., 1., 0., 0., 0., 2.]);
    for (component, expected_component) in multizip((face.iter(), expected_face_components.iter()))
    {
        assert!(relative_eq!(
            component,
            expected_component,
            epsilon = MINIMUM_PRECISION
        ));
    }
}

#[test]
fn parse_complex_asteroid() {
    let path = Path::new("rsc/obj/sphere_light.obj");
    let mut obj = Object3D::new(path);
    let number_faces = obj.number_faces();

    // test vertices
    assert_eq!((3, 98), obj.vertices().shape());

    let first_vertex = obj.vertices().column(0);
    let mut first_it = first_vertex.iter();
    assert!(relative_eq!(
        *first_it.next().unwrap(),
        128.970123,
        epsilon = MINIMUM_PRECISION
    ));
    assert!(relative_eq!(
        *first_it.next().unwrap(),
        54.452602,
        epsilon = MINIMUM_PRECISION
    ));
    assert!(relative_eq!(
        *first_it.next().unwrap(),
        484.155609,
        epsilon = MINIMUM_PRECISION
    ));

    let last_vertex = obj.vertices().column(obj.vertices().ncols() - 1);
    let mut last_it = last_vertex.iter();
    assert!(relative_eq!(
        *last_it.next().unwrap(),
        355.764282,
        epsilon = MINIMUM_PRECISION
    ));
    assert!(relative_eq!(
        *last_it.next().unwrap(),
        -38.189651,
        epsilon = MINIMUM_PRECISION
    ));
    assert!(relative_eq!(
        *last_it.next().unwrap(),
        -358.453003,
        epsilon = MINIMUM_PRECISION,
    ));

    // test faces
    assert_eq!((3, 192), obj.faces().shape());

    let first_face = obj.faces().column(0);
    let mut first_it = first_face.iter();
    assert_eq!(*first_it.next().unwrap(), 1);
    assert_eq!(*first_it.next().unwrap(), 7);
    assert_eq!(*first_it.next().unwrap(), 2);

    let last_face = obj.faces().column(number_faces - 1);
    let mut last_it = last_face.iter();
    assert_eq!(*last_it.next().unwrap(), 4);
    assert_eq!(*last_it.next().unwrap(), 96);
    assert_eq!(*last_it.next().unwrap(), 95);

    obj.recompute_faces();

    // test centers
    assert_eq!((3, 192), obj.centers().shape());

    let first_center = obj.centers().column(0).into_owned();
    let mut first_it = first_center.iter();
    assert!(relative_eq!(
        *first_it.next().unwrap(),
        420.69162,
        epsilon = MINIMUM_PRECISION
    ));
    assert!(relative_eq!(
        *first_it.next().unwrap(),
        0.469593,
        epsilon = MINIMUM_PRECISION
    ));
    assert!(relative_eq!(
        *first_it.next().unwrap(),
        252.43807,
        epsilon = MINIMUM_PRECISION,
    ));

    let last_center = obj.centers().column(number_faces - 1).into_owned();
    let mut last_it = last_center.iter();
    assert!(relative_eq!(
        *last_it.next().unwrap(),
        449.064178,
        epsilon = MINIMUM_PRECISION
    ));
    assert!(relative_eq!(
        *last_it.next().unwrap(),
        85.713607,
        epsilon = MINIMUM_PRECISION
    ));
    assert!(relative_eq!(
        *last_it.next().unwrap(),
        -169.05079,
        epsilon = MINIMUM_PRECISION,
    ));

    // // test normals
    assert_eq!((3, 192), obj.normals().shape());

    let first_normal = obj.normals().column(0).into_owned();
    let mut first_it = first_normal.iter();
    assert!(relative_eq!(
        *first_it.next().unwrap(),
        0.892254,
        epsilon = MINIMUM_PRECISION
    ));
    assert!(relative_eq!(
        *first_it.next().unwrap(),
        -0.008871,
        epsilon = MINIMUM_PRECISION
    ));
    assert!(relative_eq!(
        *first_it.next().unwrap(),
        0.451447,
        epsilon = MINIMUM_PRECISION,
    ));

    let last_normal = obj.normals().column(number_faces - 1).into_owned();
    let mut last_it = last_normal.iter();
    assert!(relative_eq!(
        *last_it.next().unwrap(),
        0.958274,
        epsilon = MINIMUM_PRECISION
    ));
    assert!(relative_eq!(
        *last_it.next().unwrap(),
        0.104864,
        epsilon = MINIMUM_PRECISION
    ));
    assert!(relative_eq!(
        *last_it.next().unwrap(),
        -0.265922,
        epsilon = MINIMUM_PRECISION,
    ));

    // // test sphericals
    assert_eq!((3, 192), obj.sphericals().shape());

    let first_spherical = obj.sphericals().column(0).into_owned();
    let mut first_it = first_spherical.iter();
    assert!(relative_eq!(
        *first_it.next().unwrap(),
        0.001116,
        epsilon = MINIMUM_PRECISION
    ));
    assert!(relative_eq!(
        *first_it.next().unwrap(),
        0.54046,
        epsilon = MINIMUM_PRECISION
    ));
    assert!(relative_eq!(
        *first_it.next().unwrap(),
        490.618629,
        epsilon = MINIMUM_PRECISION,
    ));

    let last_spherical = obj.sphericals().column(number_faces - 1).into_owned();
    let mut last_it = last_spherical.iter();
    assert!(relative_eq!(
        *last_it.next().unwrap(),
        0.188603,
        epsilon = MINIMUM_PRECISION
    ));
    assert!(relative_eq!(
        *last_it.next().unwrap(),
        -0.354183,
        epsilon = MINIMUM_PRECISION
    ));
    assert!(relative_eq!(
        *last_it.next().unwrap(),
        487.42551,
        epsilon = MINIMUM_PRECISION,
    ));

    // test areas
    assert_eq!((1, 192), obj.areas().shape(),);
    assert!(relative_eq!(
        15883.672171,
        obj.areas()[0],
        epsilon = MINIMUM_PRECISION
    ));
    assert!(relative_eq!(
        18269.665449,
        obj.areas()[number_faces - 1],
        epsilon = MINIMUM_PRECISION
    ));
}

#[test]
fn parse_equator() {
    // single sphere
    let mut obj = Object3D::new(Path::new("rsc/obj/sphere.obj"));

    // checking number of faces before applying the mask
    assert_eq!(obj.number_faces(), 960);

    // checking the first face before applying the mask
    let face = obj.face_iter_all().next().unwrap();
    assert!(relative_eq!(
        face[0],
        298.997528,
        epsilon = MINIMUM_PRECISION
    ));
    assert!(relative_eq!(
        face[1],
        -41.056031,
        epsilon = MINIMUM_PRECISION
    ));
    assert!(relative_eq!(
        face[2],
        -395.00768,
        epsilon = MINIMUM_PRECISION
    ));
    assert!(relative_eq!(face[3], 0.633159, epsilon = MINIMUM_PRECISION));
    assert!(relative_eq!(
        face[4],
        -0.062361,
        epsilon = MINIMUM_PRECISION
    ));
    assert!(relative_eq!(
        face[5],
        -0.771506,
        epsilon = MINIMUM_PRECISION
    ));
    assert!(relative_eq!(
        face[6],
        -0.136459,
        epsilon = MINIMUM_PRECISION
    ));
    assert!(relative_eq!(
        face[7],
        -0.918368,
        epsilon = MINIMUM_PRECISION
    ));
    assert!(relative_eq!(
        face[8],
        497.107822,
        epsilon = MINIMUM_PRECISION
    ));
    assert!(relative_eq!(
        face[9],
        2661.112178,
        epsilon = MINIMUM_PRECISION
    ));

    // apply the mask
    obj.set_equator_mask(None);

    // checking number of faces after applying the mask
    assert_eq!(obj.number_faces(), 64);

    // checking the first face after applying the mask
    let face = obj.face_iter_all().next().unwrap();
    assert!(relative_eq!(
        face[0],
        490.454163,
        epsilon = MINIMUM_PRECISION
    ));
    assert!(relative_eq!(
        face[1],
        -64.405357,
        epsilon = MINIMUM_PRECISION
    ));
    assert!(relative_eq!(
        face[2],
        32.515083,
        epsilon = MINIMUM_PRECISION
    ));
    assert!(relative_eq!(face[3], 0.990438, epsilon = MINIMUM_PRECISION));
    assert!(relative_eq!(face[4], -0.09755, epsilon = MINIMUM_PRECISION));
    assert!(relative_eq!(face[5], 0.09755, epsilon = MINIMUM_PRECISION));
    assert!(relative_eq!(
        face[6],
        -0.130571,
        epsilon = MINIMUM_PRECISION
    ));
    assert!(relative_eq!(face[7], 0.065637, epsilon = MINIMUM_PRECISION));
    assert!(relative_eq!(
        face[8],
        495.732354,
        epsilon = MINIMUM_PRECISION
    ));
    assert!(relative_eq!(
        face[9],
        4803.45851,
        epsilon = MINIMUM_PRECISION
    ));
}
