use kalast::{Body, Properties, MINIMUM_PRECISION};
use std::path::Path;
use tool::Vector;
use tool::{ASTRONAUMICAL_UNIT, HOUR, TAU};

#[test]
fn body_creation() {
    let name = "Asteroid";
    let frame = "J2000".to_string();
    let pos = Vector::new(0., 1., 0.) * ASTRONAUMICAL_UNIT;
    let path = Path::new("rsc/obj/sphere_light.obj");
    let prop = Properties::new(
        11.92 * HOUR,        // revolution period
        162.0 * TAU / 360.0, // obliquity
        500.,                // thermal inertia
        2146.,               // density
        600.,                // heat capacity
        0.07,                // albedo
        0.9,                 // emissivity
    );
    let _body = Body::new(name, frame, pos, path, prop);
}

#[test]
fn check_properties_computations() {
    let mut prop = Properties::new(
        11.92 * HOUR, // revolution period
        0.,           // obliquity
        500.,         // thermal inertia
        2146.,        // density
        600.,         // heat capacity
        0.07,         // albedo
        0.9,          // emissivity
    );
    prop.set_time_step(30.);
    prop.set_number_thermal_skin_depth(1);

    assert!(relative_eq!(
        prop.conductivity(),
        0.1941596769182976,
        epsilon = MINIMUM_PRECISION
    ));
    assert!(relative_eq!(
        prop.diffusivity(),
        0.00000015079192056407084,
        epsilon = MINIMUM_PRECISION
    ));
    assert!(relative_eq!(
        prop.thermal_skin_depth(),
        0.1425782732630658,
        epsilon = MINIMUM_PRECISION
    ));
    assert!(relative_eq!(
        prop.ground_depth(),
        0.1425782732630658,
        epsilon = MINIMUM_PRECISION
    ));
    assert!(relative_eq!(
        prop.ground_step(),
        0.004253825392242669,
        epsilon = MINIMUM_PRECISION
    ));
    assert_eq!(prop.ground_vector().len(), 35);
    assert!(relative_eq!(
        prop.ground_vector()[prop.ground_vector().len() - 1],
        0.1425782732630658,
        epsilon = MINIMUM_PRECISION
    ));

    let mut it = prop.rotation_axis().iter();
    assert!(relative_eq!(
        *it.next().unwrap(),
        0.,
        epsilon = MINIMUM_PRECISION
    ));
    assert!(relative_eq!(
        *it.next().unwrap(),
        0.,
        epsilon = MINIMUM_PRECISION
    ));
    assert!(relative_eq!(
        *it.next().unwrap(),
        1.,
        epsilon = MINIMUM_PRECISION
    ));

    assert!(relative_eq!(
        prop.rotation_angle(),
        0.004392607177838077,
        epsilon = MINIMUM_PRECISION
    ));

    let mut it = prop.rotation_matrix().matrix().iter();
    assert!(relative_eq!(
        *it.next().unwrap(),
        0.999990352516603,
        epsilon = MINIMUM_PRECISION
    ));
    assert!(relative_eq!(
        *it.next().unwrap(),
        0.00439259305196072,
        epsilon = MINIMUM_PRECISION
    ));
    assert!(relative_eq!(
        *it.next().unwrap(),
        0.,
        epsilon = MINIMUM_PRECISION
    ));
    assert!(relative_eq!(
        *it.next().unwrap(),
        -0.00439259305196072,
        epsilon = MINIMUM_PRECISION
    ));
    assert!(relative_eq!(
        *it.next().unwrap(),
        0.999990352516603,
        epsilon = MINIMUM_PRECISION
    ));
    assert!(relative_eq!(
        *it.next().unwrap(),
        0.,
        epsilon = MINIMUM_PRECISION
    ));
    assert!(relative_eq!(
        *it.next().unwrap(),
        0.,
        epsilon = MINIMUM_PRECISION
    ));
    assert!(relative_eq!(
        *it.next().unwrap(),
        0.,
        epsilon = MINIMUM_PRECISION
    ));
    assert!(relative_eq!(
        *it.next().unwrap(),
        1.,
        epsilon = MINIMUM_PRECISION
    ));
}
