use crate::toolbox::general::Chrono;
use console::Emoji;
use indicatif::{ProgressBar, ProgressStyle};

/// Emoji milky way.
static MILKY_WAY: Emoji<'_, '_> = Emoji("🌌  ", "");
/// Emoji configuration.
static CONFIGURATION: Emoji<'_, '_> = Emoji("🔧  ", "");
/// Emoji ROTATION.
static ROTATION: Emoji<'_, '_> = Emoji("🌔  ", "");

pub struct Information {
    /// Progress bar.
    pub bar: Option<ProgressBar>,
    pub chrono: Option<Chrono>,
}

impl Information {
    /// Constructor.
    pub fn new() -> Self {
        Self {
            bar: None,
            chrono: None,
        }
    }

    /// Initialize the progress bar.
    pub fn set_bar(&mut self, size: usize) {
        self.message_bar_init();
        let bar = ProgressBar::new(size as u64);
        bar.set_style(
            ProgressStyle::default_bar()
                .template("    {prefix}{bar:30.cyan/blue} [ELAPSED: {elapsed:>2}] [ETA: {eta:>2}]"),
        );
        self.bar = Some(bar);
        self.chrono = Some(Chrono::tic());
    }

    /// Get the progress bar.
    pub fn bar(&self) -> &ProgressBar {
        self.bar.as_ref().unwrap()
    }

    /// Get the chrono.
    pub fn chrono(&self) -> &Chrono {
        self.chrono.as_ref().unwrap()
    }

    /// Increment the progress bar.
    pub fn increment_bar(&self) {
        self.bar().inc(1);
    }

    /// Finish the progress bar.
    pub fn end_bar(&self) {
        self.bar().finish_and_clear();
        println!("    elapsed time: {}", self.chrono().toc_str());
    }

    /// Message for the start of the main loop.
    pub fn message_bar_init(&self) {
        println!("{}Simulation started...", ROTATION,);
    }

    /// Message for initialized world.
    pub fn message_initialized(&self) {
        println!("{}kalast world initialized.", MILKY_WAY,);
    }

    /// Message for main loop preparation.
    pub fn message_preparation(&self) {
        println!("{}Preparing before simulation...", CONFIGURATION,);
    }
}

impl Default for Information {
    fn default() -> Self {
        Self::new()
    }
}
