/*!
# **kalast**

Next generation thermophysical modelling of asteroids

## Using **kalast**

You will need the last stable build of the [rust compiler](https://www.rust-lang.org)
and the official package manager: [cargo](https://github.com/rust-lang/cargo).

Simply add the following to your `Cargo.toml` file:

```.ignore
[dependencies]
// replace * by the latest version of the crates
kalast = "*"
rustool = "*"
```

Useful functionalities of **kalast** are grouped in the root module `kalast::`.

## **kalast** in action

Dimorphos case using HERA kernels.

```
// Initialize the library.
let mut kalast = kalast::init();

let revolution = 11.92 * kalast::HOUR;
let duration = 192.0 * kalast::DAY;

// System center of mass position.
kalast.set_system_name("DIDYMOS_BARYCENTER");
kalast.set_kernel_system(
    "rsc/kernels/hera_PO_EMA_2024.tm",
    "J2000",
    "SUN",
    "2027-MAR-23 16:00:00",
);

// Add a body to the system without a mesh.
kalast.add_body(
    "DIMORPHOS",           // name
    tool::Vector::zeros(), // relative position to center of mass
    kalast::Properties::new(
        revolution,                  // revolution period
        162.0 * kalast::TAU / 360.0, // obliquity
        500.,                        // thermal inertia
        2146.,                       // density
        600.,                        // heat capacity
        0.07,                        // albedo
        0.9,                         // emissivity
    ),
    Some("rsc/obj/g_08438mm_lgt_obj_didb_0000n00000_v002.obj"), // path to 3D mesh
);

tool::debug!(
    "Barycenter position start: {:.2} AU",
    tool::distance(&kalast.system.position, &kalast.system.sun_position)
        / kalast::ASTRONAUMICAL_UNIT
);

// Some configurations about the system.
kalast.enable_self_heating(false);

// Set simulation duration and time step.
kalast.set_time(duration, 10.0 * kalast::MINUTE);

// Some configuration for exporting before running the simulation.
kalast.export_surface(true);

// Start the simulation.
kalast.start();

// Distance after simulation.
tool::debug!(
    "barycenter position end: {:.2} au",
    tool::distance(&kalast.system.position, &kalast.system.sun_position)
        / kalast::ASTRONAUMICAL_UNIT
);

// Temperatures at the end.
let temp = kalast.get_target().surface_temperatures();
println!(
    "Surface temperatures > max: {:.2} K, min: {:.2} K",
    temp.max(),
    temp.min()
);

kalast.kernel.as_mut().unwrap().unload().unwrap();
```

You can also read other [examples](https://gitlab-as.oma.be/gregoireh/kalast/-/tree/main/examples).

## Features

**kalast** is meant for binary system of asteroids surface thermophysical modelling. The physics of this engine includes these features:

+ read shape model with Wavefront (.obj) format
+ custom shape model by creating vertices & facets
+ celestial body revolution around tilted spin axis
+ orbital positions using Spice kernels
+ surface mask (e.g., to only get equator of asteroid)
+ solar flux
+ self view factor (for craters and surface roughness)
+ self heating
+ view factor (for binary systems)
+ mutual heating (for binary systems)
+ one-dimension conduction (finite difference methods)
+ TODO: 3D conduction / finite elemen methods
+ TODO: mutual occultations
+ TODO: shadowing

## Explore

If your want to explore the documentation, you can visite these pages:

+ [`Object3D`]: parse 3D object file and compute atributes of faces (centers, normals, ...)
+ [`Body`]: the representation for a celestial body
+ [`System`]: the environment system
+ [`Kalast`]: the simulation manager
+ [`Properties`]: all the properties to characterise a body
*/

extern crate indicatif;
#[macro_use]
extern crate itertools;
extern crate nalgebra as na;
extern crate obj;
extern crate tool;

/// Core features.
pub mod core;
/// Generic functions for math, physics, matrix operations.
#[macro_use]
pub mod toolbox;

use na::{Dynamic, Matrix, SliceStorage, SliceStorageMut, VecStorage, U1, U10};

pub use crate::core::*;
pub use crate::toolbox::*;

/// Type alias for [`Object3D`]. The matrix has a fixed number of rows (10) and a dynamical number
/// of columns.
pub type Matrix10xX<T> = Matrix<T, U10, Dynamic, VecStorage<T, U10, Dynamic>>;

/// Type alias for [`Object3D`]. The slice matrix has a fixed number of rows (10) and a fixed
/// number of columns (1).
pub type MatrixSlice10x1<'a, T, RStride = U10, CStride = U1> =
    Matrix<T, U10, U1, SliceStorage<'a, T, U10, U1, RStride, CStride>>;

/// Type alias for [`Object3D`]. The mutable slice matrix has a fixed number of rows (10) and a
/// fixed number of columns (1).
pub type MatrixSliceMut10x1<'a, T, RStride = U10, CStride = U1> =
    Matrix<T, U10, U1, SliceStorageMut<'a, T, U10, U1, RStride, CStride>>;
