use crate::core::body::{Body, Properties};
use crate::core::environment_system::System;
use crate::core::export::ExportManager;
use crate::core::time::Time;
use crate::toolbox::information::Information;
use tool::Vector;

/// Main manager.
pub struct Kalast {
    /// Celestial body system (can be single body or binary aswell).
    pub system: System,
    /// Time manager.
    pub time: Time,
    /// The screen display manager.
    pub information: Information,
    /// Export data manager.
    pub export: std::cell::RefCell<ExportManager>,
    /// Kernel name.
    pub kernel: Option<spice::Kernel>,
    /// Reference frame.
    pub frame: Option<String>,
    /// Observer.
    pub observer: Option<String>,
    /// Start date.
    pub start_date: Option<String>,
    /// Ephemeris time.
    pub ephemeris_time: Option<f64>,
}

/// Function called at the beginning of the program to initialize the library.
pub fn init() -> Kalast {
    tool::log_init!();
    let kalast = Kalast {
        system: System::default(),
        time: Time::default(),
        information: Information::default(),
        export: std::cell::RefCell::new(ExportManager::default()),
        kernel: None,
        frame: None,
        observer: None,
        start_date: None,
        ephemeris_time: None,
    };
    kalast.information.message_initialized();
    kalast
}

impl Kalast {
    /// Set system name.
    pub fn set_system_name(&mut self, name: &str) {
        self.system.name = name.to_string();
    }

    /// Set system frame.
    pub fn set_system_frame(&mut self, name: &str) {
        self.system.frame = name.to_string();
    }

    /// Set system position.
    pub fn set_system_position(&mut self, position: Vector<f64>) {
        self.system.set_position(position);
    }

    /// Set the spice system from a kernel.
    pub fn set_kernel_system<S: Into<String>>(
        &mut self,
        kernel_name: S,
        frame: S,
        observer: S,
        start_date: S,
    ) {
        // Save fields.
        self.kernel = Some(spice::Kernel::new(kernel_name.into()).unwrap());
        self.frame = Some(frame.into());
        self.observer = Some(observer.into());
        self.start_date = Some(start_date.into());
        self.ephemeris_time = Some(spice::str2et(self.start_date.as_ref().unwrap().clone()));

        // Set position of start.
        let time = spice::str2et(self.start_date.as_ref().unwrap().clone());
        let (position_start, _) = spice::spkpos(
            self.system.name.clone(),
            time,
            self.frame.as_ref().unwrap().clone(),
            "NONE".to_string(),
            self.observer.as_ref().unwrap().clone(),
        );
        self.set_system_position(position_start * 1e3);
    }

    /// Add body to system.
    /// Position is the relative position of the body in the system.
    pub fn add_body(
        &mut self,
        name: &str,
        frame: Option<&str>,
        position: Vector<f64>,
        properties: Properties,
        path: Option<&str>,
    ) {
        self.system
            .add_body(name, frame, position, properties, path);
        // If kernel has been given.
        if let (Some(frame), Some(ephemeris_time)) = (self.frame.clone(), self.ephemeris_time) {
            let body = self.get_body_mut(name);
            // Tilt body on its spin axis.
            let rotate = spice::pxform(frame, body.frame.clone(), ephemeris_time);
            body.rotate(&rotate);
        }
    }

    /// Get a body in the system from its name.
    pub fn get_body(&self, name: &str) -> &Body {
        self.system.get_body(name)
    }

    /// Get a body as mutable in the system from its name.
    pub fn get_body_mut(&mut self, name: &str) -> &mut Body {
        self.system.get_body_mut(name)
    }

    /// Get targeted body in the system.
    pub fn get_target(&self) -> &Body {
        self.system.get_target()
    }

    /// Get targeted body as mutable in the system.
    pub fn get_target_mut(&mut self) -> &mut Body {
        self.system.get_target_mut()
    }

    /// Iterate over other bodies than the target.
    pub fn iter_other_bodies(&self) -> impl Iterator<Item = &Body> {
        self.system.iter_other_bodies()
    }

    /// Whether to enable the mutual heating.
    pub fn enable_mutual_heating(&mut self, enable: bool) {
        self.system.enable_mutual_heating(enable);
    }

    /// Whether to enable the self heating.
    pub fn enable_self_heating(&mut self, enable: bool) {
        self.system.enable_self_heating(enable);
    }

    /// Whether to enable the solar flux.
    pub fn enable_solar_flux(&mut self, enable: bool) {
        self.system.enable_solar_flux(enable);
    }

    /// Set time duration and time step.
    pub fn set_time(&mut self, duration: f64, time_step: f64) {
        self.time = Time::new(duration, time_step);
        self.share_time_step();
    }

    /// Set time duration in number of target revolution and also set time step.
    pub fn set_time_with_revolution_target(&mut self, number_revolution: f64, time_step: f64) {
        self.time = Time::new(
            self.get_target().properties.rotation_period() * number_revolution,
            time_step,
        );
        self.share_time_step();
    }

    /// Share time step with body for initialization.
    pub fn share_time_step(&mut self) {
        let time_step = self.time.time_step();
        self.get_target_mut().share_time_step(time_step);
    }

    /// Preparation before the start of the simulation.
    pub fn prepare(&mut self) {
        self.information.message_preparation();
        self.system.initialize_temperatures();
        // self.time
        // .recompute_to_match_last_revolution(self.get_target().properties.rotation_period());
        self.information.set_bar(self.time.size());
        self.export.borrow_mut().initialize(self);
    }

    /// Preparation for the current time iteration.
    pub fn prepare_iteration(&mut self, time_step: f64) {
        if self.ephemeris_time.is_some() {
            let time_old = self.ephemeris_time.unwrap();
            self.ephemeris_time = Some(time_old + time_step);
        }
        self.system.orbit_and_revolution_iteration(
            time_step,
            self.frame.clone(),
            self.observer.clone(),
            self.ephemeris_time,
        );
    }

    /// Function called at the end of each iteration of the main time loop.
    pub fn finish_iteration(&mut self, current_time: f64) -> bool {
        // Stop when last day recording is finished.
        let stop = self.export.borrow_mut().after_iteration(self, current_time);
        self.information.increment_bar();
        stop
    }

    /// Simulation time loop, returns the remaining time that has been skipped.
    pub fn main_time_loop(&mut self) -> f64 {
        let mut time_skipped = 0.0;
        for (index, _time) in self.time.data().clone().iter().cloned().enumerate().skip(1) {
            let time_step = self.time.time_step_backward(index);
            self.prepare_iteration(time_step);
            self.system
                .compute_and_apply_heat_flux_at_boundaries_conditions();
            self.system.ground_conduction();
            let stop = self.finish_iteration(_time);
            if stop {
                time_skipped = self.time.last() - _time;
                break;
            }
        }
        time_skipped
    }

    /// Called after the time loop ended.
    pub fn finish(&self, time_skipped: f64) {
        self.information.end_bar();
        self.export.borrow_mut().export(self, time_skipped);
    }

    /// Simulation time loop.
    pub fn start(&mut self) {
        self.prepare();
        let time_skipped = self.main_time_loop();
        self.finish(time_skipped);
    }

    /// Set path.
    pub fn export_path(&self, path: &str) {
        self.export.borrow_mut().set_path(path);
    }

    /// Whether to enable the save of the ground.
    pub fn export_ground(&self, enable: bool) {
        self.export.borrow_mut().enable_ground(enable);
    }

    pub fn export_surface(&mut self, enable: bool) {
        self.export.borrow_mut().enable_surface(enable);
    }

    /// Whether to enable the save of the temperatures during the last day.
    pub fn export_last_day(&mut self, enable: bool) {
        self.export.borrow_mut().enable_last_day(enable);
    }

    /// Whether to enable the last hour angle of the prime meridian is noon.
    pub fn export_end_prime_noon(&mut self, enable: bool) {
        self.export.borrow_mut().enable_end_prime_noon(enable);
    }
}
