use tool::{direct_angle, projection_plane, Vector, TAU};

/// Compute the maximum value for the ground step to respect stability criterion of heat
/// conduction.
pub fn compute_ground_step_from_time_and_stability(
    time_step: f64,
    diffusivity: f64,
    numerical_stability: f64,
) -> f64 {
    (time_step * diffusivity / numerical_stability).sqrt()
}

/// Compute the maximum value for the ground step to respect stability criterion of heat
/// conduction from basic properties.
pub fn compute_ground_step_from_basic(
    time_step: f64,
    thermal_inertia: f64,
    density: f64,
    heat_capacity: f64,
    numerical_stability: f64,
) -> f64 {
    compute_ground_step_from_time_and_stability(
        time_step,
        compute_diffusivity_from_basic(thermal_inertia, density, heat_capacity),
        numerical_stability,
    )
}

/// Compute the diffusivity.
pub fn compute_diffusivity(conductivity: f64, density: f64, heat_capacity: f64) -> f64 {
    conductivity / (density * heat_capacity)
}

/// Compute the diffusivity from basic properties.
pub fn compute_diffusivity_from_basic(
    thermal_inertia: f64,
    density: f64,
    heat_capacity: f64,
) -> f64 {
    (thermal_inertia / (density * heat_capacity)).powi(2)
}

/// Compute the conductivity.
pub fn compute_conductivity(thermal_inertia: f64, density: f64, heat_capacity: f64) -> f64 {
    thermal_inertia.powi(2) / (density * heat_capacity)
}

/// Get the hour angle from the last revolution, in radians.
/// It consideres the last revolution is set to be a full hour angle from 0 radians to TAU radians.
pub fn hour_angle(current_time: f64, duration: f64, revolution_period: f64) -> f64 {
    let start_last_revolution = duration - revolution_period;
    (current_time - start_last_revolution) / revolution_period * TAU
}

/// Get the exact hour angle in radians of a meridian at the surface of an asteroid.
///
/// ## Definition
///
/// The exact hour angle is defined as the direct angle between the current meridian and the
/// subsolar merdidian.
///
/// You need to provide this function three vectors:
///
/// + `meridian vector`: a vector defined by the direction of the position at the surface of the
///   asteroid from its geocentric center
/// + `solar_meridian_vector`: a vector built as the direction of the Sun from the geocentric
///   center of the asteroid
/// + `plane_normal`: a plane on which the two previous vectors are projected to compute the angle
///   between them
pub fn compute_exact_hour_angle(
    meridian_vector: &Vector<f64>,
    solar_meridian_vector: &Vector<f64>,
    plane_normal: &Vector<f64>,
) -> f64 {
    // Project `meridian_vector` & `solar_meridian_vector` onto the plane described by
    // `plane_normal`.
    let meridian = projection_plane(meridian_vector, plane_normal);
    let solar_meridian = projection_plane(solar_meridian_vector, plane_normal);
    // Compute the direct angle between `meridian` & `solar_meridian` on the plane.
    TAU - direct_angle(&meridian, &solar_meridian, plane_normal)
}
