use crate::{Body, Kalast};
use serde_json::{json, Value};
use tool::writejs;
use tool::TAU;

/// Default location of the ouput.
const DEFAULT_PATH: &str = "rsc/data/tmp.json";

/// The type `ExportManager` contains the parameters for the export of the results into json
/// format.
pub struct ExportManager {
    /// File location.
    path: String,
    /// Properties for the export.
    properties: ExportProperties,
    /// JSON contents.
    json: Value,
    /// Enabled when the last synodic rotation period has started.
    last_synodic_started: bool,
    /// Previous value of the hour angle.
    previous_hour_angle: Option<f64>,
}

impl ExportManager {
    /// Constructor.
    pub fn new<S: Into<String>>(path: S) -> Self {
        Self {
            path: path.into(),
            properties: ExportProperties::default(),
            json: json!({}),
            last_synodic_started: false,
            previous_hour_angle: None,
        }
    }

    /// Get path.
    pub fn path(&self) -> String {
        self.path.clone()
    }

    /// Set path.
    pub fn set_path<S: Into<String>>(&mut self, path: S) {
        self.path = path.into();
    }

    /// Get properties.
    pub fn properties(&self) -> &ExportProperties {
        &self.properties
    }

    /// Get properties as mutable.
    pub fn properties_mut(&mut self) -> &mut ExportProperties {
        &mut self.properties
    }

    /// Whether to enable the save of the surface.
    pub fn enable_surface(&mut self, enable: bool) {
        self.properties.enable_surface(enable);
    }

    /// Whether to enable the save of the ground.
    pub fn enable_ground(&mut self, enable: bool) {
        self.properties.enable_ground(enable);
    }

    /// Whether to enable the save of the temperatures during the last day.
    pub fn enable_last_day(&mut self, enable: bool) {
        self.properties.enable_last_day(enable);
    }

    /// Whether to enable the last hour angle of the prime meridian is noon.
    pub fn enable_end_prime_noon(&mut self, enable: bool) {
        self.properties.enable_end_prime_noon(enable);
    }

    /// Get hour angle.
    fn hour_angle(&mut self, kalast: &Kalast) -> f64 {
        let body = kalast.get_target();
        let face = body.object.get_face_closest_longitude(0.0);
        let center = face.1.fixed_rows::<3>(0);
        let meridian_vector = center.normalize();
        let solar_meridian_vector = -kalast.system.position_target().normalize();
        let plane_normal = kalast.system.get_target().properties.rotation_axis();
        crate::compute_exact_hour_angle(&meridian_vector, &solar_meridian_vector, plane_normal)
    }

    /// Apply properties to export the data.
    pub fn initialize(&mut self, kalast: &Kalast) {
        let body = kalast.get_target();
        self.json = json!({
            "context": {
                "duration": kalast.time.duration(),
                "time_step": kalast.time.time_step(),
                "name": body.name,
                "frame": body.frame,
                "heliocentric_distance": kalast.system.heliocentric_distance_target(),
                "properties": {
                    "rotaton_period": body.properties.rotation_period(),
                    "obliquity": body.properties.obliquity(),
                    "thermal_inertia": body.properties.thermal_inertia(),
                    "density": body.properties.density(),
                    "heat_capacity": body.properties.heat_capacity(),
                    "albedo": body.properties.albedo(),
                    "emissivity": body.properties.emissivity(),
                },
                "system": {
                    "name": kalast.system.name,
                    "frame": kalast.system.frame,
                }
            },
            "mesh": {
                "file": body.object.path(),
                "sphericals": body.object.sphericals().into_owned(),
            },
        });
    }

    /// Export temperatures.
    pub fn export_temperatures(&mut self, ha: f64, body: &Body) {
        if self.properties.surface {
            if self.json.get("surface_temperatures").is_none() {
                self.json["surface_temperatures"] = json!({
                    "hour_angle": [],
                    "temperatures": [],
                });
            }
            self.json["surface_temperatures"]["hour_angle"]
                .as_array_mut()
                .unwrap()
                .push(json!(ha));
            self.json["surface_temperatures"]["temperatures"]
                .as_array_mut()
                .unwrap()
                .push(json!(body.surface_temperatures_cloned()));
        }
        if self.properties.ground {
            if self.json.get("ground_temperatures").is_none() {
                self.json["ground_temperatures"] = json!({
                    "hour_angle": [],
                    "temperatures": [],
                });
            }
            self.json["ground_temperatures"]["hour_angle"]
                .as_array_mut()
                .unwrap()
                .push(json!(ha));
            self.json["ground_temperatures"]["temperatures"]
                .as_array_mut()
                .unwrap()
                .push(json!(body.ground_temperatures()));
        }
    }

    /// Function called by kalast main time loop at the end of each iteration to check if
    /// temperatures needs to be saved. Return a boolean flag to end the time loop when needed,
    /// with the value `true`.
    pub fn after_iteration(&mut self, kalast: &Kalast, current_time: f64) -> bool {
        let rot = kalast.get_target().properties.rotation_period();
        let last_two_rot = kalast
            .time
            .did_last_two_rotations_started(current_time, rot);
        // If last two sideral rotation periods have started.
        if last_two_rot && (self.properties.last_day || self.properties.end_prime_noon) {
            let ha = self.hour_angle(kalast);
            // Detect start of a new day.
            let new_day = self.previous_hour_angle.is_some()
                && (ha - self.previous_hour_angle.unwrap()).abs() > TAU / 2.0;
            // If last synodic rotation period has not started yet
            // -> start it
            // else -> stop time loop
            if new_day {
                if !self.last_synodic_started {
                    self.last_synodic_started = true;
                } else {
                    return true;
                }
            }
            // if last synodic rotation period
            // -> save it
            if self.last_synodic_started && self.properties.last_day {
                self.export_temperatures(ha, kalast.get_target())
            }
            self.previous_hour_angle = Some(ha);
        }
        false
    }

    /// Apply properties to export the data.
    pub fn export(&mut self, kalast: &Kalast, time_skipped: f64) {
        self.json["context"]["time_skipped"] = json!(time_skipped);
        if !self.properties.last_day {
            let ha = self.hour_angle(kalast);
            self.export_temperatures(ha, kalast.get_target());
        }
        writejs!(self.path(), self.json);
    }
}

impl Default for ExportManager {
    fn default() -> Self {
        Self::new(DEFAULT_PATH)
    }
}

/// The type `ExportProperties` is the list of the parameters to control the export.
pub struct ExportProperties {
    /// Whether to save the surface.
    pub surface: bool,
    /// Whether to save the ground.
    pub ground: bool,
    /// Whether to save the temperatures during the last day.
    pub last_day: bool,
    /// Whether the prime meridian should end at noon.
    pub end_prime_noon: bool,
}

impl ExportProperties {
    /// Constructor.
    pub fn new() -> Self {
        Self {
            surface: false,
            ground: false,
            last_day: false,
            end_prime_noon: false,
        }
    }

    /// Whether to enable the save of the surface.
    pub fn enable_surface(&mut self, enable: bool) {
        self.surface = enable;
    }

    /// Whether to enable the save of the ground.
    pub fn enable_ground(&mut self, enable: bool) {
        self.ground = enable;
    }

    /// Whether to enable the save of the temperatures during the last rotation.
    pub fn enable_last_day(&mut self, enable: bool) {
        self.last_day = enable;
    }

    /// Whether to enable the last hour angle of the prime meridian is noon.
    pub fn enable_end_prime_noon(&mut self, enable: bool) {
        self.end_prime_noon = enable;
    }
}

impl Default for ExportProperties {
    fn default() -> Self {
        Self::new()
    }
}
