use crate::core::body::{Body, Properties};
use crate::core::constants::{SOLAR_CONSTANT, STEFAN_BOLTZMANN};
use crate::itertools::multizip;
use na::DMatrix;
use std::collections::HashMap;
use tool::{List, Vector, Vectors};
use tool::{ASTRONAUMICAL_UNIT, TAU};

pub struct System {
    /// Name of the system.
    pub name: String,
    /// Name of the frame.
    pub frame: String,
    /// Position of the center of mass of system.
    pub position: Vector<f64>,
    /// Bodies in the system.
    pub bodies: HashMap<String, Body>,
    /// Which body is the target, by default the first body added.
    pub target_name: Option<String>,
    /// Sun position.
    pub sun_position: Vector<f64>,
    /// Whether the mutual heating is computed for binary systems.
    pub enable_mutual_heating: bool,
    /// Whether the self heating is computed.
    pub enable_self_heating: bool,
    /// Whether the solar flux is computed.
    pub enable_solar_flux: bool,
    /// Whether the ground temperatures should be initialized to global equilibrium temperatures.
    pub enable_initialization_global_equilibrium: bool,
}

impl System {
    /// Constructor.
    pub fn new() -> Self {
        Self {
            name: "Default system".to_string(),
            frame: "J2000".to_string(),
            position: Vector::zeros(),
            bodies: HashMap::new(),
            target_name: None,
            sun_position: Vector::zeros(),
            enable_mutual_heating: true,
            enable_self_heating: true,
            enable_solar_flux: true,
            enable_initialization_global_equilibrium: true,
        }
    }

    /// Set position of the center of mass of the system.
    pub fn set_position(&mut self, position: Vector<f64>) {
        self.position = position;
    }

    /// Add body to system.
    /// Position is the relative position of the body in the system.
    pub fn add_body(
        &mut self,
        name: &str,
        frame: Option<&str>,
        position: Vector<f64>,
        properties: Properties,
        path: Option<&str>,
    ) {
        if self.bodies.len() >= 2 {
            panic!("System of more than 2 bodies not implemented!",);
        }
        if self.bodies.contains_key(&name.to_string()) {
            panic!("A body already exists with this name.",);
        }
        let _frame = match frame {
            Some(f) => f.to_string(),
            None => self.frame.clone(),
        };
        let body = match path {
            Some(path) => Body::new(name, _frame, position, path, properties),
            None => Body::empty(name, _frame, position, properties),
        };
        self.bodies.insert(name.to_string(), body);
        if self.bodies.len() == 1 {
            self.target_name = Some(name.to_string());
        }
        // Note: when adding in the future the possibility to remove bodies, be careful to update
        // the possible target body change.
    }

    /// Get body with name.
    pub fn get_body(&self, name: &str) -> &Body {
        match self.bodies.get(&name.to_string()) {
            Some(body) => body,
            None => panic!("Body {} not found.", name),
        }
    }

    /// Get body as mutable with name.
    pub fn get_body_mut(&mut self, name: &str) -> &mut Body {
        match self.bodies.get_mut(&name.to_string()) {
            Some(body) => body,
            None => panic!("Body {} not found.", name),
        }
    }

    /// Get targeted body.
    pub fn get_target(&self) -> &Body {
        match &self.target_name {
            Some(name) => self.get_body(name.as_str()),
            None => panic!("No body added to the system."),
        }
    }

    /// Get targeted body as mutable.
    pub fn get_target_mut(&mut self) -> &mut Body {
        match self.target_name.clone() {
            Some(name) => self.get_body_mut(name.as_str()),
            None => panic!("No body added to the system."),
        }
    }

    /// Iterate over other bodies than the target.
    pub fn iter_other_bodies(&self) -> impl Iterator<Item = &Body> {
        self.bodies
            .iter()
            .filter(move |(name, _)| {
                self.target_name.is_none() || *name != self.target_name.as_ref().unwrap()
            })
            .map(|(_, body)| body)
    }

    /// Whether to enable the mutual heating.
    pub fn enable_mutual_heating(&mut self, enable: bool) {
        self.enable_mutual_heating = enable;
    }

    /// Whether to enable the self heating.
    pub fn enable_self_heating(&mut self, enable: bool) {
        self.enable_self_heating = enable;
    }

    /// Whether to enable the solar flux.
    pub fn enable_solar_flux(&mut self, enable: bool) {
        self.enable_solar_flux = enable;
    }

    /// Whether to initialize ground temperatures at global equilibrium temperatures.
    pub fn enable_initialization_global_equilibrium(&mut self, enable: bool) {
        self.enable_initialization_global_equilibrium = enable;
    }

    /// Get position of a body in the inertial frame wrt to the position of the Sun.
    pub fn position_body(&self, body: &Body) -> Vector<f64> {
        self.position + body.position - self.sun_position
    }

    /// Get position of the target in the inertial frame wrt to the position of the Sun.
    pub fn position_target(&self) -> Vector<f64> {
        self.position_body(self.get_target())
    }

    /// Get heliocentric distance of a body.
    pub fn heliocentric_distance_body(&self, body: &Body) -> f64 {
        self.position_body(body).norm()
    }

    /// Get heliocentric distance of the target.
    pub fn heliocentric_distance_target(&self) -> f64 {
        self.heliocentric_distance_body(self.get_target())
    }

    /// Proceed to one iteration in the orbit and in revolution of the bodies in the system.
    pub(crate) fn orbit_and_revolution_iteration(
        &mut self,
        time_step: f64,
        frame: Option<String>,
        observer: Option<String>,
        ephemeris_time: Option<f64>,
    ) {
        // If kernel has been given.
        if let (Some(frame), Some(observer), Some(ephemeris_time)) =
            (frame, observer, ephemeris_time)
        {
            // Barycenter position.
            let (position, _) = spice::spkpos(
                self.name.clone(),
                ephemeris_time,
                frame.clone(),
                "NONE".to_string(),
                observer,
            );
            self.set_position(position * 1e3);

            // Each body.
            for (_, body) in self.bodies.iter_mut() {
                // Body position.
                let (position, _) = spice::spkpos(
                    body.name.clone(),
                    ephemeris_time,
                    frame.clone(),
                    "NONE".to_string(),
                    self.name.clone(),
                );
                body.set_position(position * 1e3);

                // Body rotation.
                let rotate_step = spice::pxfrm2(
                    body.frame.clone(),
                    body.frame.clone(),
                    ephemeris_time - time_step,
                    ephemeris_time,
                );
                body.rotate(&rotate_step);
            }
        } else {
            // Rotation with obliquity.
            for (_, body) in self.bodies.iter_mut() {
                body.properties.set_time_step(time_step);
                body.rotation_iteration();
            }
        }
    }

    /// Temperatures initialization during simulation preparation.
    pub(crate) fn initialize_temperatures(&mut self) {
        // Initialize ground temperatures methods.
        if self.enable_solar_flux && self.enable_initialization_global_equilibrium {
            let initial_temperature = self.global_equilibrium_temperature();
            self.get_target_mut()
                .recompute_ground_temperatures(Some(initial_temperature));
        }

        // Initialize view factor
        if self.enable_solar_flux && self.enable_self_heating {
            self.get_target_mut().compute_self_view_factor()
        }
    }

    /// Compute the direction of the Sun from each face at the surface of the celestial body.
    pub fn directions_facets_to_sun(&self, body: &Body) -> Vectors<f64> {
        let mut directions = body.object.centers().into_owned();
        for mut face in directions.column_iter_mut() {
            face.copy_from(&(self.position_body(body) + &face).normalize());
        }
        -directions
    }

    /// Compute the cosine of the illumation angles for each face at the surface of the targeted
    /// body (angle between the direction from each face to the Sun and the normal of each face).
    ///
    /// ## Expression
    ///
    /// $$\cos\varsigma=\bm{\hat{q_{i\odot}}}\cdot\bm{\hat{n_i}}$$
    ///
    /// where $\varsigma$ is the illumination angle, $\bm{\hat{q_{i\odot}}}$ the direction to the
    /// Sun from each facet, and $\bm{\hat{n_i}}$ the normal of each facet.
    pub fn illumination_cosine(&self, body: &Body) -> List<f64> {
        let directions_facets_to_sun = self.directions_facets_to_sun(body);
        tool::dot_products(
            &directions_facets_to_sun,
            &body.object.normals().into_owned(),
        )
    }

    /// Compute the global temperature at equilibrium of the targeted body.
    ///
    /// ## Expression
    ///
    #[allow(broken_intra_doc_links)]
    /// $$T=\sqrt[4]{\frac{S_\odot\left(1-A\right)}{4\epsilon\sigma r_H^2}}$$
    ///
    /// where $S_\odot$ is the [Solar Constant][SOLAR_CONSTANT], $A$ the albedo, $\epsilon$ the
    /// emissivity, $\sigma$ the [Stefan-Boltzmann constant][STEFAN_BOLTZMANN], and $r_H$ the
    /// heliocentric distance in [AU][ASTRONAUMICAL_UNIT].
    pub fn global_equilibrium_temperature(&self) -> f64 {
        let body = self.get_target();
        (SOLAR_CONSTANT * (1. - body.properties.albedo())
            / (4.
                * body.properties.emissivity()
                * STEFAN_BOLTZMANN
                * (self.heliocentric_distance_body(body) / ASTRONAUMICAL_UNIT).powi(2)))
        .sqrt()
        .sqrt()
    }

    /// Compute the direct solar flux at the surface of the targeted body.
    ///
    /// ## Expression
    ///
    /// $$U=\frac{S_\odot\left(1-A\right)\cos\varsigma_i\left(t\right)}{r_H^2\left(t\right)}$$
    ///
    /// where $S_\odot$ is the [Solar Constant][SOLAR_CONSTANT], $A$ the albedo, $\varsigma_i$ the
    /// illumination angle of each facet, and $r_H$ the heliocentric distance in
    /// [AU][ASTRONAUMICAL_UNIT].
    pub fn direct_solar_flux(&self, body: &Body) -> List<f64> {
        tool::clip(
            &(SOLAR_CONSTANT * (1. - body.properties.albedo()) * self.illumination_cosine(body)
                / (tool::distance(&(self.position + body.position), &self.sun_position)
                    / ASTRONAUMICAL_UNIT)
                    .powi(2)),
            Some(0.),
            None,
        )
    }

    /// Compute the diffuse solar self-radiation of the targeted body.
    ///
    /// ## Expression
    ///
    /// The diffuse solar self-radiation contribution from all $N$ facets $i$ of a body onto the facet $j$ is
    /// defined as,
    ///
    /// $$W_j=\sum_{i\cancel{=}j}^{N}V_{ij}\frac{S_\odot A\cos\varsigma_j\left(t\right)}{r_H^2\left(t\right)}$$
    ///
    /// where $V_{ij}$ is the view factor describing the fraction of energy emitted from one facet
    /// $i$ towards the facet $j$, $S_\odot$ is [Solar Constant][SOLAR_CONSTANT], $A$ the albedo,
    /// $\varsigma_j$ the illumination angle of the facet $j$, and $r_H$ the heliocentric distance
    /// in [AU][ASTRONAUMICAL_UNIT].
    pub fn self_diffuse_solar_radiation(&mut self) -> List<f64> {
        let mut diffuse_solar_radiation = List::zeros(self.get_target().object.number_faces());
        let direct_solar_flux_from_other = self.direct_solar_flux(self.get_target())
            * self.get_target().properties.albedo()
            / (1. - self.get_target().properties.albedo());
        let view_factor = self.get_target_mut().self_view_factor();
        for (flux, view_factor) in izip!(
            diffuse_solar_radiation.iter_mut(),
            view_factor.column_iter()
        ) {
            *flux = (view_factor.component_mul(&direct_solar_flux_from_other)).sum();
        }
        diffuse_solar_radiation
    }

    /// Compute the direct thermal self-heating of the targeted body.
    ///
    /// ## Expression
    ///
    /// The direct thermal self-heating contribution from all $N$ facets $i$ of a body onto the
    /// facet $j$ is defined as,
    ///
    /// $$u_{j}=\sum_{i\cancel{=}j}^{N}V_{ij}\epsilon\sigma T_{i}^4$$
    ///
    /// where $V_{ij}$ is the view factor describing the fraction of energy emitted from one facet
    /// $i$ towards the facet $j$, $\epsilon$ the emissivity, $\sigma$ the
    /// [Stefan-Boltzmann constant][STEFAN_BOLTZMANN], and $T_i$ the temperature of the facet $i$.
    pub fn self_direct_thermal_heating(&mut self) -> List<f64> {
        let mut direct_thermal_heating = List::zeros(self.get_target().object.number_faces());
        let other_temperatures_4 =
            tool::pows(&self.get_target().surface_temperatures_cloned_vec(), 4);
        let emissivity = self.get_target().properties.emissivity();
        let view_factor = self.get_target_mut().self_view_factor();
        for (flux, view_factor) in
            izip!(direct_thermal_heating.iter_mut(), view_factor.column_iter())
        {
            *flux =
                (view_factor.component_mul(&other_temperatures_4) * emissivity * STEFAN_BOLTZMANN)
                    .sum();
        }
        direct_thermal_heating
    }

    /// Relative position between the targeted body and another body.
    pub fn relative_position(&self, other_body: &Body) -> Vector<f64> {
        other_body.position - self.get_target().position
    }

    /// Compute the view factor describing the fraction of energy emitted from one facet of the
    /// targeted body toward another body of the system.
    ///
    /// ## Definition
    ///
    /// The viewing factor is null if the line of sight is intercepted by topography (TODO) or if
    /// any of the angles between the facet outward and the line between facet centers is superior
    /// to 90 degrees.
    /// Rows are the faces of the other body and columns are the faces of the targeted body.
    /// Meaning, if you read this matrix column by column, you get the view of the face n°(indice
    /// of the column) of the target body regarding all the faces of the primary body.
    ///
    /// ## Expression
    ///
    /// $$V_{ij}=\frac{a_i\cos\Theta_i\cos\Theta_j}{\pi r^2}$$
    ///
    /// where $a_i$ is the area of the facet $i$, $\Theta_x$ the angle between the centers of the
    /// facets $i$ and $j$ and the normal of the facet $x$, and $r$ is the distance between the
    /// facets.
    pub fn view_factor(&self, other_body: &Body) -> DMatrix<f64> {
        let mut view_factor = DMatrix::<f64>::zeros(
            other_body.object.number_faces(),
            self.get_target().object.number_faces(),
        );
        let position_relative = self.relative_position(other_body);
        // Cartesian product of secondary's and primary's faces.
        for ((face_target, face_other), view) in izip!(
            iproduct!(
                self.get_target().object.face_iter_all(),
                other_body.object.face_iter_all()
            ),
            view_factor.iter_mut()
        ) {
            let normal_target = face_target.fixed_rows::<3>(3);
            let normal_other = face_other.fixed_rows::<3>(3);
            let center_target = face_target.fixed_rows::<3>(0);
            let center_other = face_other.fixed_rows::<3>(0);
            let area_other = face_other[9];
            let relative_position_faces = position_relative + center_other - center_target;
            let direction_faces = relative_position_faces.normalize();
            let distance_faces = relative_position_faces.norm();
            let angle_target = normal_target.angle(&direction_faces);
            let angle_other = normal_other.angle(&-direction_faces);
            if angle_target > TAU / 4.0 || angle_other > TAU / 4.0 {
                continue;
            }
            *view = 2. * area_other * angle_target.cos() * angle_other.cos()
                / (TAU * distance_faces.powi(2));
        }
        view_factor
    }

    /// Compute the diffuse solar radiation from another body of the sytem.
    ///
    /// ## Expression
    ///
    /// The diffuse solar radiation contribution from all $N$ facets $i$ the other body onto the
    /// facet $j$ of the body is defined as,
    ///
    /// $$W_{j}=\sum_{i\cancel{=}j}^{N}V_{ij}\frac{S_\odot A\cos\varsigma_j\left(t\right)}{r_H^2\left(t\right)}$$
    ///
    /// where $V_{ij}$ is the view factor describing the fraction of energy emitted from one facet
    /// $i$ towards the facet $j$, $S_\odot$ is [Solar Constant][SOLAR_CONSTANT], $A$ the albedo,
    /// $\varsigma_j$ the illumination angle of the facet $j$, and $r_H$ the heliocentric distance
    /// in [AU][ASTRONAUMICAL_UNIT].
    pub fn diffuse_solar_radiation(&self, other_body: &Body) -> List<f64> {
        let mut diffuse_solar_radiation = List::zeros(other_body.object.number_faces());
        let direct_solar_flux_from_other = self.direct_solar_flux(other_body)
            * other_body.properties.albedo()
            / (1. - other_body.properties.albedo());
        for (flux, view_factor) in izip!(
            diffuse_solar_radiation.iter_mut(),
            self.view_factor(other_body).column_iter()
        ) {
            *flux = (view_factor.component_mul(&direct_solar_flux_from_other)).sum();
        }
        diffuse_solar_radiation
    }

    /// Compute the direct thermal heating from another body of the system.
    ///
    /// ## Expression
    ///
    /// The direct thermal heating contribution from all $N$ facets $i$ of the other body onto the
    /// facet $j$ of the body is defined as,
    ///
    /// $$u_{j}=\sum_{i\cancel{=}j}^{N}V_{ij}\epsilon\sigma T_{i}^4$$
    ///
    /// where $V_{ij}$ is the view factor describing the fraction of energy emitted from one facet
    /// $i$ towards the facet $j$, $\epsilon$ the emissivity, $\sigma$ the
    /// [Stefan-Boltzmann constant][STEFAN_BOLTZMANN], and $T_i$ the temperature of the facet $i$.
    pub fn direct_thermal_heating(&self, other_body: &Body) -> List<f64> {
        let mut direct_thermal_heating = List::zeros(self.get_target().object.number_faces());
        let primary_temperature: f64 = other_body.surface_temperatures().max();
        for (flux, view_factor) in izip!(
            direct_thermal_heating.iter_mut(),
            self.view_factor(other_body).column_iter()
        ) {
            *flux = (view_factor
                * self.get_target().properties.emissivity()
                * STEFAN_BOLTZMANN
                * primary_temperature.powi(4))
            .sum();
        }
        direct_thermal_heating
    }

    /// Compute the surface heat flux of the targeted body.
    ///
    /// ## Definition
    ///
    /// The surface heat flux is, at minimum, the [direct solar flux][System::direct_solar_flux].
    /// In case the self-heating is enabled, the surface heat flux includes the
    /// [direct thermal self-heating][System::self_direct_thermal_heating] and the
    /// [difusse solar self-radiation][System::self_diffuse_solar_radiation]. In case the system
    /// contains two bodies and the mutual heating is enabled, the surface heat flux also includes
    /// the [direct thermal heating][System::direct_thermal_heating] and the
    /// [diffuse solar radiation][System::diffuse_solar_radiation].
    pub fn compute_surface_heat_flux(&mut self) {
        let mut flux;
        if !self.enable_solar_flux {
            flux = List::zeros(self.get_target().object.number_faces());
        } else {
            flux = self.direct_solar_flux(self.get_target());
            if self.enable_self_heating {
                flux += self.self_diffuse_solar_radiation() + self.self_direct_thermal_heating()
            }
            if self.enable_mutual_heating {
                for other_body in self.iter_other_bodies() {
                    flux += self.diffuse_solar_radiation(other_body)
                        + self.direct_thermal_heating(other_body)
                }
            }
        }
        self.get_target_mut().set_surface_heat_flux(flux);
    }

    /// Compute surface and ground limit heat fluxes and set conditions at boundaries.
    pub(crate) fn compute_and_apply_heat_flux_at_boundaries_conditions(&mut self) {
        let number_faces = self.get_target().object.number_faces();
        let ground_size = self.get_target().properties.ground_vector().len();

        // TODO: insert quick function to recompute mutual view factor

        // Surface temperatures.
        let surface_temperatures;
        if self.get_target().fixed_surface_temperatures() {
            surface_temperatures = self.get_target().surface_temperatures_cloned_vec()
        } else {
            // Compute heat flux at the surface.
            self.compute_surface_heat_flux();
            // Apply surface heat flux.
            surface_temperatures = tool::newton_method(
                self.get_target().surface_temperatures_cloned_vec(),
                newton_method_function,
                newton_method_derivative,
                CustomNewtonMethodArguments::new(
                    self.get_target().ground_temperatures(),
                    self.get_target().surface_heat_flux(),
                    &self.get_target().properties,
                ),
            )
        }
        // Apply.
        self.get_target_mut()
            .set_surface_temperatures(surface_temperatures);

        // Ground limit temperatures.
        if ground_size > 2 {
            let ground_limit_temperatures;
            if self.get_target().fixed_ground_limit_temperatures() {
                let last_column = &self
                    .get_target()
                    .ground_temperatures()
                    .column(ground_size - 1);
                ground_limit_temperatures =
                    List::from_iterator(last_column.len(), last_column.iter().cloned());
            } else {
                // Compute heat flux at the ground limit.
                let ground_limit_heat_flux = List::<f64>::zeros(number_faces);
                // Apply ground limit heat flux.
                let body = self.get_target();
                let properties = &body.properties;
                ground_limit_temperatures = -properties.ground_step() / properties.conductivity()
                    * ground_limit_heat_flux
                    + body
                        .ground_temperatures()
                        .column(ground_size - 2)
                        .transpose();
            }
            // Apply.
            self.get_target_mut()
                .ground_temperatures_mut()
                .set_column(ground_size - 1, &ground_limit_temperatures.transpose());
        }
    }

    /// Apply 1D heat conduction
    ///
    /// # Expression
    ///
    /// $$u\left(x,t+\Delta t\right)=u\left(x,t\right)+\frac{\alpha\Delta t}{\Delta x^2}\left(u\left(x-\Delta x,t\right)-2u\left(x,t\right)+u\left(x+\Delta x,t\right)\right)$$
    pub(crate) fn ground_conduction(&mut self) {
        let ground_size = self.get_target().properties.ground_vector().len();
        if ground_size == 1 {
            return;
        }

        // Define 1D neighbouring layers:
        //     + layer:       temperatures at ([x_1, x_{n-1}], t_{i  })
        //     + layer_up:    temperatures at ([x_0, x_{n-2}], t_{i  })
        //     + layer_down:  temperatures at ([x_2, x_{n  }], t_{i  })
        //     + new_layer:   temperatures at ([x_1, x_{n-1}], t_{i+1})
        let diffusivity_field = *self.get_target().properties.diffusivity_field();
        let ground_temperatures = self.get_target().ground_temperatures().clone();
        let layer_iter = ground_temperatures
            .column_iter()
            .enumerate()
            .filter(|&(i, _)| ![0, ground_size - 1].contains(&i))
            .map(|(_, e)| e);
        let layer_up_iter = ground_temperatures
            .column_iter()
            .enumerate()
            .filter(|&(i, _e)| i < ground_size - 2)
            .map(|(_, e)| e);
        let layer_down_iter = ground_temperatures.column_iter().skip(2);
        let new_layer_iter = self
            .get_target_mut()
            .ground_temperatures_mut()
            .column_iter_mut()
            .enumerate()
            .filter(|&(i, _)| ![0, ground_size - 1].contains(&i))
            .map(|(_, e)| e);

        // Apply 1D ground conduction.
        for (mut new_layer, layer, layer_up, layer_down) in
            multizip((new_layer_iter, layer_iter, layer_up_iter, layer_down_iter))
        {
            new_layer
                .copy_from(&(layer + diffusivity_field * (layer_up - 2. * layer + layer_down)));
        }
    }
}

impl Default for System {
    fn default() -> Self {
        Self::new()
    }
}

/// Struct that holds the parameters to be use for the Newton's method to compute the temperature
/// at the surface of the celestial body.
pub(crate) struct CustomNewtonMethodArguments<'a> {
    ground_temperatures: &'a DMatrix<f64>,
    solar_fluxes: &'a List<f64>,
    properties: &'a Properties,
}

impl<'a> CustomNewtonMethodArguments<'a> {
    /// Constructor.
    fn new(
        ground_temperatures: &'a DMatrix<f64>,
        solar_fluxes: &'a List<f64>,
        properties: &'a Properties,
    ) -> Self {
        CustomNewtonMethodArguments {
            ground_temperatures,
            solar_fluxes,
            properties,
        }
    }
}

/// # Newton's method argument trait
///
/// Implement the trait [`tool::NewtonMethodArguments`] to the struct that holds the paramaters in
/// order to be accepted by the Newton's method as a valid parameter.
impl<'a> tool::NewtonMethodArguments for CustomNewtonMethodArguments<'a> {}

/// Implementation of the function of the Newton's method to compute the temperatures at the
/// surface of the celestial body.
pub(crate) fn newton_method_function(
    value: &List<f64>,
    newton_method_arguments: &CustomNewtonMethodArguments,
) -> List<f64> {
    let mut result = newton_method_arguments.solar_fluxes
        - newton_method_arguments.properties.emissivity() * STEFAN_BOLTZMANN * tool::pows(value, 4);
    // If the body ground vector size is only the surface, consider the 2nd and 3rd layer the same
    // as the surface
    if newton_method_arguments.properties.ground_vector().len() > 2 {
        result += (-newton_method_arguments
            .ground_temperatures
            .column(2)
            .transpose()
            + 4. * newton_method_arguments
                .ground_temperatures
                .column(1)
                .transpose()
            - 3. * value)
            * newton_method_arguments.properties.conductivity()
            / (2. * newton_method_arguments.properties.ground_step());
    }
    result
}

/// Implementation of the derivative of the function of the Newton's method to compute the
/// temperature at the surface of the celestial body.
pub(crate) fn newton_method_derivative(
    value: &List<f64>,
    newton_method_arguments: &CustomNewtonMethodArguments,
) -> List<f64> {
    let mut result = -4.
        * newton_method_arguments.properties.emissivity()
        * STEFAN_BOLTZMANN
        * tool::pows(value, 3);
    if newton_method_arguments.properties.ground_vector().len() > 2 {
        result = result.add_scalar(
            -3. * newton_method_arguments.properties.conductivity()
                / (2. * newton_method_arguments.properties.ground_step()),
        );
    }
    result
}
