# kalast

[![logo image]][crate link]

[![crate badge]][crate link]
[![doc badge]][doc link]
[![license badge]][license link]
[![pre-commit badge]][pre-commit link]

> Next generation thermophysical modelling of asteroids

---

[Demo](#demo) |
[In Action](#in-action) |
[Installation](#installation) |
[If you clone this repo](#if-you-clone-this-repo) |
[License](#license)

---

## Demo

[![demo screencast]][crate link]

## In action

Computation of the temperature of a single point at the surface of an asteroid
(initialized as sub-solar-point):

```rust
// src/main.rs

fn main() {
    // Initialize the library.
    let mut kalast = kalast::init();

    // System center of mass position.
    let dau = tool::ASTRONAUMICAL_UNIT * 1.664;
    let position = tool::Vector::new(1.0, 0.0, 0.0) * dau;
    kalast.set_system_position(position);

    // Add a body to the system without a mesh.
    kalast.add_body(
        "Asteroid",            // name
        None,                  // frame
        tool::Vector::zeros(), // relative position to center of mass
        kalast::Properties::new(
            11.92 * tool::HOUR,    // revolution period
            162.0 * tool::DEG2RAD, // obliquity
            500.,                  // thermal inertia
            2146.,                 // density
            600.,                  // heat capacity
            0.07,                  // albedo
            0.9,                   // emissivity
        ),
        Some("rsc/obj/ellipsoid.obj"), // path to 3D mesh
    );

    // Some configurations about the system.
    kalast.enable_self_heating(false);

    // Set simulation duration and time step.
    kalast.set_time_with_revolution_target(30.0, 10.0 * tool::MINUTE);

    // Some configuration for exporting before running the simulation.
    kalast.export_surface(true);
    kalast.export_path("rsc/data/tmp/simple.json");

    // Start the simulation.
    kalast.start();

    // Some prints.
    let temp = kalast.get_target().surface_temperatures();
    println!(
        "Surface temperatures > max: {:.2} K, min: {:.2} K",
        temp.max(),
        temp.min()
    );
}

```

## Installation

Get [Rust][rust install]:

```sh
curl --proto '=https' --tlsv1.2 -sSf https://sh.rustup.rs | sh
```

Create a new Rust project:

```sh
# Move to the directory of your choice and create a project
cargo new my_project

# Move inside your project
cd my_project
```

Add the dependency **kalast** to your `Cargo.toml`:

```toml
...
[dependencies]
kalast = "*" # replace * by the latest version of the crates
```

To get a working code:

+ copy to your `src/main.rs` either the code [above](#in-action) or one from the
  [examples](examples)
+ get the necessary 3D object [files](rsc/obj)
+ make sure the paths to your 3D file are correct
+ build and run your `src/main.rs` from the root of your project with:

```sh
cargo run --release
```

## If you clone this repo

To build and run the program, tests or examples:

```sh
Usage: ./compile.py [OPTIONS] COMMAND [ARGS]...

  Rust Project Builder Assistant.

  By default, compile the binary from the main program in debug mode, and
  run it. If you want to compile in release mode, for the main command or
  any subcommand, use the release option.

Options:
  -v, --version  Show version.
  -r, --release  Enable release mode.
  -h, --help     Show this message and exit.

Commands:
  doc              Build documentation.
  example          Compile and run an example.
  publish          Publish your library on crates.io.
  tests            Compile and execute unit and integration tests.
  update-versions  Update versions in all files according to the manifest.
```

Some examples:

```sh
# Compile and run src/main.rs in debug mode
./compile.py
# Compile and run src/main.rs in release mode
./compile.py -r
# Compile and run examples/spheres_mutual.rs in release mode
./compile.sh -r example spheres_mutual
# Execute all unit and integration tests
./compile.sh tests -a
```

This compile script requires few dependencies:

```sh
dotmap=*
envtoml=*
pathlib=*
subprocess=*
termcolor=*
typing=*
re=*
```

But if you want to compile yourself, follow [this][cargo run link] link.

Do not forget to work on a development branch:

```sh
# To create the branch.
git checkout -b dev
```

The projet uses [pre-commit][pre-commit link] to prevent mistakes before commits.
To install the hook scripts:

```sh
pre-commit install
```

A hook script uses an external tool called [embedme][embedme link] to ensure the
demo code displayed in the section [In action](#in-action) matches automatically
the code in src/main.rs without having to copy paste it.

## License

Licensed under the [Apache License, Version 2.0][license link].

[repository link]: https://gitlab-as.oma.be/sbm/kalast
[logo image]: https://gitlab-as.oma.be/sbm/kalast/-/raw/main/rsc/img/all/SBMT-dimorphos.png
[crate link]: https://crates.io/crates/kalast
[crate badge]: https://meritbadge.herokuapp.com/kalast?style=flat-square
[doc link]: https://docs.rs/kalast
[doc badge]: https://docs.rs/kalast/badge.svg
[license link]: https://gitlab-as.oma.be/sbm/kalast/-/raw/main/LICENSE
[license badge]: https://img.shields.io/badge/License-Apache%202.0-blue.svg
[pre-commit link]: https://pre-commit.com
[pre-commit badge]: https://img.shields.io/badge/pre--commit-enabled-brightgreen?logo=pre-commit&logoColor=white
[demo screencast]: https://gitlab-as.oma.be/sbm/kalast/-/raw/main/rsc/cast/demo.gif
[rust install]: https://www.rust-lang.org/learn/get-started
[cargo run link]: https://doc.rust-lang.org/cargo/commands/cargo-run.html
[embedme link]: https://github.com/zakhenry/embedme
