use crate::{
    keys::{
        doc::gen_doc_comment,
        meta::{get_meta, MetaSpec},
        seq::{gen_field_assignments, gen_field_defs},
    },
    utils::{get_attribute, prop_err, sc_to_ucc, MacroError, StackTrace},
    Result,
};

use proc_macro2::{Ident, Span, TokenStream};
use quote::{quote, ToTokens};
use yaml_rust::{yaml, Yaml};

pub fn get_types(info: &TypeInfo) -> Result<Vec<(Ident, yaml::Hash)>> {
    let map = info.map;
    let types = match get_attribute!(map; "types" as Yaml::Hash(h) => h; "get_types") {
        Ok(t) => t,
        // types is not a required field in ksy.
        Err(MacroError::RequiredAttrNotFound(..)) => return Ok(Vec::new()),
        Err(e) => return Err(e),
    };
    let mut result = Vec::new();

    for (name, ty) in types {
        let ident = match name {
            Yaml::String(s) => Ident::new(&sc_to_ucc(s), Span::call_site()),
            _ => {
                return Err(MacroError::InvalidAttrType {
                    attr: "type name".to_owned(),
                    pat: "Yaml::String(s)".to_owned(),
                    actual: name.clone(),
                    st: StackTrace::from("get_types"),
                })
            }
        };
        match ty {
            Yaml::Hash(h) => result.push((ident, h.clone())),
            _ => {
                return Err(MacroError::InvalidAttrType {
                    attr: "type".to_owned(),
                    pat: "Yaml::Hash(h)".to_owned(),
                    actual: ty.clone(),
                    st: StackTrace::from("get_types"),
                })
            }
        }
    }

    Ok(result)
}

pub fn gen_type_defs(info: &TypeInfo) -> Result<Vec<TokenStream>> {
    let types = prop_err!(get_types(info); "gen_type_defs");
    let mut result = Vec::new();

    for (ident, ty) in types {
        let meta = prop_err!(get_meta(info); "gen_type_defs");
        let child_info = TypeInfo {
            map: &ty,
            ident,
            attrs: Vec::new(),
            visibility: info.visibility.clone(),
            inherited_meta: Some(meta),
        };

        result.push(prop_err!(gen_type(&child_info); "gen_type_defs"));
    }

    Ok(result)
}

#[derive(Clone, Eq, PartialEq, Hash, Debug)]
pub struct TypeInfo<'a> {
    /// The hash map containing the necessary data to generate the type.
    pub map: &'a yaml::Hash,
    /// The ident that should be used for the generated struct.
    pub ident: proc_macro2::Ident,
    /// The attributes that should be applied to the generated struct.
    pub attrs: Vec<syn::Attribute>,
    /// The desired visibility of the generated struct.
    pub visibility: syn::Visibility,
    /// The meta information that applies to the type. The field is always set to
    /// [`Some`](Option::Some) if the [`TypeOptions`] is describing a user-defined type (i.e. any
    /// type specified in the `TypesSpec` of the ksy file).
    pub inherited_meta: Option<MetaSpec>,
}

/// Function that generates a Rust struct definition from a `yaml::Hash` in the format of a KSY
/// `TypeSpec`. The function is called recursively on all the types of the given `TypeSpec`. The
/// `struct`s generated by the function are public.
pub fn gen_type(info: &TypeInfo) -> Result<TokenStream> {
    let map = info.map;
    let type_defs = prop_err!(gen_type_defs(&TypeInfo {
        map,
        // The child inherits the parents `MetaSpec`.
        inherited_meta: Some(info.inherited_meta.clone().unwrap_or(prop_err!(get_meta(info); "gen_type"))),
        ..info.clone()
    }); "gen_type");

    let doc_comment = prop_err!(gen_doc_comment(map); "gen_type");

    let attrs: Vec<proc_macro2::TokenStream> =
        info.attrs.iter().map(|a| a.to_token_stream()).collect();

    let visibility = &info.visibility;

    let ident = &info.ident;

    let field_defs = prop_err!(gen_field_defs(map); "gen_type");

    let field_assignments = prop_err!(gen_field_assignments(info); "gen_type");

    let result = quote! {
        #(#type_defs)*

        #doc_comment
        #(#attrs)*
        #visibility struct #ident {
            #(#field_defs,)*
        }

        // The `automatically_derived` attribute is applied to implementations generated by derive
        // macros. Since this `kaitai_source` isn't a derive macro, the `automatically_derived`
        // attributed must be applied manually.
        #[automatically_derived]
        impl ::kaitai::runtime::KaitaiStruct for #ident {
            fn new<S: ::kaitai::runtime::KaitaiStream>(buf: &mut S) -> ::kaitai::Result<Self> {
                Ok(#ident {
                    #(#field_assignments,)*
                })
            }
            fn read<S: ::kaitai::runtime::KaitaiStream>(&mut self, _: &mut S) -> ::kaitai::Result<()> {
                todo!();
            }
        }
    };
    Ok(result)
}
