mod aws;

use argh::FromArgs;
use async_compression::{
	tokio::{bufread::ZstdDecoder, write::ZstdEncoder}, Level
};
use futures::{
	future::{try_join_all, Fuse}, FutureExt, StreamExt
};
use pin_project::pin_project;
use rusoto_core::{HttpClient, Region};
use rusoto_credential::StaticProvider;
use rusoto_s3::S3Client;
use std::{
	collections::BTreeMap, env, error::Error, fs, future::Future, io, path::{Path, PathBuf}, pin::Pin, str::FromStr, sync::Arc, task::{Context, Poll}
};
use tokio::io::{AsyncBufRead, AsyncRead, AsyncReadExt, AsyncWriteExt, ReadBuf};
use walkdir::WalkDir;

const WEB_SAFE: &str = "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789-_~";

#[derive(FromArgs, Debug)]
/// command
struct Cmd {
	#[argh(subcommand)]
	cmd: Cmd_,
}

#[derive(FromArgs, Debug)]
#[argh(subcommand)]
enum Cmd_ {
	Cache(CmdCache),
	Uncache(CmdUncache),
}

#[derive(FromArgs, Debug)]
/// cache
#[argh(subcommand, name = "cache")]
struct CmdCache {
	#[argh(positional)]
	/// keys
	keys: Vec<String>,
}

#[derive(FromArgs, Debug)]
/// uncache
#[argh(subcommand, name = "uncache")]
struct CmdUncache {
	#[argh(positional)]
	/// keys. will try from left to right
	keys: Vec<String>,
}

#[tokio::main]
async fn main() -> Result<(), Box<dyn Error>> {
	env::set_var("RUST_BACKTRACE", "1");
	let cmd: Cmd = argh::from_env();

	let http_client = Arc::new(HttpClient::new().expect("failed to create request dispatcher"));
	let creds = StaticProvider::new(env::var("AWS_ACCESS_KEY_ID").unwrap(), env::var("AWS_SECRET_ACCESS_KEY").unwrap(), None, None);
	let s3_client = &S3Client::new_with(http_client.clone(), creds.clone(), Region::from_str(&env::var("AWS_REGION").unwrap()).unwrap());
	let bucket = &env::var("AWS_BUCKET").unwrap();

	let target_dir = PathBuf::from(env::var("TARGET_DIR").unwrap());
	let cargo_dir = PathBuf::from(env::var("CARGO_HOME").unwrap());

	match cmd.cmd {
		Cmd_::Cache(CmdCache { keys }) => {
			let id: [u8; 16] = rand::random();
			let id = base_x::encode(WEB_SAFE, &id);
			let id = &format!("/blobs/{}.tar.zst", id);
			let tar = tar(walk(PathBuf::from("cargo"), cargo_dir).chain(walk(PathBuf::from("target"), target_dir)));
			aws::upload(s3_client, bucket.clone(), id.clone(), String::from("application/zstd"), Some(31536000), tar).await?;
			let _: Vec<()> = try_join_all(keys.into_iter().map(|key| async move {
				let key = format!("/keys/{}.txt", key);
				aws::upload(s3_client, bucket.clone(), key, String::from("text/plain"), Some(600), id.as_bytes()).await
			}))
			.await?;
		}
		Cmd_::Uncache(CmdUncache { keys }) => {
			for key in keys {
				let key = format!("/keys/{}.txt", key);
				if let Ok(mut id_) = aws::download(s3_client, bucket.clone(), key).await {
					let mut id = String::new();
					let _ = id_.read_to_string(&mut id).await?;
					let id = format!("/blobs/{}.tar.zst", id);
					if let Ok(tar) = aws::download(s3_client, bucket.clone(), id).await {
						return Ok(untar(
							vec![(PathBuf::from("cargo"), cargo_dir), (PathBuf::from("target"), target_dir)].into_iter().collect(),
							tar,
						)
						.await?);
					}
				}
			}
			return Err("cache miss :(".into());
		}
	}
	Ok(())
}

fn walk(slug: PathBuf, base: PathBuf) -> impl Iterator<Item = (PathBuf, PathBuf, PathBuf)> {
	WalkDir::new(&base).sort_by(|a, b| a.file_name().cmp(b.file_name())).into_iter().filter_map(move |entry| {
		let entry = entry.unwrap();
		let t = entry.file_type();
		let path = entry.into_path().strip_prefix(&base).unwrap().to_owned();
		let non_empty = path.components().next().is_some();
		if non_empty && (t.is_file() || t.is_symlink() || t.is_dir()) { Some((slug.clone(), base.clone(), path)) } else { None }
	})
}

fn tar(paths: impl Iterator<Item = (PathBuf, PathBuf, PathBuf)>) -> impl AsyncRead {
	let (writer, reader) = async_pipe::pipe();
	let task = async move {
		// create a .tar.zsd
		// https://community.centminmod.com/threads/round-4-compression-comparison-benchmarks-zstd-vs-brotli-vs-pigz-vs-bzip2-vs-xz-etc.18669/
		let tar_ = ZstdEncoder::with_quality(writer, Level::Precise(6));
		let mut tar_ = tokio_tar::Builder::new(tar_);
		tar_.mode(tokio_tar::HeaderMode::Complete);

		// add resources (check for docker images) to tar
		for (slug, base, relative) in paths {
			tar_.append_path_with_name(base.join(&relative), slug.join(&relative)).await.unwrap();
		}

		// flush writers
		let mut tar_ = tar_.into_inner().await.unwrap();
		tar_.shutdown().await.unwrap();
		let _tar = tar_.into_inner();
	};

	AlsoPollFuture { reader, task: task.fuse() }
}

async fn untar(slugs: BTreeMap<PathBuf, PathBuf>, tar: impl AsyncBufRead) -> Result<(), io::Error> {
	tokio::pin!(tar);
	let tar = ZstdDecoder::new(tar);
	let tar = tokio::io::BufReader::new(tar);

	let mut entries = tokio_tar::Archive::new(tar).entries().unwrap();

	while let Some(entry) = entries.next().await {
		let mut entry = entry.unwrap();
		let path = entry.path().unwrap().into_owned();
		let slug = path.components().next().unwrap();
		let slug: &Path = slug.as_ref();
		let path = path.strip_prefix(slug).unwrap();
		let base = slugs.get(slug).unwrap();
		let path = base.join(path);
		let _ = fs::create_dir_all(&path.parent().unwrap());
		let _ = entry.unpack(&path).await.unwrap();
	}

	Ok(())
}

#[pin_project]
struct AlsoPollFuture<R, F> {
	#[pin]
	reader: R,
	#[pin]
	task: Fuse<F>,
}
impl<R, F> AsyncRead for AlsoPollFuture<R, F>
where
	R: AsyncRead,
	F: Future<Output = ()>,
{
	fn poll_read(self: Pin<&mut Self>, cx: &mut Context<'_>, buf: &mut ReadBuf<'_>) -> Poll<io::Result<()>> {
		let self_ = self.project();
		let _ = self_.task.poll(cx);
		self_.reader.poll_read(cx, buf)
	}
}

#[cfg(test)]
mod tests {
	use super::*;

	#[tokio::test]
	#[ignore]
	async fn loopback() -> Result<(), Box<dyn Error>> {
		let target_dir = PathBuf::from(env::var("TARGET_DIR").unwrap());
		let cargo_dir = PathBuf::from(env::var("CARGO_HOME").unwrap());

		let paths = walk(PathBuf::from("cargo"), cargo_dir).chain(walk(PathBuf::from("target"), target_dir));
		let tar = tar(paths);
		untar(
			vec![(PathBuf::from("cargo"), PathBuf::from("/tmp/cargo")), (PathBuf::from("target"), PathBuf::from("/tmp/target"))]
				.into_iter()
				.collect(),
			tokio::io::BufReader::new(tar),
		)
		.await?;

		// rsync -avnc --delete target/ target2
		// https://unix.stackexchange.com/questions/57305/rsync-compare-directories/351112#351112
		Ok(())
	}
}
