use super::*;

impl AWSClusterSpec {
    pub fn region(&self) -> Option<&str> {
        self.region.as_deref()
    }

    pub fn network(&self) -> Option<&NetworkSpec> {
        self.network.as_ref()
    }

    pub fn vpc(&self) -> Option<&VPCSpec> {
        self.network().and_then(|network| network.vpc.as_ref())
    }

    pub fn vpc_id(&self) -> Option<&str> {
        self.vpc().and_then(|vpc| vpc.id.as_deref())
    }
}

impl AWSClusterStatus {
    pub fn ready(&self) -> bool {
        self.ready
    }
    pub fn network(&self) -> Option<&NetworkStatus> {
        self.network_status.as_ref()
    }
}

impl AWSCluster {
    pub fn with_name(name: &str) -> Self {
        let spec = AWSClusterSpec::default();
        Self::new(name, spec)
    }

    pub fn set_region(self, region: &str) -> Self {
        Self {
            spec: AWSClusterSpec {
                region: Some(region.to_string()),
                ..self.spec
            },
            ..self
        }
    }

    pub fn set_sshkey(self, sshkey: &str) -> Self {
        Self {
            spec: AWSClusterSpec {
                ssh_key_name: Some(sshkey.to_string()),
                ..self.spec
            },
            ..self
        }
    }

    pub fn set_vpc_cidr_block(self, vpc_cidr_block: &str) -> Self {
        let network = self.spec.network.unwrap_or_default();
        let vpc = network.vpc.unwrap_or_default();
        let vpc = VPCSpec {
            cidr_block: Some(vpc_cidr_block.to_string()),
            ..vpc
        };
        let network = NetworkSpec {
            vpc: Some(vpc),
            ..network
        };

        Self {
            spec: AWSClusterSpec {
                network: Some(network),
                ..self.spec
            },
            ..self
        }
    }

    pub fn add_subnet(self, availability_zone: &str, cidr_block: &str, is_public: bool) -> Self {
        let network = self.spec.network.unwrap_or_default();
        let mut subnets = network.subnets.unwrap_or_default().0;
        subnets.push(network::SubnetSpec {
            availability_zone: Some(availability_zone.to_string()),
            cidr_block: Some(cidr_block.to_string()),
            is_public: Some(is_public),
            ..Default::default()
        });
        let network = NetworkSpec {
            subnets: Some(network::Subnets(subnets)),
            ..network
        };
        Self {
            spec: AWSClusterSpec {
                network: Some(network),
                ..self.spec
            },
            ..self
        }
    }
}
