//! Provides a wrapper around the `sha2` crate for K-Anonymous SHA256
//!
//! Usage:
//! ```
//! use k_anon_hash::hash::KAnonHash;
//!
//! let data = String::from("remember to drink water uwu");
//! let hash = KAnonHash::calculate_string_hash(&data, 3);
//! println!("K-Anonymous hash: {}", hash.k_anon());
//! ```

pub mod hash;

#[cfg(test)]
mod tests {
    use crate::hash::KAnonHash;

    const TEST_BIT_LEN: u8 = 3;

    #[test]
    fn it_works() {
        let hash = KAnonHash::calculate_string_hash(&String::from("test uwu"), TEST_BIT_LEN);

        assert_eq!(
            hash.to_string(),
            String::from("f29647e40fa8e71f059fae548c4179adba008e29237aae93d37709d059a69e7e")
        );
        assert_eq!(hash.k_anon(), 2);
    }

    #[test]
    fn large_mask() {
        let hash = KAnonHash::calculate_string_hash(&String::from("fox is soft uwu"), 6);

        assert_eq!(
            hash.to_string(),
            String::from("227cb7692de461abd56166c95d704bd46f1c5e247635209fec3560af8fbfee9f")
        );
        assert_eq!(hash.k_anon(), 34);
    }

    #[test]
    fn to_from_string() {
        let hash = KAnonHash::calculate_string_hash(
            &String::from("<script>alert(document.domain)</script>"),
            TEST_BIT_LEN,
        );

        assert_eq!(
            hash.to_string(),
            String::from("09ef621068dedf82998a50bc0a60b9edbc2462765d6e384fab045af8843cd60b")
        );
        assert_eq!(
            KAnonHash::from_str(
                "09ef621068dedf82998a50bc0a60b9edbc2462765d6e384fab045af8843cd60b",
                TEST_BIT_LEN
            )
            .unwrap(),
            hash
        );
    }
}
