use sha2::{Digest, Sha256};
use std::{convert::TryInto, fmt::Display};

#[derive(Copy, Clone, Debug)]
pub struct KAnonHash {
    bitmask: u64,
    hash: [u8; 0x20],
}

impl KAnonHash {
    pub const KANON_MASK: u32 = 0b11;

    /// Calculates a hash over `data` and creates a nee KAnonHash wrapper with the mask of `bit_count` bits, `bit_count` is maxed at 64
    pub fn calculate_hash(data: &[u8], bit_count: u8) -> Self {
        Self {
            bitmask: (2u64.pow(bit_count.min(64) as u32)) - 1,
            hash: Sha256::digest(data)[..].try_into().unwrap(),
        }
    }

    /// Calculates a hash over `data` and creates a nee KAnonHash wrapper with the mask of `bit_count` bits, `bit_count` is maxed at 64. Internally calls `calculate_hash` over data as bytes
    pub fn calculate_string_hash(data: &str, bit_count: u8) -> Self {
        Self::calculate_hash(data.as_bytes(), bit_count)
    }

    /// Returns the K-Anonymous part of the hash
    pub fn k_anon(&self) -> u64 {
        u64::from_le_bytes(self.hash[0..8].try_into().unwrap()) & self.bitmask
    }

    /// Returns an immutable reference to the internal hash buffer
    pub fn as_slice(&self) -> &[u8; 0x20] {
        &self.hash
    }

    pub fn from_str(s: &str, bit_count: u8) -> Result<Self, String> {
        let val = hex::decode(s).map_err(|e| e.to_string())?;
        Ok(Self {
            bitmask: (2u64.pow(bit_count as u32)) - 1,
            hash: val[..]
                .try_into()
                .map_err(|_| String::from("Hash too small!"))?,
        })
    }
}

impl PartialEq for KAnonHash {
    fn eq(&self, other: &Self) -> bool {
        self.hash == other.hash
    }
}

impl Display for KAnonHash {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        f.write_str(hex::encode(&self.hash).as_str())
    }
}
