// Copyright (C) 2018  Aprila Bank ASA
//
// alcoholic_jwt is free software: you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation, either version 3 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.

use super::*;

#[test]
fn test_decode_find_jwks() {
    let json = "{\"keys\":[{\"kty\":\"RSA\",\"alg\":\"RS256\",\"use\":\"sig\",\"kid\":\"mUjI\\/rIMLLtung35BKZfdbrqtlEAAYJ4JX\\/SKvnLxJc=\",\"n\":\"ngRRjNbXgPW29oNtF0JgsyyfTwPyEL0u_X16s453X2AOc33XGFxVKLEQ7R_TiMenaKcr-tPifYqgps_deyi0XOr4I3SOdOMtAVKDZJCANe--CANOHZb-meIfjKhCHisvT90fm5Apd6qPRVsXsZ7A8pmClZHKM5fwZUkBv8NsPLm2Xy2sGOZIiwP_7z8m3j0abUzniPQsx2b3xcWimB9vRtshFHN1KgPUf1ALQ5xzLfJnlFkCxC7kmOxKC7_NpQ4kJR_DKzKFV_r3HxTqf-jddHcXIrrMcLQXCSyeLQtLaz7whQ4F-EfL42z4XgwPr4ji3sct2gWL13EqlbE5DDxLKQ\",\"e\":\"GK7oLCDbNPAF59LhvyseqcG04hDnPs58qGYolr_HHmaR4lulWJ90ozx6e4Ut363yKG2p9vwvivR5UIC-aLPtqT2qr-OtjhBFzUFVaMGZ6mPCvMKk0AgMYdOHvWTgBSqQtNJTvl1yYLnhcWyoE2fLQhoEbY9qUyCBCEOScXOZRDpnmBtz5I8q5yYMV6a920J24T_IYbxHgkGcEU2SGg-b1cOMD7Rja7vCfV---CQ2pR4leQ0jufzudDoe7z3mziJm-Ihcdrz2Ujy5kPEMdz6R55prJ-ENKrkD_X4u5aSlSRaetwmHS3oAVkjr1JwUNbqnpM-kOqieqHEp8LUmez-Znw\"}]}";
    let jwks: JWKS = serde_json::from_str(json).expect("Failed to decode JWKS");
    let jwk = jwks
        .find("mUjI/rIMLLtung35BKZfdbrqtlEAAYJ4JX/SKvnLxJc=")
        .expect("Failed to find required JWK");

    public_key_from_jwk(&jwk).expect("Failed to construct public key from JWK");
}

#[test]
fn test_token_kid() {
    let jwt = "eyJraWQiOiI4ckRxOFB3MEZaY2FvWFdURVZRbzcrVGYyWXpTTDFmQnhOS1BDZWJhYWk0PSIsImFsZyI6IlJTMjU2IiwidHlwIjoiSldUIn0.eyJpc3MiOiJhdXRoLnRlc3QuYXByaWxhLm5vIiwiaWF0IjoxNTM2MDUwNjkzLCJleHAiOjE1MzYwNTQyOTMsInN1YiI6IjQyIiwiZXh0Ijoic21va2V0ZXN0IiwicHJ2IjoiYXJpc3RpIiwic2NwIjoicHJvY2VzcyJ9.gOLsv98109qLkmRK6Dn7WWRHLW7o8W78WZcWvFZoxPLzVO0qvRXXRLYc9h5chpfvcWreLZ4f1cOdvxv31_qnCRSQQPOeQ7r7hj_sPEDzhKjk-q2aoNHaGGJg1vabI--9EFkFsGQfoS7UbMMssS44dgR68XEnKtjn0Vys-Vzbvz_CBSCH6yQhRLik2SU2jR2L7BoFvh4LGZ6EKoQWzm8Z-CHXLGLUs4Hp5aPhF46dGzgAzwlPFW4t9G4DciX1uB4vv1XnfTc5wqJch6ltjKMde1GZwLR757a8dJSBcmGWze3UNE2YH_VLD7NCwH2kkqr3gh8rn7lWKG4AUIYPxsw9CB";

    let kid = token_kid(&jwt).expect("Failed to extract token KID");
    assert_eq!(
        Some("8rDq8Pw0FZcaoXWTEVQo7+Tf2YzSL1fBxNKPCebaai4=".into()),
        kid,
        "Extracted KID did not match expected KID"
    );
}

#[test]
fn test_validate_jwt() {
    let jwks_json = r#"{"keys":[{"kid":"dQE-dH8KdCihQZouWLJd0yai1bl9rWPzr9grU872fkw","kty":"RSA","alg":"RS256","use":"sig","n":"rzRoSYHU6V8hl8I3m2LYDJVuzqfjYpamEUMF-Ra28HpVkbMhLElwsj1dGYLeOKfeYmbxsGu_ToYC4yFGMBmYubWOkXjYhG-87DIzu0R2kcLPtagTsQFukj7HkQSUOcgze3z4hm4oWtWbjxqIrf7ELl2kJ1bONLsIgfB6h7Lfk3wcECwTnn_SjK27TtscIXJuGenlgGzoYtDafXMbF3Bev0W4BH_r0WwAR0mONa251xFc5LwOUwme7jKKKTrNGfoAHTV_nK0j6DY4DYkQPDNZiJmU47fi0saZltVXmf_rFcpGWaUXdZ-6MFpDyyj-zxA1pCZldR0qyClS5rqmTQRurw","e":"AQAB","x5c":["MIICpTCCAY0CBgF4OeNrijANBgkqhkiG9w0BAQsFADAWMRQwEgYDVQQDDAt5ZWxsb3ctemluYzAeFw0yMTAzMTYwNzEzMjhaFw0zMTAzMTYwNzE1MDhaMBYxFDASBgNVBAMMC3llbGxvdy16aW5jMIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEArzRoSYHU6V8hl8I3m2LYDJVuzqfjYpamEUMF+Ra28HpVkbMhLElwsj1dGYLeOKfeYmbxsGu/ToYC4yFGMBmYubWOkXjYhG+87DIzu0R2kcLPtagTsQFukj7HkQSUOcgze3z4hm4oWtWbjxqIrf7ELl2kJ1bONLsIgfB6h7Lfk3wcECwTnn/SjK27TtscIXJuGenlgGzoYtDafXMbF3Bev0W4BH/r0WwAR0mONa251xFc5LwOUwme7jKKKTrNGfoAHTV/nK0j6DY4DYkQPDNZiJmU47fi0saZltVXmf/rFcpGWaUXdZ+6MFpDyyj+zxA1pCZldR0qyClS5rqmTQRurwIDAQABMA0GCSqGSIb3DQEBCwUAA4IBAQAsUzQZBpsb8cQZOuwXvCPztnohnuVVrSF4O2sxG9Zqi171B4fqHAadDc9AC+kfhSOCnTcfCS01LDiiBLnXpyMyNDl7CT/iKQNhWRcyS+MOWnT4xzsFjDg2w3ybvXw6OsFxLeou0ojHTUZjjF79Tw3G0DM7a/0k02098O6BKggbu9n+6XCuPwFC62X7C0pVQMosYlXeLR6Xm1nbJEm9mE+ssUaeMGD2eROVMl1bhZsWVEW/XUtHcQO8Geiahx8F+qRy3Lm+CiNYpcNI1mlRO102RoNMCo6x0+RkGvMTVwsGlWJFvMVfo/esn1sOTCdGhflGjE1Fg46XsKMNUNCRNmp5"],"x5t":"3KUWhXTnLBn6e759Zjz2IgktBpo","x5t#S256":"O67T_fxFPl5V_pjGPtAsvTP0i-joKU3zciCGXArnWUA"}]}"#;

    let jwks: JWKS = serde_json::from_str(jwks_json).expect("Failed to decode JWKS");

    let jwk = jwks
        .find("dQE-dH8KdCihQZouWLJd0yai1bl9rWPzr9grU872fkw")
        .expect("Failed to find required JWK");

    let pkey = public_key_from_jwk(&jwk).expect("Failed to construct public key");

    let jwt = JWT("eyJhbGciOiJSUzI1NiIsInR5cCIgOiAiSldUIiwia2lkIiA6ICJkUUUtZEg4S2RDaWhRWm91V0xKZDB5YWkxYmw5cldQenI5Z3JVODcyZmt3In0.eyJleHAiOjE2MjY0NDY2MDQsImlhdCI6MTYyNjQ0NjU0NCwiYXV0aF90aW1lIjoxNjI2NDQ2NTQ0LCJqdGkiOiI0ZjAxOWQwNi1kY2M1LTRhOTktYmYxZC1kNDRhZjRmZjFmZTIiLCJpc3MiOiJodHRwOi8vbG9jYWxob3N0OjYzMTQvYXV0aC9yZWFsbXMveWVsbG93LXppbmMiLCJhdWQiOlsicmVhbG0tbWFuYWdlbWVudCIsImFjY291bnQiXSwic3ViIjoiMTA0MWMwZjAtNDJlZi00ODQ0LTk2MGQtOTU0MTlkNmFhMDhlIiwidHlwIjoiQmVhcmVyIiwiYXpwIjoieWVsbG93LXppbmMiLCJub25jZSI6IjVhZTI4MWM3LTliZTItNGM3My04ZTliLTc1Y2M5ZTc5ZjQ0YyIsInNlc3Npb25fc3RhdGUiOiI0NzJjMTQxNy0yNGJlLTQ0ZDctOTQxOC0zYjg0MzRlNjJlMTEiLCJhY3IiOiIxIiwiYWxsb3dlZC1vcmlnaW5zIjpbImh0dHA6Ly9sb2NhbGhvc3Q6MzAwMCJdLCJyZWFsbV9hY2Nlc3MiOnsicm9sZXMiOlsib2ZmbGluZV9hY2Nlc3MiLCJ1bWFfYXV0aG9yaXphdGlvbiJdfSwicmVzb3VyY2VfYWNjZXNzIjp7InJlYWxtLW1hbmFnZW1lbnQiOnsicm9sZXMiOlsidmlldy1yZWFsbSIsInZpZXctaWRlbnRpdHktcHJvdmlkZXJzIiwibWFuYWdlLWlkZW50aXR5LXByb3ZpZGVycyIsImltcGVyc29uYXRpb24iLCJtYW5hZ2UtdXNlcnMiLCJxdWVyeS1yZWFsbXMiLCJ2aWV3LWF1dGhvcml6YXRpb24iLCJxdWVyeS1jbGllbnRzIiwicXVlcnktdXNlcnMiLCJ2aWV3LWV2ZW50cyIsInZpZXctdXNlcnMiLCJ2aWV3LWNsaWVudHMiLCJtYW5hZ2UtYXV0aG9yaXphdGlvbiIsInF1ZXJ5LWdyb3VwcyJdfSwieWVsbG93LXppbmMiOnsicm9sZXMiOlsiYWNjZXNzIiwiZWRpdCJdfSwiYWNjb3VudCI6eyJyb2xlcyI6WyJtYW5hZ2UtYWNjb3VudCIsIm1hbmFnZS1hY2NvdW50LWxpbmtzIiwidmlldy1wcm9maWxlIl19fSwic2NvcGUiOiJvcGVuaWQgcHJvZmlsZSBlbWFpbCIsImVtYWlsX3ZlcmlmaWVkIjpmYWxzZSwiaHR0cHM6Ly9oYXN1cmEuaW8vand0L2NsYWltcyI6eyJ4LWhhc3VyYS1kZWZhdWx0LXJvbGUiOiJhY2Nlc3MiLCJ4LWhhc3VyYS11c2VyLWlkIjoiMTA0MWMwZjAtNDJlZi00ODQ0LTk2MGQtOTU0MTlkNmFhMDhlIiwieC1oYXN1cmEtYWxsb3dlZC1yb2xlcyI6WyJhY2Nlc3MiLCJlZGl0Il19LCJwcmVmZXJyZWRfdXNlcm5hbWUiOiJhZG1pbiJ9.exnBrUm7UMyx0P54DJZgwX95bq1VJv6JAFtHgOgHe9ej5vJ6ycu5Yq5U3aetj7kSJ1OvmNpnCJZ64ZSWxQtWt_uD0DcgjcvYwkObms9mLMeZDCj-IkR1_9pd7FMWGqgX-6mDV7u6jdUSi0xoJ0snUxkWY5CygSq1SqOh2UEPerNX4BiWgio_Su2JoSFBaQICidED-FRRg5M-d4PmTbe-2hNGX4-mozwhWaeqawc0t4KcpBC2pVHdLM2BTVMzsrzpFoK8u7uFOjBBI0uQril7Eiwj7kdzozBQekbhPX_VXfM0edtymhfL8VmP11pxlI5y_elU9txQQSPjrZQMkYIsrA");

    validate_jwt_signature(&jwt, pkey).expect("Validation failed unexpectedly");
}
