// Copyright (C) 2018  Aprila Bank ASA
//
// alcoholic_jwt is free software: you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation, either version 3 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.

//! Implements a library for for **validation** of **RS256** JWTs
//! using keys from a JWKS. Nothing more, nothing less.
//!
//! (Originally alcoholic_jwt. Added support for fetching and caching
//! the JWKS.)
//!
//! The name of the library stems from the potential side-effects of
//! trying to use the other Rust libraries that are made for similar
//! purposes.
//!
//! This library is specifically aimed at developers that consume
//! tokens from services which provide their RSA public keys in
//! [JWKS][] format.
//!
//! ## Usage example (JWKS served from some trusted URL)
//!
//! ```rust,no_run
//! use std::time::Duration;
//! use jwks_rsa_easy::{JWKSUrl, Validation};
//!
//! # async {
//! let jwks_url = JWKSUrl::new("https://example.com/jwks".into(), None, Duration::from_secs(300));
//!
//! let token = "received jwt token";
//!
//! // Several types of built-in validations are provided:
//! let validations = [
//!   Validation::NotExpired,
//!   Validation::Issuer("auth.test.aprila.no".into()),
//!   Validation::SubjectPresent,
//! ];
//!
//! jwks_url.validate(token, &validations).await.expect("token validation has failed!");
//! # };
//! ```
//!
//! [JWKS]: https://tools.ietf.org/html/rfc7517

use base64::{Config, DecodeError, URL_SAFE_NO_PAD};
use reqwest::Client;
use ring::signature::{RsaPublicKeyComponents, RSA_PKCS1_2048_8192_SHA256};
use serde::{de::DeserializeOwned, Deserialize};
use serde_json::Value;
use std::{
    fmt::Debug,
    time::{Duration, Instant, SystemTime, UNIX_EPOCH},
};
use tokio::sync::{RwLock, RwLockReadGuard};

#[cfg(test)]
mod tests;

/// URL-safe character set without padding that allows trailing bits,
/// which appear in some JWT implementations.
///
/// Note: The functions on `base64::Config` are not marked `const`,
/// and the constructors are not exported, which is why this is
/// implemented as a function.
fn jwt_forgiving() -> Config {
    URL_SAFE_NO_PAD.decode_allow_trailing_bits(true)
}

/// JWT algorithm used. The only supported algorithm is currently
/// RS256.
#[derive(Clone, Deserialize, Debug)]
enum KeyAlgorithm {
    RS256,
}

/// Type of key contained in a JWT. The only supported key type is
/// currently RSA.
#[derive(Clone, Deserialize, Debug)]
enum KeyType {
    RSA,
}

/// Representation of a single JSON Web Key. See [RFC
/// 7517](https://tools.ietf.org/html/rfc7517#section-4).
#[allow(dead_code)] // kty & alg only constrain deserialisation, but aren't used
#[derive(Clone, Debug, Deserialize)]
pub struct JWK {
    kty: KeyType,
    alg: Option<KeyAlgorithm>,
    kid: Option<String>,

    // Shared modulus
    n: String,

    // Public key exponent
    e: String,
}

impl JWK {
    /// Validate the signature of a JSON Web Token and optionally apply
    /// claim validations. Signatures are always verified before claims,
    /// and if a signature verification passes *all* claim validations are
    /// run and returned.
    ///
    /// If validation succeeds a representation of the token is returned
    /// that contains the header and claims as simple JSON values.
    ///
    /// It is the user's task to ensure that the correct JWK is passed in
    /// for validation.
    pub fn validate<'a>(
        &self,
        token: &str,
        validations: impl IntoIterator<Item = &'a Validation>,
    ) -> JWTResult<ValidJWT> {
        let jwt = JWT(token);
        let public_key = public_key_from_jwk(&self)?;
        validate_jwt_signature(&jwt, public_key)?;

        // Split out all three parts of the JWT this time, deserialising
        // the first and second as appropriate.
        let parts: Vec<&str> = jwt.0.splitn(3, '.').collect();
        if parts.len() != 3 {
            // This is unlikely considering that validation has already
            // been performed at this point, but better safe than sorry.
            return Err(ValidationError::InvalidComponents);
        }

        // Perform claim validations before constructing the valid token:
        let partial_claims = deserialize_part(parts[1])?;
        validate_claims(partial_claims, validations)?;

        let headers = deserialize_part(parts[0])?;
        let claims = deserialize_part(parts[1])?;
        let valid_jwt = ValidJWT { headers, claims };

        Ok(valid_jwt)
    }
}

/// Representation of a set of JSON Web Keys. See [RFC
/// 7517](https://tools.ietf.org/html/rfc7517#section-5).
#[derive(Clone, Debug, Deserialize)]
pub struct JWKS {
    // This is a vector instead of some kind of map-like structure
    // because key IDs are in fact optional.
    //
    // Technically having multiple keys with the same KID would not
    // violate the JWKS-definition either, but behaviour in that case
    // is unspecified.
    keys: Vec<JWK>,
}

impl JWKS {
    /// Attempt to find a JWK by its key ID.
    pub fn find(&self, kid: &str) -> Option<&JWK> {
        self.keys.iter().find(|jwk| jwk.kid.as_deref() == Some(kid))
    }
}

#[derive(Debug)]
struct JWKSCache {
    jwks: JWKS,
    valid_until: Instant,
}

/// JWKS served from an URL. Automatically fetch and cache the JWKS.
#[derive(Debug)]
pub struct JWKSUrl {
    url: String,
    client: Client,
    cache_duration: Duration,
    cache: RwLock<Option<JWKSCache>>,
}

impl JWKSUrl {
    pub fn new(url: String, client: Option<Client>, cache_duration: Duration) -> Self {
        Self {
            url,
            client: client.unwrap_or_default(),
            cache_duration,
            cache: RwLock::new(None),
        }
    }

    async fn get_jwks(&self) -> JWTResult<RwLockReadGuard<'_, JWKS>> {
        let cache = self.cache.read().await;
        if let Some(c) = &*cache {
            if c.valid_until
                .checked_duration_since(Instant::now())
                .is_some()
            {
                return Ok(RwLockReadGuard::map(cache, |c| &c.as_ref().unwrap().jwks));
            }
        }
        drop(cache);

        let mut cache = self.cache.write().await;
        if let Some(c) = &*cache {
            if c.valid_until
                .checked_duration_since(Instant::now())
                .is_some()
            {
                return Ok(RwLockReadGuard::map(cache.downgrade(), |c| {
                    &c.as_ref().unwrap().jwks
                }));
            }
        }
        let jwks = self
            .client
            .get(&self.url)
            .header("accept", "application/json")
            .send()
            .await?
            .json()
            .await?;

        *cache = Some(JWKSCache {
            jwks,
            valid_until: std::time::Instant::now() + self.cache_duration,
        });

        Ok(RwLockReadGuard::map(cache.downgrade(), |c| {
            &c.as_ref().unwrap().jwks
        }))
    }

    pub async fn validate<'a>(
        &self,
        token: &str,
        validations: impl IntoIterator<Item = &'a Validation>,
    ) -> JWTResult<ValidJWT> {
        let jwks = self.get_jwks().await?;
        if let Some(kid) = token_kid(token)? {
            if let Some(k) = jwks.find(&kid) {
                return k.validate(token, validations);
            }
        }
        Err(ValidationError::InvalidJWK)
    }
}

/// Representation of an undecoded JSON Web Token. See [RFC
/// 7519](https://tools.ietf.org/html/rfc7519).
struct JWT<'a>(&'a str);

/// Representation of a decoded and validated JSON Web Token.
///
/// Specific claim fields are only decoded internally in the library
/// for validation purposes, while it is generally up to the consumer
/// of the validated JWT what structure they would like to impose.
pub struct ValidJWT {
    /// JOSE header of the JSON Web Token. Certain fields are
    /// guaranteed to be present in this header, consult section 5 of
    /// RFC7519 for more information.
    pub headers: Value,

    /// Claims (i.e. primary data) contained in the JSON Web Token.
    /// While there are several registered and recommended headers
    /// (consult section 4.1 of RFC7519), the presence of no field is
    /// guaranteed in these.
    pub claims: Value,
}

/// Possible token claim validations. This enumeration only covers
/// common use-cases, for other types of validations the user is
/// encouraged to inspect the claim set manually.
pub enum Validation {
    /// Validate that the issuer ("iss") claim matches a specified
    /// value.
    Issuer(String),

    /// Validate that a subject value is present.
    SubjectPresent,

    /// Validate that the expiry time of the token ("exp"-claim) has
    /// not yet been reached.
    NotExpired,
}

/// Possible results of a token validation.
#[derive(Debug)]
pub enum ValidationError {
    /// Invalid number of token components (not a JWT?)
    InvalidComponents,

    /// Token segments had invalid base64-encoding.
    InvalidBase64(DecodeError),

    /// Decoding of the provided JWK failed.
    InvalidJWK,

    /// Signature validation failed, i.e. because of a non-matching
    /// public key.
    InvalidSignature,

    /// JSON decoding into a provided type failed.
    JSON(serde_json::Error),

    /// Fetching the JWKS failed.
    Reqwest(reqwest::Error),

    /// One or more claim validations failed. This variant contains
    /// human-readable validation errors.
    InvalidClaims(Vec<&'static str>),
}

type JWTResult<T> = Result<T, ValidationError>;

impl From<serde_json::Error> for ValidationError {
    fn from(err: serde_json::Error) -> Self {
        ValidationError::JSON(err)
    }
}

impl From<DecodeError> for ValidationError {
    fn from(err: DecodeError) -> Self {
        ValidationError::InvalidBase64(err)
    }
}

impl From<reqwest::Error> for ValidationError {
    fn from(err: reqwest::Error) -> Self {
        ValidationError::Reqwest(err)
    }
}

/// Attempt to extract the `kid`-claim out of a JWT's header claims.
///
/// This function is normally used when a token provider has multiple
/// public keys in rotation at the same time that could all still have
/// valid tokens issued under them.
///
/// This is only safe if the key set containing the currently allowed
/// key IDs is fetched from a trusted source.
fn token_kid(token: &str) -> JWTResult<Option<String>> {
    // Fetch the header component of the JWT by splitting it out and
    // dismissing the rest.
    let parts: Vec<&str> = token.splitn(2, '.').collect();
    if parts.len() != 2 {
        return Err(ValidationError::InvalidComponents);
    }

    // Decode only the first part of the token into a specialised
    // representation:
    #[derive(Deserialize)]
    struct KidOnly {
        kid: Option<String>,
    }

    let kid_only: KidOnly = deserialize_part(parts[0])?;

    Ok(kid_only.kid)
}

// Internal implementation
//
// The functions in the following section are not part of the public
// API of this library.

/// Decode a single key fragment (base64-url encoded integer) to an
/// OpenSSL BigNum.
fn decode_fragment(fragment: &str) -> JWTResult<Vec<u8>> {
    base64::decode_config(fragment, jwt_forgiving()).map_err(|_| ValidationError::InvalidJWK)
}

/// Decode an RSA public key from a JWK by constructing it directly
/// from the public RSA key fragments.
fn public_key_from_jwk(jwk: &JWK) -> JWTResult<RsaPublicKeyComponents<Vec<u8>>> {
    let n = decode_fragment(&jwk.n)?;
    let e = decode_fragment(&jwk.e)?;

    Ok(RsaPublicKeyComponents { n, e })
}

/// Decode a base64-URL encoded string and deserialise the resulting
/// JSON.
fn deserialize_part<T: DeserializeOwned>(part: &str) -> JWTResult<T> {
    let json = base64::decode_config(part, jwt_forgiving())?;
    serde_json::from_slice(&json).map_err(Into::into)
}

/// Validate the signature on a JWT using a provided public key.
///
/// A JWT is made up of three components (headers, claims, signature)
/// - only the first two are part of the signed data.
fn validate_jwt_signature(jwt: &JWT, key: RsaPublicKeyComponents<Vec<u8>>) -> JWTResult<()> {
    // Split the token from the back to a maximum of two elements.
    // There are technically three components using the same separator
    // ('.'), but we are interested in the first two together and
    // splitting them is unnecessary.
    let token_parts: Vec<&str> = jwt.0.rsplitn(2, '.').collect();
    if token_parts.len() != 2 {
        return Err(ValidationError::InvalidComponents);
    }

    // Second element of the vector will be the signed payload.
    let data = token_parts[1];

    // First element of the vector will be the (encoded) signature.
    let sig_b64 = token_parts[0];
    let sig = base64::decode_config(sig_b64, jwt_forgiving())?;

    key.verify(&RSA_PKCS1_2048_8192_SHA256, data.as_bytes(), &sig)
        .map_err(|_| ValidationError::InvalidSignature)
}

/// Internal helper struct for claims that are relevant for claim
/// validations.
#[derive(Deserialize)]
struct PartialClaims {
    iss: Option<String>,
    sub: Option<String>,
    exp: Option<u64>,
}

/// Apply a single validation to the claim set of a token.
fn apply_validation(claims: &PartialClaims, validation: &Validation) -> Result<(), &'static str> {
    match validation {
        // Validate that an 'iss' claim is present and matches the
        // supplied value.
        Validation::Issuer(ref iss) => match claims.iss {
            None => Err("'iss' claim is missing"),
            Some(ref claim) => {
                if claim == iss {
                    Ok(())
                } else {
                    Err("'iss' claim does not match")
                }
            }
        },

        Validation::SubjectPresent => match claims.sub {
            Some(_) => Ok(()),
            None => Err("'sub' claim is missing"),
        },

        Validation::NotExpired => match claims.exp {
            None => Err("'exp' claim is missing"),
            Some(exp) => {
                // Determine the current timestamp in seconds since
                // the UNIX epoch.
                let now = SystemTime::now()
                    .duration_since(UNIX_EPOCH)
                    // this is an unrecoverable, critical error. There
                    // aren't many ways this can occur, other than
                    // system time being set into the far future or
                    // this library being used in some sort of future
                    // museum.
                    .expect("system time is likely incorrect");

                // Convert the expiry time (which is also in epoch
                // seconds) to a duration.
                let exp_duration = Duration::from_secs(exp);

                // The token has not expired if the expiry duration is
                // larger than (i.e. in the future from) the current
                // time.
                if exp_duration > now {
                    Ok(())
                } else {
                    Err("token has expired")
                }
            }
        },
    }
}

/// Apply all requested validations to a partial claim set.
fn validate_claims<'a>(
    claims: PartialClaims,
    validations: impl IntoIterator<Item = &'a Validation>,
) -> JWTResult<()> {
    let validation_errors: Vec<_> = validations
        .into_iter()
        .map(|v| apply_validation(&claims, v))
        .filter_map(|result| match result {
            Ok(_) => None,
            Err(err) => Some(err),
        })
        .collect();

    if validation_errors.is_empty() {
        Ok(())
    } else {
        Err(ValidationError::InvalidClaims(validation_errors))
    }
}
