//     junitify - Takes cargo test JSON and transform to JUnit XML
//
//         The MIT License (MIT)
//
//      Copyright (c) KoresFramework (https://gitlab.com/Kores/)
//      Copyright (c) contributors
//
//      Permission is hereby granted, free of charge, to any person obtaining a copy
//      of this software and associated documentation files (the "Software"), to deal
//      in the Software without restriction, including without limitation the rights
//      to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//      copies of the Software, and to permit persons to whom the Software is
//      furnished to do so, subject to the following conditions:
//
//      The above copyright notice and this permission notice shall be included in
//      all copies or substantial portions of the Software.
//
//      THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//      IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//      FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//      AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//      LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//      OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
//      THE SOFTWARE.
mod test_parser;
mod test_producer;

use crate::test_parser::{ParsedTestSuite, TestParser};
use crate::test_producer::{RenderedSuite, TestProducer};
use clap::{AppSettings, Clap};
use handlebars::RenderError;
use serde_json::Error;
use std::fmt::Debug;
use std::fs::OpenOptions;
use std::io::{self, BufRead, BufReader, Write};
use std::path::{Path, PathBuf};

#[derive(Clap)]
#[clap(version = "1.0", author = "Jonathan H. R. Lopes <jhrldev@gmail.com>")]
#[clap(setting = AppSettings::ColoredHelp)]
struct Opts {
    #[clap(short, long, about = "Output directory for produced .xml files.")]
    out: Option<String>,
    #[clap(short, long, about = "Mirror stdin in stdout.")]
    mirror: bool,
    #[clap(short, long, about = "File to read from (default: stdin)")]
    file: Option<String>,
}

#[derive(Debug)]
enum AppError {
    Serde(Error),
    Template(RenderError),
    IO(std::io::Error),
}

fn main() -> Result<(), AppError> {
    let opts: Opts = Opts::parse();
    let render = if let Some(file) = opts.file {
        read_from_file(file, opts.mirror)?
    } else {
        render_from_stdin(opts.mirror)?
    };

    if let Some(ref out) = opts.out {
        write_to_file(&out, render)?;
    } else {
        write_to_stdout(render)?;
    }
    Ok(())
}

fn write_to_stdout(render: Vec<RenderedSuite>) -> Result<(), AppError> {
    for r in render {
        println!("<{}>", r.name);
        println!("{}", r.rendered);
        println!("</{}>", r.name);
    }

    Ok(())
}

fn write_to_file(out: &String, render: Vec<RenderedSuite>) -> Result<(), AppError> {
    let root = PathBuf::from(out);
    if !root.exists() {
        std::fs::create_dir(root.clone()).map_err(|e| AppError::IO(e))?;
    }

    for i in 0..render.len() {
        let rendered = render.get(i).unwrap();
        let path = root.join(format!("{}-{}.xml", rendered.name, i));

        let mut file = OpenOptions::new()
            .create(true)
            .write(true)
            .truncate(true)
            .open(path)
            .map_err(|e| AppError::IO(e))?;

        file.write_all(rendered.rendered.as_bytes())
            .map_err(|e| AppError::IO(e))?;
    }

    Ok(())
}

fn render_from_stdin(mirror: bool) -> Result<Vec<RenderedSuite>, AppError> {
    let mut buffer = String::new();
    let stdin = io::stdin();

    let mut suites: Vec<ParsedTestSuite> = vec![];
    let mut parser = TestParser::new();
    let mut line;
    loop {
        line = stdin.read_line(&mut buffer);
        let has_more = read_from_buffer(&mut parser, line, &mut buffer, &mut suites, mirror)?;
        if !has_more {
            break;
        }
    }

    TestProducer::produce(suites).map_err(|e| AppError::Template(e))
}

fn read_from_file<P>(path: P, mirror: bool) -> Result<Vec<RenderedSuite>, AppError>
where
    P: AsRef<Path>,
{
    let mut buffer = String::new();
    let open = OpenOptions::new()
        .read(true)
        .open(path)
        .map_err(|e| AppError::IO(e))?;

    let mut buf = BufReader::new(open);

    let mut suites: Vec<ParsedTestSuite> = vec![];
    let mut parser = TestParser::new();
    let mut line;
    loop {
        line = buf.read_line(&mut buffer);
        let has_more = read_from_buffer(&mut parser, line, &mut buffer, &mut suites, mirror)?;
        if !has_more {
            break;
        }
    }

    TestProducer::produce(suites).map_err(|e| AppError::Template(e))
}

fn read_from_buffer(
    parser: &mut TestParser,
    read: Result<usize, std::io::Error>,
    buffer: &mut String,
    suites: &mut Vec<ParsedTestSuite>,
    mirror: bool,
) -> Result<bool, AppError> {
    if let Ok(l) = read {
        if l > 0 && buffer.len() > 0 {
            if mirror {
                println!("{}", buffer)
            }

            let parse = parser.parse(&buffer);
            if let Ok(p) = parse {
                if let Some(suite) = p {
                    suites.push(suite);
                    parser.reset();
                }
                buffer.clear();
                return Ok(true);
            } else if let Err(e) = parse {
                return Err(AppError::Serde(e));
            }
        } else {
            return Ok(false);
        }
    } else {
        return Ok(false);
    }

    return Ok(false);
}

#[cfg(test)]
mod tests {
    use crate::read_from_file;
    use crate::test_parser::{parse_test, Test};

    #[test]
    fn suite() {
        let suite = r#"{ "type": "suite", "event": "started", "test_count": 14 }"#;
        let result = parse_test(suite).map_err(|e| e.to_string());
        assert_eq!(
            Ok(Test::new_suite_started("started".to_string(), 14)),
            result
        );
    }

    #[test]
    fn test() {
        let suite =
            r#"{ "type": "test", "event": "started", "name": "tests::bench_1mb_tar_detect" }"#;
        let result = parse_test(suite).map_err(|e| e.to_string());
        assert_eq!(
            Ok(Test::Test {
                event: "started".to_string(),
                name: "tests::bench_1mb_tar_detect".to_string(),
                stdout: None,
                exec_time: None
            }),
            result
        );
    }

    #[test]
    fn test_with_time() {
        let suite = r#"{ "type": "test", "name": "tests::test_rar_sfx_detect", "event": "ok", "exec_time": 0.07352109 }"#;
        let result = parse_test(suite).map_err(|e| e.to_string());
        assert_eq!(
            Ok(Test::Test {
                event: "ok".to_string(),
                name: "tests::test_rar_sfx_detect".to_string(),
                stdout: None,
                exec_time: Some(0.07352109f64)
            }),
            result
        );
    }

    #[test]
    fn suite_ok() {
        let suite = r#"{ "type": "suite", "event": "ok", "passed": 1, "failed": 0, "allowed_fail": 0, "ignored": 0, "measured": 0, "filtered_out": 0, "exec_time": 0.279672772 }"#;
        let result = parse_test(suite).map_err(|e| e.to_string());
        assert_eq!(
            Ok(Test::new_suite_ok(
                "ok".to_string(),
                1,
                0,
                0,
                0,
                0,
                0,
                0.279672772f64
            )),
            result
        );
    }

    #[test]
    fn test_read_case_1() {
        let result = read_from_file("test/test-case-1", false);
        assert_eq!(result.is_ok(), true);
    }

    #[test]
    fn test_read_case_2() {
        let result = read_from_file("test/test-case-2", false);
        assert_eq!(result.is_ok(), true);
    }
}
