//     junitify - Takes cargo test JSON and transform to JUnit XML
//
//         The MIT License (MIT)
//
//      Copyright (c) KoresFramework (https://gitlab.com/Kores/)
//      Copyright (c) contributors
//
//      Permission is hereby granted, free of charge, to any person obtaining a copy
//      of this software and associated documentation files (the "Software"), to deal
//      in the Software without restriction, including without limitation the rights
//      to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//      copies of the Software, and to permit persons to whom the Software is
//      furnished to do so, subject to the following conditions:
//
//      The above copyright notice and this permission notice shall be included in
//      all copies or substantial portions of the Software.
//
//      THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//      IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//      FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//      AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//      LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//      OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
//      THE SOFTWARE.
use crate::test_parser::{ParsedTestSuite, TestStatus};
use chrono::Utc;
use handlebars::{Handlebars, RenderError};
use serde_json::json;
use std::io::ErrorKind;

const TEMPLATE: &'static str = include_str!("test_xml.hbs");

pub(crate) struct TestProducer {}

#[derive(Debug)]
pub(crate) struct RenderedSuite {
    pub(crate) name: String,
    pub(crate) rendered: String,
}

impl TestProducer {
    pub(crate) fn produce(tests: Vec<ParsedTestSuite>) -> Result<Vec<RenderedSuite>, RenderError> {
        let mut rendered: Vec<RenderedSuite> = vec![];
        for suite in tests {
            let mapped_cases: Vec<_> = suite
                .tests
                .iter()
                .map(|c| {
                    if c.status == TestStatus::Ok {
                        json!({
                            "name": c.name,
                            "module": c.module,
                            "time": 0f32,
                            "failure": false
                        })
                    } else {
                        let stdout = c.std_out.clone();
                        json!({
                            "name": c.name,
                            "module": c.module,
                            "time": 0f32,
                            "failure": true,
                            "failure_message": "assertion failed",
                            "failure_std_out": stdout.unwrap_or("?".to_string())
                        })
                    }
                })
                .collect();

            let reg = Handlebars::new();
            let host = hostname::get();
            let host_string = host.and_then(|s| {
                s.into_string()
                    .map_err(|_| std::io::Error::from(ErrorKind::InvalidData))
            });

            let render = reg.render_template(
                TEMPLATE,
                &json!({
                    "suite_name": suite.suite_name,
                    "tests": suite.test_count,
                    "skipped": suite.ignored,
                    "failures": suite.failed,
                    "errors": 0,
                    "timestamp": timestamp(),
                    "time": suite.exec_time,
                    "hostname": host_string.unwrap_or("".to_string()),
                    "test_case": mapped_cases
                }),
            )?;

            rendered.push(RenderedSuite {
                name: suite.suite_name,
                rendered: render,
            })
        }

        return Ok(rendered);
    }
}

fn timestamp() -> String {
    let now = Utc::now();
    now.format("%Y-%m-%dT%H:%M:%S").to_string()
}
