use serde::Serialize;
use serde_json::{json, Map, Result, Value};
use std::vec;

pub fn from_str(s: &str) -> Result<Value> {
    from_value(&serde_json::from_str::<Value>(s)?)
}

pub fn from_value(value: &Value) -> Result<Value> {
    let mut path_segments: Vec<String> = Vec::new();
    let mut flattened: Map<String, Value> = Map::new();

    process_node(&value, &mut path_segments, &mut flattened);

    Ok(Value::Object(flattened))
}

pub fn from<T>(value: &T) -> Result<Value>
where
    T: Serialize,
{
    from_str(&serde_json::to_string(value)?)
}

fn process_node(value: &Value, path_segments: &mut Vec<String>, result: &mut Map<String, Value>) {
    match value {
        Value::Null => {
            result.insert(path_segments.concat(), Value::Null);
        }
        Value::Bool(x) => {
            result.insert(path_segments.concat(), Value::Bool(x.clone()));
        }
        Value::Number(x) => {
            result.insert(path_segments.concat(), Value::Number(x.clone()));
        }
        Value::String(x) => {
            result.insert(path_segments.concat(), Value::String(x.clone()));
        }
        Value::Array(x) => {
            result.insert(path_segments.concat(), json!([]));
            x.iter().enumerate().for_each(|(i, v)| {
                path_segments.push(format!("{}{}", "/", i));
                process_node(v, path_segments, result);
            });
        }
        Value::Object(x) => {
            result.insert(path_segments.concat(), json!({}));
            for (k, v) in x {
                path_segments.push(format!("{}{}", "/", escape(k.as_str())));
                process_node(v, path_segments, result);
            }
        }
    }
    path_segments.pop();
}

fn escape<'a>(value: &'a str) -> String {
    value.replace("~", "~0").replace("/", "~1")
}

#[cfg(test)]
mod test {
    use super::*;

    #[test]
    fn test_serialize() {
        let result = from_str(
            r#"
        {
            "one": 1
        }
        "#,
        );

        println!(
            "{}",
            serde_json::to_string_pretty(&result.unwrap()).unwrap()
        );
    }

    #[test]
    fn test_spec_values() {
        let value = json!(
            {
                "foo": ["bar", "baz"],
                "": 0,
                "a/b": 1,
                "c%d": 2,
                "e^f": 3,
                "g|h": 4,
                "i\\j": 5,
                "k\"l": 6,
                " ": 7,
                "m~n": 8
             }
        );

        let actual = from_value(&value).unwrap();

        assert!(matches!(actual, Value::Object(_)));

        assert!(actual.get("/m~0n").unwrap().eq(&json!(8)));
        assert!(actual.get("/a~1b").unwrap().eq(&json!(1)));
        assert!(actual.get("/ ").unwrap().eq(&json!(7)));
    }

    #[test]
    fn flatten_top_level_array() {
        let value = json!([true, 42]);

        let actual = from(&value).unwrap();

        assert!(actual.get("").unwrap().eq(&json!([])));
        assert!(actual.get("/0").unwrap().eq(&json!(true)));
        assert!(actual.get("/1").unwrap().eq(&json!(42)));
    }

    #[test]
    fn example_readme() {
        let value = json!(
            {
                "name": "John Smith",
                "age": 24,
                "address": {
                    "country": "US",
                    "zip": "00000"
                },
                "phones": [ "123", "456" ]
            }
        );

        let result = from_value(&value).unwrap();

        println!("{}", serde_json::to_string_pretty(&result).unwrap());
    }
}
