use std::collections::HashSet;
use std::fmt;

use serde_json::map::Entry;
use serde_json::{Number, Value};

use parser::*;

use self::expr_term::*;
use self::value_walker::JValueWalker;

mod cmp;
mod expr_term;
mod value_walker;

fn to_f64(n: &Number) -> f64 {
    if n.is_i64() {
        n.as_i64().unwrap() as f64
    } else if n.is_f64() {
        n.as_f64().unwrap()
    } else {
        n.as_u64().unwrap() as f64
    }
}

fn abs_index(n: isize, len: usize) -> usize {
    if n < 0_isize {
        (n + len as isize).max(0) as usize
    } else {
        n.min(len as isize) as usize
    }
}

#[derive(Debug, PartialEq)]
enum FilterKey {
    String(String),
    All,
}

pub enum JsonPathError {
    EmptyPath,
    EmptyValue,
    CantFlatten(Vec<Value>),
    Path(String),
    Serde(String),
}

impl fmt::Debug for JsonPathError {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        write!(f, "{}", self)
    }
}

impl fmt::Display for JsonPathError {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        match self {
            JsonPathError::EmptyPath => f.write_str("path not set"),
            JsonPathError::EmptyValue => f.write_str("json value not set"),
            JsonPathError::CantFlatten(values) => {
                write!(f, "json value '{:?}' can't be flattened", values)
            }
            JsonPathError::Path(msg) => f.write_str(&format!("path error: \n{}\n", msg)),
            JsonPathError::Serde(msg) => f.write_str(&format!("serde error: \n{}\n", msg)),
        }
    }
}

#[derive(Debug, Default)]
struct FilterTerms<'a>(Vec<Option<ExprTerm<'a>>>);

impl<'a> FilterTerms<'a> {
    fn new_filter_context(&mut self) {
        self.0.push(None);
        debug!("new_filter_context: {:?}", self.0);
    }

    fn is_term_empty(&self) -> bool {
        self.0.is_empty()
    }

    fn push_term(&mut self, term: Option<ExprTerm<'a>>) {
        self.0.push(term);
    }

    #[allow(clippy::option_option)]
    fn pop_term(&mut self) -> Option<Option<ExprTerm<'a>>> {
        self.0.pop()
    }

    fn filter_json_term<
        F: Fn(&Vec<&'a Value>, &mut Vec<&'a Value>, &mut HashSet<usize>) -> FilterKey,
    >(
        &mut self,
        e: ExprTerm<'a>,
        fun: F,
    ) {
        debug!("filter_json_term: {:?}", e);

        if let ExprTerm::Json(rel, fk, vec) = e {
            let mut tmp = Vec::new();
            let mut not_matched = HashSet::new();
            let filter_key = if let Some(FilterKey::String(key)) = fk {
                let key_contained = &vec
                    .iter()
                    .map(|v| match v {
                        Value::Object(map) if map.contains_key(&key) => map.get(&key).unwrap(),
                        _ => v,
                    })
                    .collect();
                fun(key_contained, &mut tmp, &mut not_matched)
            } else {
                fun(&vec, &mut tmp, &mut not_matched)
            };

            if rel.is_some() {
                self.0
                    .push(Some(ExprTerm::Json(rel, Some(filter_key), tmp)));
            } else {
                let filtered: Vec<&Value> = vec
                    .iter()
                    .enumerate()
                    .filter(|(idx, _)| !not_matched.contains(idx))
                    .map(|(_, v)| *v)
                    .collect();

                self.0
                    .push(Some(ExprTerm::Json(Some(filtered), Some(filter_key), tmp)));
            }
        } else {
            unreachable!("unexpected: ExprTerm: {:?}", e);
        }
    }

    fn push_json_term<
        F: Fn(&Vec<&'a Value>, &mut Vec<&'a Value>, &mut HashSet<usize>) -> FilterKey,
        FF: Fn(&Vec<&'a Value>, &mut Vec<&'a Value>, &mut HashSet<usize>) -> FilterKey,
    >(
        &mut self,
        current: &Option<Vec<&'a Value>>,
        values: Option<Box<dyn ExactSizeIterator<Item = &'a Value> + 'a>>,
        fun: F,
        fun_flattened: FF,
    ) {
        debug!("push_json_term: {:?}", &current);

        if let Some(current) = &current {
            let mut tmp = Vec::new();
            let mut not_matched = HashSet::new();
            let filter_key = fun(current, &mut tmp, &mut not_matched);
            self.0
                .push(Some(ExprTerm::Json(None, Some(filter_key), tmp)));
        } else {
            if let Some(values) = values {
                let mut tmp = Vec::new();
                let mut not_matched = HashSet::new();
                let values = values.collect();
                let filter_key = fun_flattened(&values, &mut tmp, &mut not_matched);
                self.0
                    .push(Some(ExprTerm::Json(None, Some(filter_key), tmp)));
            }
        }
    }

    fn filter<
        F: Fn(&Vec<&'a Value>, &mut Vec<&'a Value>, &mut HashSet<usize>) -> FilterKey,
        FF: Fn(&Vec<&'a Value>, &mut Vec<&'a Value>, &mut HashSet<usize>) -> FilterKey,
    >(
        &mut self,
        current: &Option<Vec<&'a Value>>,
        values: Option<Box<dyn ExactSizeIterator<Item = &'a Value> + 'a>>,
        fun: F,
        fun_flattened: FF,
    ) {
        if let Some(peek) = self.0.pop() {
            if let Some(e) = peek {
                self.filter_json_term(e, fun);
            } else {
                self.push_json_term(current, values, fun, fun_flattened);
            }
        }
    }

    fn filter_all_with_str(
        &mut self,
        current: &Option<Vec<&'a Value>>,
        values: Option<Box<dyn ExactSizeIterator<Item = &'a Value> + 'a>>,
        key: &str,
    ) {
        self.filter(
            current,
            values,
            |vec, tmp, _| {
                JValueWalker::all_with_str(&vec, tmp, key, true);
                FilterKey::All
            },
            |_, _, _| FilterKey::All,
        );

        debug!("filter_all_with_str : {}, {:?}", key, self.0);
    }

    fn filter_next_with_str(
        &mut self,
        current: &Option<Vec<&'a Value>>,
        values: Option<Box<dyn ExactSizeIterator<Item = &'a Value> + 'a>>,
        key: &str,
    ) {
        self.filter(
            current,
            values,
            |vec, tmp, not_matched| {
                let mut visited = HashSet::new();
                for (idx, v) in vec.iter().enumerate() {
                    match v {
                        Value::Object(map) => {
                            if map.contains_key(key) {
                                let ptr = *v as *const Value;
                                if !visited.contains(&ptr) {
                                    visited.insert(ptr);
                                    tmp.push(v)
                                }
                            } else {
                                not_matched.insert(idx);
                            }
                        }
                        Value::Array(vec) => {
                            not_matched.insert(idx);
                            for v in vec {
                                JValueWalker::walk_dedup(v, tmp, key, &mut visited);
                            }
                        }
                        _ => {
                            not_matched.insert(idx);
                        }
                    }
                }

                FilterKey::String(key.to_owned())
            },
            |vec, tmp, not_matched| {
                let mut visited = HashSet::new();
                for (idx, v) in vec.iter().enumerate() {
                    not_matched.insert(idx);
                    JValueWalker::walk_dedup(v, tmp, key, &mut visited);
                }

                FilterKey::String(key.to_owned())
            },
        );

        debug!("filter_next_with_str : {}, {:?}", key, self.0);
    }

    fn collect_next_with_num(
        &mut self,
        current: &Option<Vec<&'a Value>>,
        values: Option<Box<dyn ExactSizeIterator<Item = &'a Value> + 'a>>,
        index: f64,
    ) -> (Option<Vec<&'a Value>>, Option<Vec<usize>>) {
        fn _collect<'a>(tmp: &mut Vec<&'a Value>, vec: &'a [Value], index: f64) {
            let index = abs_index(index as isize, vec.len());
            if let Some(v) = vec.get(index) {
                tmp.push(v);
            }
        }

        if let Some(current) = current {
            let mut tmp = Vec::new();
            for c in current {
                match c {
                    Value::Object(map) => {
                        for k in map.keys() {
                            if let Some(Value::Array(vec)) = map.get(k) {
                                _collect(&mut tmp, vec, index);
                            }
                        }
                    }
                    Value::Array(vec) => {
                        _collect(&mut tmp, vec, index);
                    }
                    _ => {}
                }
            }

            if tmp.is_empty() {
                self.0.pop();
                return (Some(vec![]), None);
            } else {
                return (Some(tmp), None);
            }
        }

        if let Some(mut values) = values {
            let index = abs_index(index as isize, values.len());

            match values.nth(index) {
                Some(value) => return (Some(vec![value]), Some(vec![index])),
                None => {
                    self.0.pop();
                    return (Some(vec![]), None);
                }
            }
        }

        debug!("collect_next_with_num : {:?}, {:?}", &index, &current);

        (None, None)
    }

    fn collect_next_all(
        &mut self,
        current: &Option<Vec<&'a Value>>,
        values: Option<Box<dyn ExactSizeIterator<Item = &'a Value> + 'a>>,
    ) -> (Option<Vec<&'a Value>>, Option<Vec<usize>>) {
        if let Some(current) = current {
            let mut tmp = Vec::new();
            for c in current {
                match c {
                    Value::Object(map) => {
                        for (_, v) in map {
                            tmp.push(v)
                        }
                    }
                    Value::Array(vec) => {
                        for v in vec {
                            tmp.push(v);
                        }
                    }
                    _ => {}
                }
            }
            return (Some(tmp), None);
        }

        if let Some(current) = values {
            let values = current.collect::<Vec<_>>();

            let values_len = values.len();
            return (Some(values), Some((0..values_len).collect()));
        }

        debug!("collect_next_all : {:?}", &current);

        (None, None)
    }

    fn collect_next_with_str(
        &mut self,
        current: &Option<Vec<&'a Value>>,
        values: Option<Box<dyn ExactSizeIterator<Item = &'a Value> + 'a>>,
        keys: &[String],
    ) -> (Option<Vec<&'a Value>>, Option<Vec<usize>>) {
        if let Some(current) = current {
            let mut tmp = Vec::new();
            for c in current {
                if let Value::Object(map) = c {
                    for key in keys {
                        if let Some(v) = map.get(key) {
                            tmp.push(v)
                        }
                    }
                }
            }

            if tmp.is_empty() {
                self.0.pop();
                return (Some(vec![]), None);
            } else {
                return (Some(tmp), None);
            }
        }

        if values.is_some() {
            // values has array-like structure
            self.0.pop();
            return (Some(vec![]), None);
        }

        debug!("collect_next_with_str : {:?}, {:?}", keys, &current);

        (None, None)
    }

    fn collect_all(
        &mut self,
        current: &Option<Vec<&'a Value>>,
        values: Option<Box<dyn ExactSizeIterator<Item = &'a Value> + 'a>>,
    ) -> (Option<Vec<&'a Value>>, Option<Vec<usize>>) {
        if let Some(current) = current {
            let mut tmp = Vec::new();
            JValueWalker::all(&current, &mut tmp);
            return (Some(tmp), None);
        }

        if let Some(values) = values {
            let values = values.collect::<Vec<_>>();
            let values_len = values.len();
            return (Some(values), Some((0..values_len).collect()));
        }
        debug!("collect_all: {:?}", &current);

        (None, None)
    }

    fn collect_all_with_str(
        &mut self,
        current: &Option<Vec<&'a Value>>,
        values: Option<Box<dyn ExactSizeIterator<Item = &'a Value> + 'a>>,
        key: &str,
    ) -> (Option<Vec<&'a Value>>, Option<Vec<usize>>) {
        if let Some(current) = current {
            let mut tmp = Vec::new();
            JValueWalker::all_with_str(&current, &mut tmp, key, false);
            return (Some(tmp), None);
        }

        if values.is_some() {
            return (Some(vec![]), None);
        }

        debug!("collect_all_with_str: {}, {:?}", key, &current);

        (None, None)
    }

    fn collect_all_with_num(
        &mut self,
        current: &Option<Vec<&'a Value>>,
        values: Option<Box<dyn ExactSizeIterator<Item = &'a Value> + 'a>>,
        index: f64,
    ) -> (Option<Vec<&'a Value>>, Option<Vec<usize>>) {
        if let Some(current) = current {
            let mut tmp = Vec::new();
            JValueWalker::all_with_num(&current, &mut tmp, index);
            return (Some(tmp), None);
        }

        if let Some(mut values) = values {
            match values.nth(index as usize) {
                Some(value) => return (Some(vec![value]), Some(vec![index as usize])),
                None => return (Some(vec![]), None),
            }
        }

        debug!("collect_all_with_num: {}, {:?}", index, &current);

        (None, None)
    }
}

#[derive(Default)]
pub struct Selector<'a, 'b> {
    node: Option<Node>,
    node_ref: Option<&'b Node>,
    value: Option<&'a Value>,
    tokens: Vec<ParseToken>,
    current: Option<Vec<&'a Value>>,
    values: Option<Box<dyn ExactSizeIterator<Item = &'a Value> + 'a>>,
    chose_indices: Vec<usize>,
    // true if values haven't been found by key or index in JValue
    not_found_by_key_index: bool,
    selectors: Vec<Selector<'a, 'b>>,
    selector_filter: FilterTerms<'a>,
}

impl<'a, 'b> Selector<'a, 'b> {
    pub fn new() -> Self {
        Self::default()
    }

    pub fn str_path(&mut self, path: &str) -> Result<&mut Self, JsonPathError> {
        debug!("path : {}", path);

        self.node_ref.take();
        self.node = Some(Parser::compile(path).map_err(JsonPathError::Path)?);
        Ok(self)
    }

    pub fn node_ref(&self) -> Option<&Node> {
        if let Some(node) = &self.node {
            return Some(node);
        }

        if let Some(node) = &self.node_ref {
            return Some(*node);
        }

        None
    }

    pub fn compiled_path(&mut self, node: &'b Node) -> &mut Self {
        self.node.take();
        self.node_ref = Some(node);
        self
    }

    pub fn reset_value(&mut self) -> &mut Self {
        self.current = None;
        self
    }

    pub fn value(&mut self, v: &'a Value) -> &mut Self {
        self.value = Some(v);
        self
    }

    pub fn values_iter(
        &mut self,
        values: impl Iterator<Item = &'a Value> + ExactSizeIterator + 'a,
    ) -> &mut Self {
        self.values = Some(Box::new(values));
        self
    }

    fn _select(&mut self) -> Result<(), JsonPathError> {
        if self.node_ref.is_some() {
            let node_ref = self.node_ref.take().unwrap();
            self.visit(node_ref);

            return if self.not_found_by_key_index {
                Err(JsonPathError::EmptyValue)
            } else {
                Ok(())
            };
        }

        if self.node.is_none() {
            return Err(JsonPathError::EmptyPath);
        }

        let node = self.node.take().unwrap();
        self.visit(&node);
        self.node = Some(node);

        if self.not_found_by_key_index {
            Err(JsonPathError::EmptyValue)
        } else {
            Ok(())
        }
    }

    pub fn select_as<T: serde::de::DeserializeOwned>(&mut self) -> Result<Vec<T>, JsonPathError> {
        self._select()?;

        match &self.current {
            Some(vec) => {
                let mut ret = Vec::new();
                for v in vec {
                    match T::deserialize(*v) {
                        Ok(v) => ret.push(v),
                        Err(e) => return Err(JsonPathError::Serde(e.to_string())),
                    }
                }
                Ok(ret)
            }
            _ => Err(JsonPathError::EmptyValue),
        }
    }

    pub fn select_as_str(&mut self) -> Result<String, JsonPathError> {
        self._select()?;

        match &self.current {
            Some(r) => {
                Ok(serde_json::to_string(r).map_err(|e| JsonPathError::Serde(e.to_string()))?)
            }
            _ => Err(JsonPathError::EmptyValue),
        }
    }

    pub fn select(&mut self) -> Result<Vec<&'a Value>, JsonPathError> {
        self._select()?;

        match &self.current {
            Some(r) => Ok(r.to_vec()),
            _ => Err(JsonPathError::EmptyValue),
        }
    }

    pub fn select_(&mut self) -> Result<Vec<&'a Value>, JsonPathError> {
        self._select()?;

        match &self.current {
            Some(r) => Ok(r.to_vec()),
            _ => Err(JsonPathError::EmptyValue),
        }
    }

    fn compute_absolute_path_filter(&mut self, token: &ParseToken) -> bool {
        if !self.selectors.is_empty() {
            match token {
                ParseToken::Absolute | ParseToken::Relative | ParseToken::Filter(_) => {
                    let selector = self.selectors.pop().unwrap();

                    if let Some(current) = &selector.current {
                        let term = current.into();

                        if let Some(s) = self.selectors.last_mut() {
                            s.selector_filter.push_term(Some(term));
                        } else {
                            self.selector_filter.push_term(Some(term));
                        }
                    } else {
                        unreachable!()
                    }
                }
                _ => {}
            }
        }

        if let Some(selector) = self.selectors.last_mut() {
            selector.visit_token(token);
            true
        } else {
            false
        }
    }
}

impl<'a, 'b> Selector<'a, 'b> {
    fn visit_absolute(&mut self) {
        if self.current.is_some() {
            let mut selector = Selector::default();

            if let Some(value) = self.value {
                selector.value = Some(value);
                selector.current = Some(vec![value]);

                println!("current.is_some: {:?}", selector.current);
                self.selectors.push(selector);
            }
            return;
        }

        if self.values.is_some() {
            return;
        }

        if let Some(v) = &self.value {
            self.current = Some(vec![v]);
        }
    }

    fn visit_relative(&mut self) {
        if let Some(ParseToken::Array) = self.tokens.last() {
            let array_token = self.tokens.pop();
            if let Some(ParseToken::Leaves) = self.tokens.last() {
                let values = std::mem::replace(&mut self.values, None);
                self.tokens.pop();
                let (current, chose_indices) =
                    self.selector_filter.collect_all(&self.current, values);

                self.current = current;
                self.update_not_found_by_current();

                self.chose_indices.extend(chose_indices.unwrap_or_default());
            }
            self.tokens.push(array_token.unwrap());
        }
        self.selector_filter.new_filter_context();
    }

    fn visit_array_eof(&mut self) {
        if self.is_last_before_token_match(ParseToken::Array) {
            if let Some(Some(e)) = self.selector_filter.pop_term() {
                let values = std::mem::replace(&mut self.values, None);
                if let ExprTerm::String(key) = e {
                    self.selector_filter
                        .filter_next_with_str(&self.current, values, &key);
                    self.tokens.pop();
                    return;
                }

                self.selector_filter.push_term(Some(e));
            }
        }

        if self.is_last_before_token_match(ParseToken::Leaves) {
            self.tokens.pop();
            self.tokens.pop();
            if let Some(Some(e)) = self.selector_filter.pop_term() {
                let selector_filter_consumed =
                    match &e {
                        ExprTerm::Number(n) => {
                            let values = std::mem::replace(&mut self.values, None);
                            let (current, chose_indices) = self
                                .selector_filter
                                .collect_all_with_num(&self.current, values, to_f64(n));

                            self.selector_filter.pop_term();

                            self.current = current;
                            self.update_not_found_by_current();

                            self.chose_indices.extend(chose_indices.unwrap_or_default());
                            true
                        }
                        ExprTerm::String(key) => {
                            let values = std::mem::replace(&mut self.values, None);
                            let (current, chose_indices) = self
                                .selector_filter
                                .collect_all_with_str(&self.current, values, key);

                            self.selector_filter.pop_term();
                            self.current = current;
                            self.update_not_found_by_current();

                            self.chose_indices.extend(chose_indices.unwrap_or_default());
                            true
                        }
                        _ => {
                            self.selector_filter.push_term(Some(e));
                            false
                        }
                    };

                if selector_filter_consumed {
                    return;
                }
            }
        }

        if let Some(Some(e)) = self.selector_filter.pop_term() {
            match e {
                ExprTerm::Number(n) => {
                    let values = std::mem::replace(&mut self.values, None);
                    let (current, chose_indices) = self.selector_filter.collect_next_with_num(
                        &self.current,
                        values,
                        to_f64(&n),
                    );

                    self.current = current;
                    self.update_not_found_by_current();

                    self.chose_indices.extend(chose_indices.unwrap_or_default());
                }
                ExprTerm::String(key) => {
                    let values = std::mem::replace(&mut self.values, None);

                    let (current, chose_indices) =
                        self.selector_filter
                            .collect_next_with_str(&self.current, values, &[key]);

                    self.current = current;
                    self.update_not_found_by_current();

                    self.chose_indices.extend(chose_indices.unwrap_or_default());
                }
                ExprTerm::Json(rel, _, v) => {
                    if v.is_empty() {
                        self.current = Some(vec![]);
                    } else if let Some(vec) = rel {
                        self.current = Some(vec);
                    } else {
                        self.current = Some(v);
                    }

                    self.update_not_found_by_current();
                }
                ExprTerm::Bool(false) => {
                    self.current = Some(vec![]);
                }
                _ => {}
            }
        }

        self.tokens.pop();
    }

    fn is_last_before_token_match(&mut self, token: ParseToken) -> bool {
        if self.tokens.len() > 1 {
            return token == self.tokens[self.tokens.len() - 2];
        }

        false
    }

    fn visit_all(&mut self) {
        if let Some(ParseToken::Array) = self.tokens.last() {
            self.tokens.pop();
        }

        let values = std::mem::replace(&mut self.values, None);
        match self.tokens.last() {
            Some(ParseToken::Leaves) => {
                self.tokens.pop();
                let (current, chose_indices) =
                    self.selector_filter.collect_all(&self.current, values);

                self.current = current;
                self.chose_indices.extend(chose_indices.unwrap_or_default());
            }
            Some(ParseToken::In) => {
                self.tokens.pop();
                let (current, chose_indices) =
                    self.selector_filter.collect_next_all(&self.current, values);

                self.current = current;
                self.chose_indices.extend(chose_indices.unwrap_or_default());
            }
            _ => {
                let (current, chose_indices) =
                    self.selector_filter.collect_next_all(&self.current, values);

                self.current = current;
                self.chose_indices.extend(chose_indices.unwrap_or_default());
            }
        }
    }

    fn visit_key(&mut self, key: &str) {
        if let Some(ParseToken::Array) = self.tokens.last() {
            self.selector_filter
                .push_term(Some(ExprTerm::String(key.to_string())));
            return;
        }

        if let Some(t) = self.tokens.pop() {
            if self.selector_filter.is_term_empty() {
                match t {
                    ParseToken::Leaves => {
                        let values = std::mem::replace(&mut self.values, None);
                        let (current, chose_indices) =
                            self.selector_filter
                                .collect_all_with_str(&self.current, values, key);

                        self.current = current;
                        self.chose_indices.extend(chose_indices.unwrap_or_default());
                    }
                    ParseToken::In => {
                        let values = std::mem::replace(&mut self.values, None);
                        let (current, chose_indices) = self.selector_filter.collect_next_with_str(
                            &self.current,
                            values,
                            &[key.to_string()],
                        );

                        self.current = current;
                        self.update_not_found_by_current();

                        self.chose_indices.extend(chose_indices.unwrap_or_default());
                    }
                    _ => {}
                }
            } else {
                match t {
                    ParseToken::Leaves => {
                        let values = std::mem::replace(&mut self.values, None);
                        self.selector_filter
                            .filter_all_with_str(&self.current, values, key);
                    }
                    ParseToken::In => {
                        let values = std::mem::replace(&mut self.values, None);
                        self.selector_filter
                            .filter_next_with_str(&self.current, values, key);
                    }
                    _ => {}
                }
            }
        }
    }

    fn visit_keys(&mut self, keys: &[String]) {
        if !self.selector_filter.is_term_empty() {
            unimplemented!("keys in filter");
        }

        if let Some(ParseToken::Array) = self.tokens.pop() {
            let values = std::mem::replace(&mut self.values, None);
            let (current, chose_indices) =
                self.selector_filter
                    .collect_next_with_str(&self.current, values, keys);
            self.current = current;
            self.update_not_found_by_current();

            self.chose_indices.extend(chose_indices.unwrap_or_default());
        } else {
            unreachable!();
        }
    }

    fn visit_filter(&mut self, ft: &FilterToken) {
        let right = match self.selector_filter.pop_term() {
            Some(Some(right)) => right,
            Some(None) => ExprTerm::Json(
                None,
                None,
                match &self.current {
                    Some(current) => current.to_vec(),
                    _ => unreachable!(),
                },
            ),
            _ => panic!("empty term right"),
        };

        let left = match self.selector_filter.pop_term() {
            Some(Some(left)) => left,
            Some(None) => ExprTerm::Json(
                None,
                None,
                match &self.current {
                    Some(current) => current.to_vec(),
                    _ => unreachable!(),
                },
            ),
            _ => panic!("empty term left"),
        };

        let mut ret = None;
        match ft {
            FilterToken::Equal => left.eq(&right, &mut ret),
            FilterToken::NotEqual => left.ne(&right, &mut ret),
            FilterToken::Greater => left.gt(&right, &mut ret),
            FilterToken::GreaterOrEqual => left.ge(&right, &mut ret),
            FilterToken::Little => left.lt(&right, &mut ret),
            FilterToken::LittleOrEqual => left.le(&right, &mut ret),
            FilterToken::And => left.and(&right, &mut ret),
            FilterToken::Or => left.or(&right, &mut ret),
        };

        if let Some(e) = ret {
            self.selector_filter.push_term(Some(e));
        }
    }

    fn visit_range(&mut self, from: &Option<isize>, to: &Option<isize>, step: &Option<usize>) {
        if !self.selector_filter.is_term_empty() {
            unimplemented!("range syntax in filter");
        }

        if let Some(ParseToken::Array) = self.tokens.pop() {
            let mut tmp = Vec::new();
            if let Some(current) = &self.current {
                for v in current {
                    if let Value::Array(vec) = v {
                        let from = match from {
                            Some(from) => abs_index(*from, vec.len()),
                            None => 0,
                        };

                        let to = match to {
                            Some(to) => abs_index(*to, vec.len()),
                            None => vec.len(),
                        };

                        let step = match step {
                            Some(step) => *step,
                            None => 1,
                        };

                        for i in (from..to).step_by(step) {
                            if let Some(v) = vec.get(i) {
                                tmp.push(v);
                            }
                        }
                    }
                }
            } else {
                let values = std::mem::replace(&mut self.values, None);
                if let Some(values) = values {
                    let from = match from {
                        Some(from) => abs_index(*from, values.len()),
                        None => 0,
                    };

                    let to = match to {
                        Some(to) => abs_index(*to, values.len()),
                        None => values.len(),
                    };

                    let step = match step {
                        Some(step) => *step,
                        None => 1,
                    };

                    let take_count = if from > to { 0 } else { to - from };

                    let matched_values = values.skip(from).step_by(step).take(take_count);
                    tmp.extend(matched_values);

                    self.chose_indices
                        .extend((from..to).step_by(step).collect::<Vec<_>>())
                }
            }
            self.current = Some(tmp);
        } else {
            unreachable!();
        }
    }

    fn visit_union(&mut self, indices: &[isize]) {
        if !self.selector_filter.is_term_empty() {
            unimplemented!("union syntax in filter");
        }

        if let Some(ParseToken::Array) = self.tokens.pop() {
            let mut tmp = Vec::new();
            if let Some(current) = &self.current {
                for v in current {
                    if let Value::Array(vec) = v {
                        for i in indices {
                            if let Some(v) = vec.get(abs_index(*i, vec.len())) {
                                tmp.push(v);
                            }
                        }
                    }
                }
            } else {
                let indices = indices.iter().map(|&v| v as usize).collect::<HashSet<_>>();
                let values = std::mem::replace(&mut self.values, None);
                if let Some(values) = values {
                    let new_values = values
                        .enumerate()
                        .filter(|(id, _)| indices.contains(id))
                        .map(|(id, value)| {
                            self.chose_indices.push(id);
                            value
                        })
                        .collect::<Vec<_>>();

                    tmp.extend(new_values);
                }
            }

            self.current = Some(tmp);
        } else {
            unreachable!();
        }
    }

    fn update_not_found_by_current(&mut self) {
        println!("current: {:?}", self.current);

        if let Some(values) = &self.current {
            self.not_found_by_key_index |= values.is_empty();
        }
    }

    pub fn chose_indices(self) -> Vec<usize> {
        self.chose_indices
    }
}

impl<'a, 'b> NodeVisitor for Selector<'a, 'b> {
    fn visit_token(&mut self, token: &ParseToken) {
        debug!("token: {:?}, stack: {:?}", token, self.tokens);

        if self.compute_absolute_path_filter(token) {
            return;
        }

        match token {
            ParseToken::Absolute => self.visit_absolute(),
            ParseToken::Relative => self.visit_relative(),
            ParseToken::In | ParseToken::Leaves | ParseToken::Array => {
                self.tokens.push(token.clone());
            }
            ParseToken::ArrayEof => self.visit_array_eof(),
            ParseToken::All => self.visit_all(),
            ParseToken::Bool(b) => {
                self.selector_filter.push_term(Some(ExprTerm::Bool(*b)));
            }
            ParseToken::Key(key) => self.visit_key(key),
            ParseToken::Keys(keys) => self.visit_keys(keys),
            ParseToken::Number(v) => {
                self.selector_filter
                    .push_term(Some(ExprTerm::Number(Number::from_f64(*v).unwrap())));
            }
            ParseToken::Filter(ref ft) => self.visit_filter(ft),
            ParseToken::Range(from, to, step) => self.visit_range(from, to, step),
            ParseToken::Union(indices) => self.visit_union(indices),
            ParseToken::Eof => {
                debug!("visit_token eof");
            }
        }
    }
}

#[derive(Default)]
pub struct SelectorMut {
    path: Option<Node>,
    value: Option<Value>,
}

fn replace_value<F: FnMut(Value) -> Option<Value>>(
    mut tokens: Vec<String>,
    value: &mut Value,
    fun: &mut F,
) {
    let mut target = value;

    let last_index = tokens.len().saturating_sub(1);
    for (i, token) in tokens.drain(..).enumerate() {
        let target_once = target;
        let is_last = i == last_index;
        let target_opt = match *target_once {
            Value::Object(ref mut map) => {
                if is_last {
                    if let Entry::Occupied(mut e) = map.entry(token) {
                        let v = e.insert(Value::Null);
                        if let Some(res) = fun(v) {
                            e.insert(res);
                        } else {
                            e.remove();
                        }
                    }
                    return;
                }
                map.get_mut(&token)
            }
            Value::Array(ref mut vec) => {
                if let Ok(x) = token.parse::<usize>() {
                    if is_last {
                        let v = std::mem::replace(&mut vec[x], Value::Null);
                        if let Some(res) = fun(v) {
                            vec[x] = res;
                        } else {
                            vec.remove(x);
                        }
                        return;
                    }
                    vec.get_mut(x)
                } else {
                    None
                }
            }
            _ => None,
        };

        if let Some(t) = target_opt {
            target = t;
        } else {
            break;
        }
    }
}

impl SelectorMut {
    pub fn new() -> Self {
        Self::default()
    }

    pub fn str_path(&mut self, path: &str) -> Result<&mut Self, JsonPathError> {
        self.path = Some(Parser::compile(path).map_err(JsonPathError::Path)?);
        Ok(self)
    }

    pub fn value(&mut self, value: Value) -> &mut Self {
        self.value = Some(value);
        self
    }

    pub fn take(&mut self) -> Option<Value> {
        self.value.take()
    }

    fn compute_paths(&self, mut result: Vec<&Value>) -> Vec<Vec<String>> {
        fn _walk(
            origin: &Value,
            target: &mut Vec<&Value>,
            tokens: &mut Vec<String>,
            visited: &mut HashSet<*const Value>,
            visited_order: &mut Vec<Vec<String>>,
        ) -> bool {
            trace!("{:?}, {:?}", target, tokens);

            if target.is_empty() {
                return true;
            }

            target.retain(|t| {
                if std::ptr::eq(origin, *t) {
                    if visited.insert(*t) {
                        visited_order.push(tokens.to_vec());
                    }
                    false
                } else {
                    true
                }
            });

            match origin {
                Value::Array(vec) => {
                    for (i, v) in vec.iter().enumerate() {
                        tokens.push(i.to_string());
                        if _walk(v, target, tokens, visited, visited_order) {
                            return true;
                        }
                        tokens.pop();
                    }
                }
                Value::Object(map) => {
                    for (k, v) in map {
                        tokens.push(k.clone());
                        if _walk(v, target, tokens, visited, visited_order) {
                            return true;
                        }
                        tokens.pop();
                    }
                }
                _ => {}
            }

            false
        }

        let mut visited = HashSet::new();
        let mut visited_order = Vec::new();

        if let Some(origin) = &self.value {
            let mut tokens = Vec::new();
            _walk(
                origin,
                &mut result,
                &mut tokens,
                &mut visited,
                &mut visited_order,
            );
        }

        visited_order
    }

    pub fn delete(&mut self) -> Result<&mut Self, JsonPathError> {
        self.replace_with(&mut |_| Some(Value::Null))
    }

    pub fn remove(&mut self) -> Result<&mut Self, JsonPathError> {
        self.replace_with(&mut |_| None)
    }

    fn select(&self) -> Result<Vec<&Value>, JsonPathError> {
        if let Some(node) = &self.path {
            let mut selector = Selector::default();
            selector.compiled_path(&node);

            if let Some(value) = &self.value {
                selector.value(value);
            }

            Ok(selector.select()?)
        } else {
            Err(JsonPathError::EmptyPath)
        }
    }

    pub fn replace_with<F: FnMut(Value) -> Option<Value>>(
        &mut self,
        fun: &mut F,
    ) -> Result<&mut Self, JsonPathError> {
        let paths = {
            let result = self.select()?;
            self.compute_paths(result)
        };

        if let Some(ref mut value) = &mut self.value {
            for tokens in paths {
                replace_value(tokens, value, fun);
            }
        }

        Ok(self)
    }
}

#[cfg(test)]
mod select_inner_tests {
    use serde_json::Value;

    #[test]
    fn to_f64_i64() {
        let number = 0_i64;
        let v: Value = serde_json::from_str(&format!("{}", number)).unwrap();
        if let Value::Number(n) = v {
            assert_eq!((super::to_f64(&n) - number as f64).abs() == 0_f64, true);
        } else {
            panic!();
        }
    }

    #[test]
    fn to_f64_f64() {
        let number = 0.1_f64;
        let v: Value = serde_json::from_str(&format!("{}", number)).unwrap();
        if let Value::Number(n) = v {
            assert_eq!((super::to_f64(&n) - number).abs() == 0_f64, true);
        } else {
            panic!();
        }
    }

    #[test]
    fn to_f64_u64() {
        let number = u64::max_value();
        let v: Value = serde_json::from_str(&format!("{}", number)).unwrap();
        if let Value::Number(n) = v {
            assert_eq!((super::to_f64(&n) - number as f64).abs() == 0_f64, true);
        } else {
            panic!();
        }
    }
}
