extern crate core;

use clap::{Parser, Subcommand};
use colored::{ColoredString, Colorize};
use serde::{Deserialize, Serialize};
use std::io::BufRead;
use std::io::BufReader;
use std::process::{ChildStdout, Command, Stdio};

#[derive(Serialize, Deserialize)]
struct LogObject {
    timestamp: String,
    #[serde(rename(deserialize = "logger.thread_name"))]
    thread_name: String,
    #[serde(rename(deserialize = "logger.name"))]
    name: String,
    level: String,
    message: String,
    #[serde(rename(deserialize = "error.kind"))]
    error_kind: Option<String>,
    #[serde(rename(deserialize = "error.message"))]
    error_message: Option<String>,
    #[serde(rename(deserialize = "error.stack"))]
    error_stack: Option<String>,
}

#[derive(Parser)]
#[clap(name = "json-log-parse")]
#[clap(author, version, about, long_about = None)]
struct Cli {
    #[clap(subcommand)]
    command: Commands,
}

#[derive(Subcommand)]
enum Commands {
    #[clap(about = "Leverages \"stern\" to parse kubernetes resource logs in a standalone fashion")]
    Stern {
        #[clap(short = 'a', long = "app", required = true, help = "Application name")]
        app: String,
        #[clap(short = 's', long = "since", required = false, default_value = "5m")]
        since: String,
        #[clap(
            short = 'i',
            long = "include",
            help = "Include output matching a string/pattern"
        )]
        include_filter: Option<String>,
        #[clap(
            short = 'e',
            long = "exclude",
            help = "Exclude output matching a string/pattern"
        )]
        exclude_filter: Option<String>,
    },
    #[clap(about = "Reads and parses json lines from stdin")]
    Stdin,
}

trait LogColumns {
    fn timestamp_column(&self) -> ColoredString;
    fn thread_name_column(&self) -> ColoredString;
    fn name_column(&self) -> ColoredString;
    fn level_column(&self) -> ColoredString;
    fn message_column(&self) -> ColoredString;
}

const TIMESTAMP_COLUMN_SIZE: usize = 24;
const THREAD_COLUMN_SIZE: usize = 32;
const NAME_COLUMN_SIZE: usize = 48;
const INFO_COLUMN_SIZE: usize = 5;

impl LogColumns for LogObject {
    fn timestamp_column(&self) -> ColoredString {
        format!("{:<width$}", self.timestamp, width = TIMESTAMP_COLUMN_SIZE).white()
    }

    fn thread_name_column(&self) -> ColoredString {
        let thread_slice_start = self
            .thread_name
            .len()
            .checked_sub(THREAD_COLUMN_SIZE)
            .unwrap_or(0);
        let thread_slice_end = self.thread_name.len();
        let thread = &self.thread_name[thread_slice_start..thread_slice_end];

        format!("{:<width$}", thread, width = THREAD_COLUMN_SIZE).white()
    }

    fn name_column(&self) -> ColoredString {
        let name_slice_start = self.name.len().checked_sub(NAME_COLUMN_SIZE).unwrap_or(0);
        let name_slice_end = self.name.len();
        let name = &self.name[name_slice_start..name_slice_end];

        format!("{:<width$}", name, width = NAME_COLUMN_SIZE).white()
    }

    fn level_column(&self) -> ColoredString {
        let level_color = match self.level.as_str() {
            "DEBUG" => colored::Color::Green,
            "INFO" => colored::Color::BrightWhite,
            "WARN" => colored::Color::BrightYellow,
            "ERROR" => colored::Color::BrightRed,
            _ => colored::Color::White,
        };
        format!("{:<width$}", self.level, width = INFO_COLUMN_SIZE).color(level_color)
    }

    fn message_column(&self) -> colored::ColoredString {
        self.message.clone().white()
    }
}

trait ErrorDetails {
    fn error_details(&self) -> Option<ColoredString>;
}

impl ErrorDetails for LogObject {
    fn error_details(&self) -> Option<ColoredString> {
        if let Some(stack) = &self.error_stack {
            Some(
                format!("\n{}\n\n{}", "STACK TRACE:", stack)
                    .bold()
                    .bright_red(),
            )
        } else {
            None
        }
    }
}

fn exec_stern(since: &str, app: &str) -> ChildStdout {
    let mut cmd = Command::new("stern");
    cmd.arg("--all-namespaces")
        .arg("-o")
        .arg("raw")
        .arg("-l")
        .arg(format!("app={}", app))
        .arg("-c")
        .arg(app)
        .arg("-s")
        .arg(since)
        .stdout(Stdio::piped())
        .spawn()
        .unwrap()
        .stdout
        .unwrap()
}

fn parse_and_output(raw_line: &str) {
    match serde_json::from_str::<LogObject>(raw_line) {
        Err(_) => println!("raw - {}", raw_line),
        Ok(object) => {
            println!(
                "{} | {} | {} | {} | {}",
                object.timestamp_column(),
                object.thread_name_column(),
                object.name_column(),
                object.level_column(),
                object.message_column(),
            );
            if let Some(error) = object.error_details() {
                println!("{}", error);
            }
        }
    }
}

fn do_logging(app: &String, since: &String, include_filter: &Option<String>, exclude_filter: &Option<String>) {
    let reader = BufReader::new(exec_stern(&since, &app));
    reader
        .lines()
        .filter_map(|line| line.ok())
        .filter(|line| match &include_filter {
            Some(filter) => line.contains(filter),
            None => true,
        })
        .filter(|line| match &exclude_filter {
            Some(exclude) => !line.contains(exclude),
            None => true,
        })
        .for_each(|line| parse_and_output(&line));
}

fn main() {
    let args = Cli::parse();

    match args.command {
        Commands::Stern { app, since, include_filter, exclude_filter } => {
            do_logging(&app, &since, &include_filter, &exclude_filter);
        },
        Commands::Stdin => {
            let stdin = std::io::stdin();

            for line in stdin.lock().lines() {
                parse_and_output(&line.expect("fatal error while reading stdin"));
            }
        },
    }
}
