pub mod parser_error;
pub mod node_type;
pub mod node;

use parser_error::ParserError;
use node_type::NodeType;
use node::Node;

///
/// ## Convert a jq filter to AST
/// Example:
///
/// ```
/// use jq_lang::to_ast;
/// use jq_lang::node_type::NodeType;
///
/// let ast = to_ast(".").unwrap();
/// assert_eq!(ast.node_type, NodeType::Program);
///```
///
/// Providing an empty string will result in an error:
/// ```
/// use jq_lang::to_ast;
/// let parser_error = to_ast("");
/// assert_eq!(parser_error.is_err(), true);
/// ```
pub fn to_ast(filter: &str) -> Result<Node, ParserError> {
    if filter.len() == 0 {
        return Err(ParserError::new("Cannot parse empty string. Expected filter"))
    }

    let mut children = vec!();
    children.push(Node {
        node_type: NodeType::Identity,
        children: None
    });

    Ok(Node::new(NodeType::Program, Some(children)))
}

#[cfg(test)]
mod tests {
    use crate::to_ast;
    use crate::NodeType;

    #[test]
    fn identity_filter_produces_root_node() {
        let ast = to_ast(".").unwrap();
        assert_eq!(ast.node_type, NodeType::Program);
    }

    #[test]
    fn identity_filter_produces_identity_node() {
        let ast = to_ast(".").unwrap();
        let children = ast.children.unwrap();
        let child_node = children.get(0).unwrap();
        assert_eq!(child_node.node_type, NodeType::Identity);
    }

    #[test]
    fn handles_empty_string_produces_error() {
        let ast = to_ast("");
        assert_eq!(ast.is_err(), true);
    }

    #[test]
    fn handles_empty_string_error_has_error_message() {
        let ast = to_ast("");
        assert_eq!(ast.err().unwrap().to_string(), "Cannot parse empty string. Expected filter");
    }
}
