use std::collections::HashMap;
use std::sync::{Arc, Mutex};

use serde::{Deserialize, Serialize};

/// Tasks can cycle through these states as they see fit to.
///
/// Only the following states are gaurenteed to happen:
/// - Inserted
/// - Allocated
/// - Consumed
/// - Exited, where Exited(true) indicates a successful run and Exited(false) indicates a failed run
pub enum State {
    /// The task has been inserted into the system, and it is pending allocation
    Inserted,
    /// The task has been allocated to a executor, but has not been Scheduled by the executor
    Allocated,
    /// The task has been Scheduled by the executor, and is presumed to be running
    Scheduled,

    /// The task ran a preflight routine and it has passed
    PreflightComplete,

    /// The task is explicitly set to running
    Running,

    /// The task is indicating progress, in the form of X/Y which can be converted into a percentage
    Progress(u16, u16),

    ///
    CleaningUp,

    /// AllocationFailure indicates that the executor could not allocate the requested resources,
    /// which could happen if resources were consumed by something else in the time it took the
    /// executor to load the task. These tasks will be rescheduled.
    AllocationFailure,

    /// The task has been lost. The governor has either lost contact with the executor, or the executor
    /// does not have any record of the running task
    Lost,

    Exited(i8),
}

#[derive(Serialize, Deserialize, Clone, Debug)]
///
/// TODO Add abiloty to reference resources against each other. If cpu_physical and cpu_cores are defined, cpu_physical should be able to invalidate cpu_cores
pub struct ResourceManager {
    lake: Arc<Mutex<HashMap<String, usize>>>,
    /// -> amount, block size, overcommit
    lake_cfg: HashMap<String, (usize, usize, f32)>,
}

impl ResourceManager {
    pub fn new() -> anyhow::Result<Self> {
        Ok(ResourceManager {
            lake: Default::default(),
            lake_cfg: Default::default(),
        })
    }

    ///
    ///
    /// rm.declare("cpu_cores", 8, 1, 2.5)
    ///
    pub fn declare(
        &mut self,
        name: &str,
        amount: usize,
        block_size: usize,
        overcommit: f32,
    ) -> anyhow::Result<()> {
        let amt = amount as f32 * overcommit;
        if amt % block_size as f32 != 0.0 {
            anyhow::bail!(
                "{} - resource amount ({}) * overcommit ({}) % block_size ({}) != 0.0",
                &name,
                &amount,
                &overcommit,
                &block_size
            );
        }
        self.lake
            .lock()
            .unwrap()
            .insert(name.to_string(), (amt * overcommit) as usize);
        self.lake_cfg
            .insert(name.to_string(), (amount, block_size, overcommit));
        Ok(())
    }

    ///
    ///
    /// rm.consume("cpu_cores", 3)
    ///
    pub fn consume(&mut self, name: &str, amount: usize) -> anyhow::Result<ResourceChunk> {
        if !self.lake.lock().unwrap().contains_key(name) {
            anyhow::bail!("resource manager does not know about resource `{}`", &name);
        }

        let mut inner = self.lake.lock().unwrap();
        let key = inner.get_mut(name).unwrap();
        if amount < *key {
            anyhow::bail!("");
        }

        *key -= amount;

        Ok(ResourceChunk {
            amount,
            key: name.to_string(),
            lake: self.lake.clone(),
        })
    }

    pub fn contains(&self, name: &str, amount: usize) -> anyhow::Result<bool> {
        if !self.lake.lock().unwrap().contains_key(name) {
            anyhow::bail!("resource manager does not know about resource `{}`", &name);
        }

        let current = self.lake.lock().unwrap();
        Ok(amount >= *current.get(name).unwrap())
    }

    /// Returns the amount of the requested resources
    pub fn get(&self, name: &str) -> anyhow::Result<usize> {
        if !self.lake.lock().unwrap().contains_key(name) {
            anyhow::bail!("resource manager does not know about resource `{}`", &name);
        }

        let current = self.lake.lock().unwrap();
        Ok(*current.get(name).unwrap())
    }

    /// Dump the lake
    pub fn dump(&self) -> anyhow::Result<HashMap<String, usize>> {
        Ok(self.lake.lock().unwrap().clone())
    }
}

impl Default for ResourceManager {
    fn default() -> Self {
        ResourceManager {
            lake: Arc::new(Mutex::new(Default::default())),
            lake_cfg: Default::default(),
        }
    }
}

#[derive(Clone, Debug)]
pub struct ResourceChunk {
    amount: usize,
    key: String,
    lake: Arc<Mutex<HashMap<String, usize>>>,
}

impl Drop for ResourceChunk {
    fn drop(&mut self) {
        *self.lake.lock().unwrap().get_mut(&self.key).unwrap() += self.amount
    }
}
