use prost_build;
use prost_build::Config;
use walkdir::WalkDir;

use crate::proto::ServiceTraitGenerator;
use crate::HandlerResult;
use std::ffi::OsStr;
use std::path::PathBuf;
use std::str::FromStr;

/// Build all protobuf files in `search` directory. `include` needs to contain the search directory.
/// If `search` is __src/proto__, then `include` will need to be __src__
pub fn build_directory(search: &str, include: &str) {
    let mut files = vec![];
    for entry in WalkDir::new(search) {
        let path = entry.unwrap().path().to_path_buf();
        let ext = path.extension().and_then(OsStr::to_str);
        if ext.is_some() && ext.unwrap() == "proto" {
            files.push(path);
        }
    }

    Config::new()
        .protoc_arg("--experimental_allow_proto3_optional")
        .service_generator(Box::new(ServiceTraitGenerator))
        .type_attribute(".", "#[derive(Serialize, Deserialize)]")
        .compile_protos(&*files, &[PathBuf::from_str(include).unwrap()])
        .unwrap();
}

pub fn handle_result<T: prost::Message + serde::Serialize>(
    message: crate::Result<T>,
) -> HandlerResult {
    fn err(error: anyhow::Error) -> HandlerResult {
        HandlerResult::Failure(serde_json::Value::String(format!("{:?}", error)))
    }
    match message {
        Ok(message) => match serde_json::to_value(message) {
            Ok(value) => HandlerResult::Success(value),
            Err(error) => err(error.into()),
        },
        Err(error) => err(error),
    }
}
