pub mod utils;

use crate::{HandlerResult, TaskContext, TaskStatus};
use prost_build;
use prost_build::{Service, ServiceGenerator};
use serde_json;
use std::collections::HashMap;
use zkstate::ZkState;

pub struct ServiceTraitGenerator;
impl ServiceGenerator for ServiceTraitGenerator {
    fn generate(&mut self, service: Service, buf: &mut String) {
        // Generate a trait for the service.
        service.comments.append_with_indent(0, buf);

        buf.push_str(&format!(
            "\n#[allow(non_snake_case)]\npub mod {0} {{\n",
            &service.name
        ));
        buf.push_str("    use anyhow;\n");
        buf.push_str("    use std::collections::HashMap;\n");
        buf.push_str("    use jotty::ZkState;\n");
        buf.push_str("    use jotty::{HandlerResult, TaskContext};\n");
        buf.push_str("    use jotty::proto::{GeneratedMethodTrait, utils};\n\n");
        buf.push_str("    pub trait Stub: Send + Sync + 'static {\n");
        for method in &service.methods {
            buf.push_str(&format!(
                "        fn {}(&self, ctx: &mut TaskContext, arguments: super::{}, shared: ZkState<HashMap<String, serde_json::Value>>) -> anyhow::Result<super::{}>;\n",
                method.name, method.input_type, method.output_type
            ));
        }
        buf.push_str("    }\n\n");
        buf.push_str("    pub struct Handler<T: Stub> { inner: std::sync::Arc<T> }\n");
        buf.push_str("    impl<T: Stub> Handler<T> {\n");
        buf.push_str(&format!("        /// Create a new Handler instance for {}\n", &service.name));
        buf.push_str("        pub fn new(inner: T) -> Box<Self> { Box::new(Self { inner: std::sync::Arc::new(inner) }) }\n");
        buf.push_str("    }\n");
        buf.push_str("    impl<T: Stub> GeneratedMethodTrait for Handler<T> {\n");
        buf.push_str("        fn handle(&self, ctx: &mut TaskContext, shared: ZkState<HashMap<String, serde_json::Value>>) -> HandlerResult {\n");
        buf.push_str("            match ctx.method.as_str() {\n");
        for method in &service.methods {
            // TODO when try_v2 becomes stable, replace this
            buf.push_str(&format!(
                "                \"{0}.{1}.{2}\" => {{\n",
                &service.package, &service.proto_name, &method.proto_name
            ));
            buf.push_str(
                "                    match serde_json::from_value(ctx.inputs.clone()) {\n",
            );
            buf.push_str(&format!("                        Ok(inner) => utils::handle_result(self.inner.{0}(ctx, inner, shared)),\n", &method.name));
            buf.push_str("                        Err(error) => HandlerResult::Failure(serde_json::Value::String(format!(\"unable to parse arguments for method `{}`. {:?}\", &ctx.method, error )))\n");
            buf.push_str("                    }\n");
            buf.push_str("                }\n");
        }
        buf.push_str("                _ => HandlerResult::Failure(serde_json::Value::String(format!(\"method `{}` not found\", &ctx.method )))\n");
        buf.push_str("            }\n");
        buf.push_str("        }\n");
        buf.push_str("        /// returns a list of methods that this handler can handle\n");
        buf.push_str("        fn contains(&self, method: &str) -> bool {\n");
        buf.push_str("            vec![");
        for method in &service.methods {
            buf.push_str(&format!(
                "\"{0}.{1}.{2}\", ",
                &service.package, &service.proto_name, &method.proto_name
            ));
        }
        buf.push_str("].contains(&method)\n");
        buf.push_str("        }\n");
        buf.push_str("    }\n\n");

        for method in &service.methods {
            buf.push_str(&format!("    pub mod {0} {{\n", method.proto_name));
            buf.push_str(&format!("        pub fn new(args: super::super::{0}) -> Result<jotty::TaskDef, serde_json::Error> {{ jotty::TaskDef::new(\"{1}.{2}.{3}\".to_string(), args) }}\n", method.input_type, &service.package, &service.proto_name, &method.proto_name));
            buf.push_str(&format!("        pub fn load(client: jotty::JoTTyClient, task_id: String) -> jotty::proto::JoTTyResult<super::super::{0}> {{ client.load_response(task_id) }}\n", method.output_type));
            buf.push_str("    }\n");
        }

        buf.push_str("}\n");
    }
    fn finalize_package(&mut self, _package: &str, buf: &mut String) {
        let mut prefix = "//\n\
        // Dynamically Generated on Build. DO NOT EDIT\n\
        // Generated with jotty 0.2.3\n\
        //\n"
            .to_string();
        prefix.push_str("use serde::{Serialize, Deserialize};\n\n");
        prefix.push_str(buf.as_str());

        // TODO make this less janky
        // inject the serde import into enum mods
        *buf = find_replace_janky(
            prefix,
            "pub enum",
            '{',
            "\n    use serde::{Serialize, Deserialize};",
        )
    }
}

/// Empty trait that is extended by the dynamically generated code.
pub trait GeneratedMethodTrait {
    fn handle(
        &self,
        ctx: &mut TaskContext,
        shared: ZkState<HashMap<String, serde_json::Value>>,
    ) -> HandlerResult;
    fn contains(&self, _: &str) -> bool;
}

#[allow(dead_code)]
pub struct JoTTyResult<T: ::prost::Message> {
    pub(crate) id: String,
    pub(crate) inner: Option<T>,
    pub(crate) status: TaskStatus,
}
impl<T: ::prost::Message> JoTTyResult<T> {}

fn find_replace_janky(body: String, find: &str, walk_back: char, inject: &str) -> String {
    let mut body = body;
    let mut offset = 0;
    loop {
        let search_index: Vec<_> = body.match_indices(find).collect();
        // we're done, nothing left to do
        if search_index.len() == offset {
            break;
        }

        let mut index = search_index[offset].0;
        let mut bytes_array = body.clone().as_bytes().to_vec();
        let mut found = false;
        loop {
            if bytes_array[index] as char == walk_back {
                found = true;
                break;
            }
            if index == 0 {
                break;
            }
            index -= 1;
        }
        offset += 1;
        if !found {
            break;
        }

        let inject_chars = inject.as_bytes();

        // move the cursor to the right of the find char
        index += 1;

        for ichar in inject_chars {
            bytes_array.insert(index, *ichar);
            index += 1;
        }

        body = String::from_utf8(bytes_array).unwrap();
    }

    body
}

#[cfg(test)]
mod tests {
    use crate::proto::find_replace_janky;

    #[test]
    fn test_one_occurance() {
        let test_string = "// junk data\n\
        //more junkj\n\
        // junkie\n\
        pub mod thingie {\n\
            // inner data\n\
            // more inner\n\
            pub enum thingie {\n\
                ABC,\n\
                DEF\n\
            }\n\
        }"
        .to_string();

        let expected = "// junk data\n\
        //more junkj\n\
        // junkie\n\
        pub mod thingie {\n\
            foobar\n\
            // inner data\n\
            // more inner\n\
            pub enum thingie {\n\
                ABC,\n\
                DEF\n\
            }\n\
        }"
        .to_string();

        assert_eq!(
            find_replace_janky(test_string.clone(), "pub enum", '{', "\nfoobar"),
            expected
        );
    }

    #[test]
    fn test_two() {
        let test_string = "// junk data\n\
        //more junkj\n\
        // junkie\n\
        pub mod thingie {\n\
            // inner data\n\
            // more inner\n\
            pub enum thingie {\n\
                ABC,\n\
                DEF\n\
            }\n\
        }\n\
        // more junk data\n\
        pub mod thingie {\n\
            // inner data\n\
            // more inner\n\
            pub enum thingie {\n\
                ABC,\n\
                DEF\n\
            }\n\
        }"
        .to_string();

        let expected = "// junk data\n\
        //more junkj\n\
        // junkie\n\
        pub mod thingie {\n\
            foobar\n\
            // inner data\n\
            // more inner\n\
            pub enum thingie {\n\
                ABC,\n\
                DEF\n\
            }\n\
        }\n\
        // more junk data\n\
        pub mod thingie {\n\
            foobar\n\
            // inner data\n\
            // more inner\n\
            pub enum thingie {\n\
                ABC,\n\
                DEF\n\
            }\n\
        }"
        .to_string();

        assert_eq!(
            find_replace_janky(test_string.clone(), "pub enum", '{', "\nfoobar"),
            expected
        );
    }
}
