use std::{collections::HashMap, sync::atomic::AtomicBool, sync::Arc, thread::JoinHandle};

use serde::{Deserialize, Serialize};
use zkstate::{Change, ZkState};
use zookeeper::{ZooKeeper, ZooKeeperExt};

use crate::proto::GeneratedMethodTrait;
use crate::rcm::ResourceManager;
use crate::{ContextWrapper, HandlerResult, Job, RetryReason, JoTTyClient, Task, TaskContext, TaskStatus};
use chrono::{DateTime, Utc};
use std::time::Duration;

#[derive(Serialize, Deserialize, Clone)]
pub struct ExecutorState {
    /// ID
    pub id: String,
    /// Total number of Channels that the executor has
    pub channels: usize,
    /// Map of running Tasks
    pub tasks: HashMap<String, String>,
    /// Resource Manager
    pub resources: ResourceManager,
    /// Last time the executor checked in
    pub heartbeat: DateTime<Utc>
}

pub(crate) type Callback = Box<
    dyn Fn(
            &String,
            TaskContext,
            &serde_json::Value,
            ZkState<HashMap<String, serde_json::Value>>,
        ) -> HandlerResult
        + Sync
        + Send,
>;
pub(crate) type Handler = Box<dyn GeneratedMethodTrait + Sync + Send + 'static>;

pub fn mk_callback<F>(f: F) -> Callback
where
    F: Fn(
            &String,
            TaskContext,
            &serde_json::Value,
            ZkState<HashMap<String, serde_json::Value>>,
        ) -> HandlerResult
        + Sync
        + Send
        + 'static,
{
    Box::new(f) as Callback
}

// TODO make sure channels are released on thread exit
#[allow(dead_code)]
pub struct JoTTyExecutor {
    zk: Arc<ZooKeeper>,

    /// Is the executor functionality enabled for the jotty instance
    _enabled: bool,
    /// Number of channels in the executor. Total number of tasks that can run at once
    channels: usize,

    handlers: Vec<Arc<Handler>>,
    /// Map of all running tasks
    handles: HashMap<String, (JoinHandle<()>, Arc<AtomicBool>)>,
    /// Executor State
    state: ZkState<ExecutorState>,

    id: String,
    namespace: String,
    client: Option<JoTTyClient>,
}
impl JoTTyExecutor {
    pub fn new(
        zk: Arc<ZooKeeper>,
        namespace: String,
        rcm: ResourceManager,
        channels: usize,
        id: Option<String>,
    ) -> Self {
        let id = id.unwrap_or_else(|| uuid::Uuid::new_v4().to_string());
        let mut path = format!("/jotty/{}/executors", namespace);

        zk.ensure_path(path.as_str()).unwrap();

        path.push('/');
        path.push_str(id.as_str());

        Self {
            _enabled: true,
            channels,
            handlers: vec![],
            handles: HashMap::new(),
            state: zkstate::ZkState::new(
                zk.clone(),
                path,
                ExecutorState {
                    id: id.clone(),
                    tasks: HashMap::new(),
                    resources: rcm,
                    channels,
                    heartbeat: Utc::now()
                },
            ).unwrap(),
            id,
            namespace,
            client: None,
            zk
        }
    }

    pub fn attach_client(mut self, client: JoTTyClient) -> Self {
        self.client = Some(client);
        self
    }

    pub fn attach_handler(&mut self, handler: Handler) {
        self.handlers.push(Arc::new(handler));
    }

    /// Start the Executor
    pub fn run(&mut self) -> anyhow::Result<()> {
        // create a thread that will send heartbeats
        let stateref = self.state.clone();
        std::thread::spawn(|| {
            let state = stateref;
            loop {
                std::thread::sleep(Duration::from_secs(10));
                if let Err(inner) = state.update(|inner| {
                    log::trace!("updating state with current time");
                   inner.heartbeat = Utc::now();
                }) {
                    log::error!("failed to write heartbeat. {:?}", inner);
                }
            }
        });

        let update_chan = self.state.get_update_channel();
        loop {
            let msg = update_chan.recv().unwrap();
            log::trace!("got state change message: {:?}", &msg);
            match msg {
                Change::Removed(_, _) => {}
                Change::Added(k, v) => {
                    if k[0] == zkstate::Key::String("tasks".to_string()) {
                        let mut task = Task::load(
                            self.zk.clone(),
                            self.namespace.clone(),
                            v.as_str().unwrap().to_string(),
                        )?;

                        match self.launch_task(&mut task) {
                            Ok(inner) => task.set_status(inner),
                            // TODO should be a better "unhandled error" state
                            Err(_error) => task.set_status(TaskStatus::Lost(RetryReason::Lost)),
                        }
                        task.save(self.zk.clone(), self.namespace.clone())?;
                    }
                }
                Change::Unchanged() => {}
                Change::Modified(_, _, _) => {}
            }
        }
    }

    /// Handles an "Add" event in our state.
    /// TODO this should be broken down into smaller chunks
    fn launch_task(&mut self, task: &mut Task) -> anyhow::Result<TaskStatus> {
        let execution_start = chrono::Utc::now();

        // make sure we have enough channels to run this task
        if self.handles.len() >= self.channels {
            log::error!(target: format!("jotty::executor::launch_task::{}", &task.id).as_str(), "number of running threads is greater than number of total channels");
            return Ok(TaskStatus::Lost(RetryReason::DimensionsExhausted(
                "channels".to_string(),
            )));
        }

        // now, allocate resources
        let mut resources = HashMap::new();

        if !task.resources.is_empty() {
            let mut resource_exhaustion = vec![];
            let _ = self.state.update(|inner| {
                for resource in &task.resources {
                    if !inner
                        .resources
                        .contains(resource.0.as_str(), *resource.1)
                        .unwrap()
                    {
                        // if we do not have enough resources for this allocation, log it and record it
                        let errmsg = format!(
                            "resource '{}' exhausted. request '{}', only have '{}'",
                            resource.0.clone(),
                            resource.1.clone(),
                            inner.resources.get(resource.0.as_str()).unwrap()
                        );
                        log::warn!("{}", &errmsg);
                        task.logs.info(format!("error - {}", errmsg));
                        resource_exhaustion.push((
                            resource.0.clone(),
                            *resource.1,
                            inner.resources.get(resource.0.as_str()).unwrap(),
                        ))
                    } else {
                        // we can allocate it, so generate a ResourceChunk object
                        // resources will be returned to the lake when dropped
                        resources.insert(
                            resource.0.clone(),
                            inner
                                .resources
                                .consume(resource.0.as_str(), *resource.1)
                                .unwrap(),
                        );
                    }
                }
            });
            if !resource_exhaustion.is_empty() {
                // we were unable to place this task, we should mark the task as failed
                task.status = TaskStatus::Failure;

                task.save(self.zk.clone(), self.namespace.clone())?;
            }
        }

        // TODO move this logic into it's own function
        let run_handle = Arc::new(AtomicBool::new(true));

        let ctxw = ContextWrapper::new(
            self.zk.clone(),
            self.namespace.clone(),
            TaskContext {
                id: task.id.clone(),
                method: task.method.clone(),
                inputs: task.args.clone(),
                running: run_handle.clone(),
                resources,
                next_steps: vec![],
            },
            self.id.clone(),
            task.clone(),
        );

        let mut handler = None;
        // loop through all registered handlers
        for proto_handler in &self.handlers {
            if proto_handler.contains(&task.method) {
                handler = Some(proto_handler);
                break;
            }
        }

        if handler.is_none() {
            log::error!(target: format!("jotty::executor::launch_task::{}", &task.id).as_str(), "method '{}' does not exist", &task.method);
            // TODO mark job as Lost, enqueue re-schedule
            anyhow::bail!("method doesn't exist");
        }
        log::info!(target: format!("jotty::executor::launch_task::{}", &task.id).as_str(), "starting '{}' with arguments {:?}", &task.method, &task.args);

        let outer_closure = handler.unwrap().clone();
        let handle = std::thread::spawn(move || {
            let mut ctxw = ctxw;

            let mut job = Job::load(
                ctxw.zk.clone(),
                ctxw.namespace.clone(),
                ctxw.task.parent_job.clone(),
            )
            .unwrap();

            // set the task status to Running
            ctxw.task.set_executor(&ctxw.executor);
            ctxw.task.set_status(TaskStatus::Run);
            if let Err(e) = ctxw.task.save(ctxw.zk.clone(), ctxw.namespace.clone()) {
                log::error!(
                    target: format!("jotty::executor::launch_task::{}", &ctxw.task.id).as_str(),
                    "unable to update task state to set status to Running. {:?}",
                    e
                );
                return;
            }

            log::debug!(target: format!("jotty::executor::launch_task::{}", &ctxw.task.id).as_str(), "---- stepping into handler method ----");

            ctxw.exit = outer_closure
                .clone()
                .handle(&mut ctxw.ctx, job.shared_data.clone().unwrap());

            log::debug!(target: format!("jotty::executor::launch_task::{}", &ctxw.task.id).as_str(), "----   handler method finished.   ----");

            if let HandlerResult::Failure(inner) = &ctxw.exit {
                log::error!(target: format!("jotty::executor::launch_task::{}", &ctxw.task.id).as_str(), "handler returned: HandlerResult::Failure('{:?}')", inner);
            }

            job.timings.insert(
                ctxw.task.id.clone(),
                (chrono::Utc::now() - execution_start)
                    .num_microseconds()
                    .unwrap() as usize,
            );

            // if we have additional jobs injected into next_step, we need to modify the current job
            // to add the jobs.
            if !ctxw.ctx.next_steps.is_empty() {
                log::debug!(target: format!("jotty::executor::launch_task::{}", &ctxw.task.id).as_str(), "TaskContext.next_step has {} entries, injecting", &ctxw.ctx.next_steps.len());
                job.injected_tasks = ctxw.ctx.next_steps.clone();
            }

            // TODO make this more efficient. Figure out a way to check if the contents of
            //     `shared_data` has been modified and only save if it has
            let save_job = job.save(ctxw.zk.clone(), ctxw.namespace.clone());
            if save_job.is_err() {
                log::error!(target: format!("jotty::executor::launch_task::{}", &ctxw.task.id).as_str(), "unable to save job definition. {:?}", save_job.err().unwrap())
            }
            let _ = ctxw.task.save(ctxw.zk.clone(), ctxw.namespace.clone());

            // TODO enqueue a re-eval task for this task so it can be processed quicker
            log::info!(target: format!("jotty::executor::launch_task::{}", &ctxw.task.id).as_str(), "{} finished", &ctxw.task.method)
        });

        // collect tha handle and then return. nothing else to do here
        self.handles.insert(task.id.clone(), (handle, run_handle));

        Ok(TaskStatus::Scheduled)
    }
}
impl Drop for JoTTyExecutor {
    fn drop(&mut self) {
        log::info!("dropping executor");

        // TODO Forcibly stop all running tasks, if any

        // TODO For all force stopped tasks, mark them as lost

        // delete ourselves from zookeeper
        let path = format!("/jotty/{}/executors/{}", &self.namespace, &self.id);
        let _ = self.zk.delete(&path, None);

        log::info!("deleted {}", &path);

    }
}