use crate::{Queues, Task, TaskDef, TaskLogger, TaskStatus, TASK_DEFAULT_POLICY};
use chrono::{DateTime, Utc};
use std::sync::Arc;
use std::thread;
use std::time::Duration;
use zookeeper::ZooKeeper;

/// Given the TaskDef, spawn the task
pub(crate) fn run_task(
    zk: Arc<ZooKeeper>,
    namespace: String,
    task: &TaskDef,
    job_id: String,
    queue: Queues,
) -> anyhow::Result<String> {
    let id = uuid::Uuid::new_v4().to_string();
    let logs = TaskLogger::new();

    log::debug!("creating task {} under job {}", &id, &job_id);

    let task = Task {
        // TODO we might want to add a way where a job can have tasks that are scheduled in the future
        status: TaskStatus::Pending(chrono::Utc::now(), 1),
        executions: 1,
        id: id.clone(),
        parent_job: job_id,
        method: task.method.clone(),
        args: task.arguments.clone(),
        resources: task.resources.clone(),
        results: Default::default(),
        executor: "__UNDEFINED".to_string(),
        logs,
        scheduler: task.policy.clone().unwrap_or(TASK_DEFAULT_POLICY),
    };

    task.save(zk, namespace)?;

    log::debug!("enqueued task {} into task evaluation queue", &task.id);
    queue.task.produce(task.id.into_bytes())?;

    Ok(id)
}

/// Returns a task
pub(crate) fn get_task(
    zk: Arc<ZooKeeper>,
    namespace: String,
    task_id: &str,
) -> anyhow::Result<Task> {
    Ok(Task::load(zk, namespace, task_id.to_string())?)
}

/// Iterate over all known tasks
pub(crate) fn task_walk(zk: Arc<ZooKeeper>, namespace: String, queue: zkmq::ZkMQ) {
    let last_run = format!("/jotty/{}/run/task_walk_last", &namespace);
    let task_path = format!("/jotty/{}/task", &namespace);
    loop {
        thread::sleep(Duration::from_secs(5)); // TODO make this dynamic

        if zk.exists(last_run.as_str(), false).unwrap().is_none() {
            zk.create(
                last_run.as_str(),
                serde_json::to_vec(&Utc::now()).unwrap(),
                zookeeper::Acl::open_unsafe().clone(),
                zookeeper::CreateMode::Persistent,
            )
            .unwrap();
        } else {
            let run_time: DateTime<Utc> =
                serde_json::from_slice(zk.get_data(last_run.as_str(), false).unwrap().0.as_slice())
                    .unwrap();
            if run_time + chrono::Duration::seconds(10) > Utc::now() {
                // TODO make this dynamic as well
                log::trace!("not time to do another task walk");
                continue;
            }
            zk.set_data(
                last_run.as_str(),
                serde_json::to_vec(&Utc::now()).unwrap(),
                None,
            )
            .unwrap();
        }

        // TODO add latch to prevent this from running more than once at the same time at /jotty/{}/run/task_walk/latch
        let children = zk.get_children(task_path.as_str(), false).unwrap();
        let chil_len = children.len();
        for task in children {
            let op = queue.produce(task.as_bytes());
            if op.is_err() {
                log::error!("unable to re-enqueue task {}. task is lost!", &task);
            } else {
                log::trace!("enqueued task {} for re-evaluation successfully", &task);
            }
        }

        if chil_len > 0 {
            log::debug!("evaluated {} tasks", chil_len);
        }
    }
}
