use log::trace;
use serde::{Deserialize, Serialize};

use crate::log_error;

#[derive(Clone, Debug, Default, Serialize, Deserialize)]
pub struct MumbleLink {
    pub ui_tick: u32,
    pub f_avatar_position: [f32; 3],
    pub f_avatar_front: [f32; 3],
    pub f_camera_position: [f32; 3],
    pub f_camera_front: [f32; 3],
    pub identity: CIdentity,
    pub context: CMumbleContext,
}
#[derive(Deserialize, Clone, Debug, Default, Serialize)]
pub struct MumbleIdentity {
    pub name: String,
    pub profession: u32,
    pub spec: u32,

    pub race: u32,

    pub map_id: u32,

    pub world_id: u32,

    pub team_color_id: u32,

    pub commander: bool,

    pub fov: f32,

    pub uisz: u32,
}
#[derive(Copy, Clone, Debug, Default, Deserialize, Serialize)]
pub struct MumbleContext {
    pub server_address: u32,

    pub map_id: u32,

    pub map_type: u32,

    pub shard_id: u32,

    pub instance: u32,

    pub build_id: u32,
    ///Bitmask: Bit 1 = IsMapOpen, Bit 2 = IsCompassTopRight,
    ///Bit 3 = DoesCompassHaveRotationEnabled, Bit 4 = Game
    ///has focus, Bit 5 = Is in Competitive game mode, Bit 6
    ///= Textbox has focus, Bit 7 = Is in Combat
    pub ui_state: u32,

    pub compass_width: u32,

    pub compass_height: u32,

    pub compass_rotation: f32,

    pub player_x: f32,

    pub player_y: f32,

    pub map_center_x: f32,

    pub map_center_y: f32,

    pub map_scale: f32,

    pub process_id: u32,

    pub mount_index: u32,
}
#[derive(Copy, Clone, Deserialize, Default, Debug, Serialize)]
pub struct WindowDimensions {
    pub x: i32,

    pub y: i32,

    pub width: i32,

    pub height: i32,
}

impl MumbleLink {
    /// The most used function probably. will check if the `link_ptr->ui_tick > &self.ui_tick`
    /// and update self's fields based on that.
    pub fn update(&mut self, link_ptr: *const CMumbleLink) -> anyhow::Result<()> {
        let cmlink = match unsafe { link_ptr.as_ref() } {
            Some(thing) => thing,
            None => panic!("link_ptr.as_ref returned None when trying to update MumbleLink. ptr is null. something is very wrong"),
        };

        if self.ui_tick == cmlink.ui_tick {
            return Ok(());
        }

        self.ui_tick = cmlink.ui_tick;

        self.f_avatar_position = cmlink.f_avatar_position;

        self.f_avatar_front = cmlink.f_avatar_front;

        self.f_camera_position = cmlink.f_camera_position;

        self.f_camera_front = cmlink.f_camera_front;
        self.identity.update(link_ptr)?;
        self.context.update(link_ptr);
        Ok(())
    }
    pub fn update_from_slice(&mut self, buffer: &[u8]) -> anyhow::Result<()> {
        assert!(buffer.len() >= 1093);
        Ok(self.update(buffer.as_ptr() as *const CMumbleLink)?)
    }
}

impl CIdentity {
    pub fn update(&mut self, link_ptr: *const CMumbleLink) -> anyhow::Result<()> {
        use widestring::U16CStr;
        let id =
            U16CStr::from_slice_with_nul(unsafe { &(*link_ptr).identity }).map_err(log_error)?;
        let id = id.to_string().map_err(log_error)?;
        *self = serde_json::from_str::<CIdentity>(&id).map_err(log_error)?;
        Ok(())
    }
}

impl MumbleContext {
    pub fn update(&mut self, link_ptr: *const CMumbleLink) {
        let mc = unsafe {
            std::ptr::read_volatile(&(*link_ptr).context as *const u8 as *const CMumbleContext)
        };
        self.server_address = u32::from_ne_bytes([
            mc.server_address[4],
            mc.server_address[5],
            mc.server_address[6],
            mc.server_address[7],
        ]);
        self.map_id = mc.map_id;
        self.map_type = mc.map_type;
        self.shard_id = mc.shard_id;
        self.instance = mc.instance;
        self.build_id = mc.build_id;
        self.ui_state = mc.ui_state;
        self.compass_width = mc.compass_width as u32;
        self.compass_height = mc.compass_height as u32;
        self.compass_rotation = mc.compass_rotation;
        self.player_x = mc.player_x;
        self.player_y = mc.player_y;
        self.map_center_x = mc.map_center_x;
        self.map_center_y = mc.map_center_y;
        self.map_scale = mc.map_scale;
        self.process_id = mc.process_id;
        self.mount_index = mc.mount_index as u32;
    }
}

impl CMumbleContext {
    pub fn update(&mut self, link_ptr: *const CMumbleLink) {
        let mc = unsafe {
            std::ptr::read_volatile(&(*link_ptr).context as *const u8 as *const CMumbleContext)
        };
        *self = mc;
    }
}
impl WindowDimensions {
    #[cfg(target_os = "windows")]
    pub fn update(&mut self, link_ptr: *const CMumbleLink) -> anyhow::Result<()> {
        *self = crate::win::get_win_pos_dim(link_ptr)?;
        Ok(())
    }

    /// buffer must be atleast 16 bytes long
    pub fn copy_raw_bytes_into(&self, buffer: &mut [u8]) {
        assert!(buffer.len() >= 16);
        buffer.copy_from_slice(bytemuck::bytes_of(self));
        trace!("{:?}", buffer);
    }
}
unsafe impl bytemuck::Zeroable for WindowDimensions {
    fn zeroed() -> Self {
        unsafe { core::mem::zeroed() }
    }
}
unsafe impl bytemuck::Pod for WindowDimensions {}

// pub const MUMBLE_LINK_SIZE: usize = 1193;
pub const C_MUMBLE_LINK_SIZE: usize = std::mem::size_of::<CMumbleLink>();
pub const USEFUL_C_MUMBLE_LINK_SIZE: usize = 1193;

#[derive(Debug, Clone, Copy)]
#[repr(C)]
pub struct CMumbleLink {
    pub ui_version: u32,
    pub ui_tick: u32,
    pub f_avatar_position: [f32; 3],
    pub f_avatar_front: [f32; 3],
    pub f_avatar_top: [f32; 3],
    pub name: [u8; 512],
    pub f_camera_position: [f32; 3],
    pub f_camera_front: [f32; 3],
    pub f_camera_top: [f32; 3],
    pub identity: [u16; 256],
    pub context_len: u32,
    pub context: [u8; 256],
    pub description: [u8; 4096],
}
#[derive(Debug, Default, Clone, Copy, Serialize, Deserialize)]
#[repr(C)]
/// The mumble context as stored inside the context field of CMumbleLink.
/// the first 48 bytes Mumble uses for identification is upto build_id field
/// the rest of the fields after build_id are provided by gw2 for addon devs.
pub struct CMumbleContext {
    /// first byte is `2` if ipv4. and `[4..7]` bytes contain the ipv4 octets.
    pub server_address: [u8; 28], // contains sockaddr_in or sockaddr_in6
    pub map_id: u32,
    pub map_type: u32,
    pub shard_id: u32,
    pub instance: u32,
    pub build_id: u32,
    // Additional data that gw2 provides us
    pub ui_state: u32, // Bitmask: Bit 1 = IsMapOpen, Bit 2 = IsCompassTopRight, Bit 3 = DoesCompassHaveRotationEnabled, Bit 4 = Game has focus, Bit 5 = Is in Competitive game mode, Bit 6 = Textbox has focus, Bit 7 = Is in Combat
    pub compass_width: u16, // pixels
    pub compass_height: u16, // pixels
    pub compass_rotation: f32, // radians
    pub player_x: f32, // continentCoords
    pub player_y: f32, // continentCoords
    pub map_center_x: f32, // continentCoords
    pub map_center_y: f32, // continentCoords
    pub map_scale: f32,
    pub process_id: u32,
    pub mount_index: u8,
}

#[derive(Debug, Default, Clone, Serialize, Deserialize)]
/// The json structure of the Identity field inside Cmumblelink.
/// the json string is null terminated and utf-16 encoded. so, need to use
/// Widestring crate's U16Cstring to first parse the bytes and then, convert to
/// String before deserializing to CIdentity
pub struct CIdentity {
    pub name: String,
    pub profession: u32,
    pub spec: u32,
    pub race: u32,
    pub map_id: u32,
    pub world_id: u32,
    pub team_color_id: u32,
    pub commander: bool,
    pub fov: f32,
    pub uisz: u32,
}

impl CMumbleLink {
    pub fn get_cmumble_link(link_ptr: *const CMumbleLink) -> CMumbleLink {
        unsafe { std::ptr::read_volatile(link_ptr) }
    }

    pub fn is_valid(link_ptr: *const CMumbleLink) -> bool {
        unsafe { (*link_ptr).ui_tick > 0 }
    }

    pub fn get_ui_tick(link_ptr: *const CMumbleLink) -> u32 {
        unsafe { (*link_ptr).ui_tick }
    }
    #[cfg(target_os = "windows")]
    pub fn new_ptr(key: &str) -> anyhow::Result<*const CMumbleLink> {
        crate::win::create_link_shared_mem(key)
    }
    pub fn copy_raw_bytes_into(link_ptr: *const CMumbleLink, buffer: &mut [u8]) {
        unsafe {
            std::ptr::copy_nonoverlapping(link_ptr as *const u8, buffer.as_mut_ptr(), buffer.len());
        }
    }
}

// mod private {

// #[derive(Debug, Default, Clone, Serialize, Deserialize)]
// #[repr(C)]
// pub struct MumbleLink {
//     //pub ui_version: u32,
//     pub ui_tick: u32,
//     pub f_avatar_position: [f32; 3],
//     pub f_avatar_front: [f32; 3],
//     //pub f_avatar_top: [f32; 3],
//     //pub name: [u16; 256],
//     pub f_camera_position: [f32; 3],
//     pub f_camera_front: [f32; 3],
//     //pub f_camera_top: [f32; 3],
//     pub identity: Identity,
//     //pub context_len: u32,
//     pub context: MumbleContext,
// }
// }

// #[cfg(test)]
// mod tests {
//     #[test]
//     fn validate_identity_json_str() {
//         let id = String::from(
//             r#"{
//             "name": "Irwene",
//             "profession": 4,
//             "spec": 55,
//             "race": 4,
//             "map_id": 50,
//             "world_id": 268435505,
//             "team_color_id": 0,
//             "commander": false,
//             "fov": 0.873,
//             "uisz": 1
//           }"#,
//         );
//         use super::Identity;
//         let identity: Identity = serde_json::from_str(&id).unwrap();
//         assert_eq!(identity.name, "Irwene");

//     }

//     #[test]
//     fn validate_identity_json_binary_utf16() {
//         let pure_utf16: Vec<u16> = vec![0x007b , 0x000a , 0x0020 , 0x0020 , 0x0022 , 0x006e , 0x0061 , 0x006d , 0x0065 , 0x0022 , 0x003a , 0x0020 , 0x0022 , 0x0049 , 0x0072 , 0x0077 , 0x0065 , 0x006e , 0x0065 , 0x0022 , 0x002c , 0x000a , 0x0020 , 0x0020 , 0x0022 , 0x0070 , 0x0072 , 0x006f , 0x0066 , 0x0065 , 0x0073 , 0x0073 , 0x0069 , 0x006f , 0x006e , 0x0022 , 0x003a , 0x0020 , 0x0034 , 0x002c , 0x000a , 0x0020 , 0x0020 , 0x0022 , 0x0073 , 0x0070 , 0x0065 , 0x0063 , 0x0022 , 0x003a , 0x0020 , 0x0035 , 0x0035 , 0x002c , 0x000a , 0x0020 , 0x0020 , 0x0022 , 0x0072 , 0x0061 , 0x0063 , 0x0065 , 0x0022 , 0x003a , 0x0020 , 0x0034 , 0x002c , 0x000a , 0x0020 , 0x0020 , 0x0022 , 0x006d , 0x0061 , 0x0070 , 0x005f , 0x0069 , 0x0064 , 0x0022 , 0x003a , 0x0020 , 0x0035 , 0x0030 , 0x002c , 0x000a , 0x0020 , 0x0020 , 0x0022 , 0x0077 , 0x006f , 0x0072 , 0x006c , 0x0064 , 0x005f , 0x0069 , 0x0064 , 0x0022 , 0x003a , 0x0020 , 0x0032 , 0x0036 , 0x0038 , 0x0034 , 0x0033 , 0x0035 , 0x0035 , 0x0030 , 0x0035 , 0x002c , 0x000a , 0x0020 , 0x0020 , 0x0022 , 0x0074 , 0x0065 , 0x0061 , 0x006d , 0x005f , 0x0063 , 0x006f , 0x006c , 0x006f , 0x0072 , 0x005f , 0x0069 , 0x0064 , 0x0022 , 0x003a , 0x0020 , 0x0030 , 0x002c , 0x000a , 0x0020 , 0x0020 , 0x0022 , 0x0063 , 0x006f , 0x006d , 0x006d , 0x0061 , 0x006e , 0x0064 , 0x0065 , 0x0072 , 0x0022 , 0x003a , 0x0020 , 0x0066 , 0x0061 , 0x006c , 0x0073 , 0x0065 , 0x002c , 0x000a , 0x0020 , 0x0020 , 0x0022 , 0x0066 , 0x006f , 0x0076 , 0x0022 , 0x003a , 0x0020 , 0x0030 , 0x002e , 0x0038 , 0x0037 , 0x0033 , 0x002c , 0x000a , 0x0020 , 0x0020 , 0x0022 , 0x0075 , 0x0069 , 0x0073 , 0x007a , 0x0022 , 0x003a , 0x0020 , 0x0031 , 0x000a , 0x007d];
//         let id = String::from_utf16_lossy(&pure_utf16);
//         println!("the string content in id is {}", id);
//         use super::Identity;
//         let identity: Identity = serde_json::from_str(&id).unwrap();
//         assert_eq!(identity.name, "meow");

//     }
// }
