use serde::{Deserialize, Serialize};
#[cfg(target_os = "windows")]
use winapi::{
    shared::{
        minwindef::{BOOL, LPARAM, LPDWORD},
        ntdef::NULL,
        windef::{HWND, LPRECT, RECT},
    },
    um::{
        errhandlingapi::GetLastError,
        handleapi::INVALID_HANDLE_VALUE,
        memoryapi::{MapViewOfFile, FILE_MAP_ALL_ACCESS},
        minwinbase::SECURITY_ATTRIBUTES,
        winbase::CreateFileMappingA,
        winnt::PAGE_READWRITE,
        winuser::{EnumWindows, GetWindowRect, GetWindowThreadProcessId},
    },
};

pub const MUMBLE_LINK_SIZE: usize = 1193;
pub const C_MUMBLE_LINK_SIZE: usize = std::mem::size_of::<CMumbleLink>();

#[derive(Debug, Clone, Copy)]
pub struct WinPosDim {
    x: u32,
    y: u32,
    width: u32,
    height: u32,
}
#[derive(Debug, Clone, Copy)]
#[repr(C)]
pub struct CMumbleLink {
    pub ui_version: u32,
    pub ui_tick: u32,
    pub f_avatar_position: [f32; 3],
    pub f_avatar_front: [f32; 3],
    pub f_avatar_top: [f32; 3],
    pub name: [u8; 512],
    pub f_camera_position: [f32; 3],
    pub f_camera_front: [f32; 3],
    pub f_camera_top: [f32; 3],
    pub identity: [u16; 256],
    pub context_len: u32,
    pub context: [u8; 256],
    pub description: [u8; 4096],
}
#[derive(Debug, Clone, Copy)]
#[repr(C)]
struct MumbleContext {
    //the 48 bytes Mumble uses for identification
    server_address: [u8; 28], // contains sockaddr_in or sockaddr_in6
    map_id: u32,
    map_type: u32,
    shard_id: u32,
    instance: u32,
    build_id: u32,
    // Additional data that gw2 provides us
    ui_state: u32, // Bitmask: Bit 1 = IsMapOpen, Bit 2 = IsCompassTopRight, Bit 3 = DoesCompassHaveRotationEnabled, Bit 4 = Game has focus, Bit 5 = Is in Competitive game mode, Bit 6 = Textbox has focus, Bit 7 = Is in Combat
    compass_width: u16, // pixels
    compass_height: u16, // pixels
    compass_rotation: f32, // radians
    player_x: f32, // continentCoords
    player_y: f32, // continentCoords
    map_center_x: f32, // continentCoords
    map_center_y: f32, // continentCoords
    map_scale: f32,
    process_id: u32,
    mount_index: u8,
}
#[derive(Serialize, Deserialize)]
pub struct Identity {
    name: String,
    profession: u8,
    spec: u8,
    race: u8,
    map_id: u16,
    world_id: u16,
    team_color_id: u16,
    commander: bool,
    fov: f32,
    uisz: u8,
}
#[derive(Debug, Clone, Copy)]
pub struct MumbleLink {
    pub ui_version: u32,
    pub ui_tick: u32,
    pub f_avatar_position: [f32; 3],
    pub f_avatar_front: [f32; 3],
    pub f_avatar_top: [f32; 3],
    pub name: [u8; 512],
    pub f_camera_position: [f32; 3],
    pub f_camera_front: [f32; 3],
    pub f_camera_top: [f32; 3],
    pub identity: [u16; 256],
    pub context_len: u32,
    pub context: [u8; 85],
}

impl CMumbleLink {
    #[cfg(target_os = "windows")]
    pub fn new(key: &str) -> *const CMumbleLink {
        let handle;
        let key_cstr = std::ffi::CString::new(key).unwrap();
        let key_cstr_ptr = key_cstr.as_ptr();
        unsafe {
            let file_handle = CreateFileMappingA(
                INVALID_HANDLE_VALUE,
                NULL as *mut SECURITY_ATTRIBUTES,
                PAGE_READWRITE,
                0,
                C_MUMBLE_LINK_SIZE as u32,
                key_cstr_ptr,
            );
            if file_handle == NULL {
                panic!(
                    "could not create file map handle, error code: {}",
                    GetLastError()
                );
            }
            handle = MapViewOfFile(file_handle, FILE_MAP_ALL_ACCESS, 0, 0, C_MUMBLE_LINK_SIZE);
            if handle == NULL {
                panic!("could not map view of file, error code: {}", GetLastError());
            }
        }
        handle as *const CMumbleLink
    }
    #[cfg(target_os = "windows")]
    pub fn get_win_pos_dim(&self) -> Option<WinPosDim> {
        let link = self;
        if !(*link).is_valid() {
            return None;
        }
        unsafe {
            let context = (*link).context.as_ptr() as *const MumbleContext;

            println!("context: {:?}", *context);
            let mut pid: isize = (*context).process_id as isize;

            let result: BOOL = EnumWindows(
                Some(Self::get_handle_by_pid),
                &mut pid as *mut isize as LPARAM,
            );
            if result != 0 {
                print!("couldn't find gw2 window. error code: {}", GetLastError());
                return None;
            }

            let mut rect: RECT = RECT {
                left: 0,
                top: 0,
                right: 0,
                bottom: 0,
            };
            let status = GetWindowRect(pid as isize as HWND, &mut rect as LPRECT);
            if status == 0 {
                return None;
            }
            Some(WinPosDim {
                x: rect.left as u32,
                y: rect.top as u32,
                width: (rect.right - rect.left) as u32,
                height: (rect.bottom - rect.top) as u32,
            })
        }
    }
    #[cfg(target_os = "windows")]
    unsafe extern "system" fn get_handle_by_pid(handle: HWND, lp: LPARAM) -> BOOL {
        let mut handle_pid: u32 = 0;
        GetWindowThreadProcessId(handle, &mut handle_pid as LPDWORD);
        if handle_pid == 0 {
            return 1;
        }

        if *(lp as *const u32) == handle_pid {
            *(lp as *mut isize) = handle as isize;
            dbg!(handle, handle_pid);
            return 0;
        }
        return 1;
    }

    pub fn copy_bytes_into(&self, buffer: &mut [u8]) {
        unsafe {
            std::ptr::copy_nonoverlapping(
                self as *const CMumbleLink as *const u8,
                buffer.as_mut_ptr(),
                buffer.len(),
            );
        }
    }

    pub fn get_mumble_link(&self) -> CMumbleLink {
        unsafe { std::ptr::read_volatile(self as *const CMumbleLink) }
    }

    pub fn is_valid(&self) -> bool {
        self.ui_tick > 0
    }
}
impl WinPosDim {
    /// buffer must be atleast 16 bytes long
    pub fn copy_bytes_into(&self, buffer: &mut [u8]) {
        unsafe {
            let p = self as *const WinPosDim as *const u8;
            std::ptr::copy_nonoverlapping(p, buffer.as_mut_ptr(), 16);
        }
    }
}
